//                                               -*- C++ -*-
/**
 *  @file  ClaytonCopula.hxx
 *  @brief The ClaytonCopula distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: ClaytonCopula.hxx 1262 2009-05-28 12:47:53Z dutka $
 */
#ifndef OPENTURNS_CLAYTONCOPULA_HXX
#define OPENTURNS_CLAYTONCOPULA_HXX

#include "ArchimedeanCopula.hxx"
#include "Exception.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {


      /**
       * @class ClaytonCopula
       *
       * The ClaytonCopula distribution.
       */
      class ClaytonCopula 
	: public Model::ArchimedeanCopula
      {
	CLASSNAME;
      public:

	typedef Base::Common::InvalidArgumentException       InvalidArgumentException;
	typedef Model::ArchimedeanCopula                     ArchimedeanCopula;  // required by SWIG
	typedef ArchimedeanCopula::NumericalPoint            NumericalPoint;
	typedef ArchimedeanCopula::NumericalPointWithDescriptionCollection  NumericalPointWithDescriptionCollection;
	typedef ArchimedeanCopula::NotDefinedException       NotDefinedException;
	typedef ArchimedeanCopula::StorageManager            StorageManager;


	/** Default constructor */
	ClaytonCopula();

	/** Parameters constructor */
	ClaytonCopula(const NumericalScalar theta)
	  throw(InvalidArgumentException);


	/** Comparison operator */
	Bool operator ==(const ClaytonCopula & other) const;

	/** String converter */
	String __repr__() const;



	/* Interface inherited from Distribution */
	/** Virtual constructor */
	virtual ClaytonCopula * clone() const;	

	/** Get one realization of the distribution */
	NumericalPoint getRealization() const;

	/** Get the DDF of the distribution */
	using ArchimedeanCopula::computeDDF;
	NumericalPoint computeDDF(const NumericalPoint & point) const;

	/** Get the PDF of the distribution */
	using ArchimedeanCopula::computePDF;
	NumericalScalar computePDF(const NumericalPoint & point) const;

	/** Get the CDF of the distribution */
	using ArchimedeanCopula::computeCDF;
	NumericalScalar computeCDF(const NumericalPoint & point, const Bool tail = false) const;

	/** Get the PDFGradient of the distribution */
	NumericalPoint computePDFGradient(const NumericalPoint & point) const;

	/** Get the CDFGradient of the distribution */
	NumericalPoint computeCDFGradient(const NumericalPoint & point) const;

	/** Get the quantile of the distribution */
	NumericalPoint computeQuantile(const NumericalScalar prob,
				       const Bool tail = false) const;

	/** Compute the CDF of Xi | X1, ..., Xi-1. x = Xi, y = (X1,...,Xi-1) */
	NumericalScalar computeConditionalCDF(const NumericalScalar x, const NumericalPoint & y) const;

	/** Compute the quantile of Xi | X1, ..., Xi-1, i.e. x such that CDF(x|y) = q with x = Xi, y = (X1,...,Xi-1) */
	NumericalScalar computeConditionalQuantile(const NumericalScalar q, const NumericalPoint & y) const;

	/** Compute the archimedean generator of the archimedean copula, i.e.
	 *  the function phi such that the CDF of the copula can
	 *  be written as CDF(t) = phi^{-1}(phi(u)+phi(v))
	 */
	NumericalScalar computeArchimedeanGenerator(const NumericalScalar t) const;

	/** Compute the inverse of the archimedean generator */
	NumericalScalar computeInverseArchimedeanGenerator(const NumericalScalar t) const;

	/** Compute the derivative of the archimedean generator */
	NumericalScalar computeArchimedeanGeneratorDerivative(const NumericalScalar t) const;

	/** Compute the second derivative of the archimedean generator */
	NumericalScalar computeArchimedeanGeneratorSecondDerivative(const NumericalScalar t) const;

	/** Parameters value and description accessor */
	NumericalPointWithDescriptionCollection getParametersCollection() const;
	using ArchimedeanCopula::setParametersCollection;
	void setParametersCollection(const NumericalPointCollection & parametersCollection);

	/** Tell if the distribution has independent copula */
	Bool hasIndependentCopula() const;

	/* Interface specific to ClaytonCopula */

	/** Theta accessor */
	void setTheta(const NumericalScalar theta)
	  throw(InvalidArgumentException);
	NumericalScalar getTheta() const;

	/** Method save() stores the object through the StorageManager */
	virtual void save(const StorageManager::Advocate & adv) const;

	/** Method load() reloads the object from the StorageManager */
	virtual void load(const StorageManager::Advocate & adv);	

      protected:
    

      private:

	/** The parameter of the ClaytonCopula distribution */
	NumericalScalar theta_;

      }; /* class ClaytonCopula */


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_CLAYTONCOPULA_HXX */
