//                                               -*- C++ -*-
/**
 *  @file  EnumerateFunction.cxx
 *  @brief The bijective function to select polynomials in the orthogonal basis
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.cxx 815 2008-05-21 09:21:38Z dutka $
 */
#include "EnumerateFunction.hxx"
#include "OSS.hxx"
#include "PersistentObjectFactory.hxx"
#include "SpecFunc.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {

      CLASSNAMEINIT(EnumerateFunction);

      static Base::Common::Factory<EnumerateFunction> RegisteredFactory("EnumerateFunction");

      typedef Distribution::SpecFunc SpecFunc;

      /* Default constructor */
      EnumerateFunction::EnumerateFunction()
	: Base::Common::PersistentObject(),
	  dimension_(0)
      {
	// Nothing to do
      }

      /* Parameter constructor */
      EnumerateFunction::EnumerateFunction(const UnsignedLong dimension)
	: Base::Common::PersistentObject(),
	  dimension_(dimension)
      {
	// Nothing to do
      }


      /* Virtual constrcutor */
      EnumerateFunction * EnumerateFunction::clone() const
      {
	return new EnumerateFunction(*this);
      }


      /* String converter */
      String EnumerateFunction::__repr__() const
      {
	return OSS() << "class=" << getClassName()
		     << " dimension=" << dimension_;
      }


      /* Find the smallest n such that Binomial(n, n + dimension) > index and also returns the value of Binomial(n - 1, n - 1 + dimension) */
      UnsignedLong EnumerateFunction::findBinomialCoefficient(const UnsignedLong index,
							      const UnsignedLong dimension,
							      UnsignedLong & binomialCoefficient) const
      {
	UnsignedLong n(0);
	binomialCoefficient = 1;
	// Use floating point arithmetic to avoid overflow
	NumericalScalar newBinomialCoefficient(1.0);
	// Special treatment to avoid roundoff error during conversion
	while(static_cast<UnsignedLong>(round(newBinomialCoefficient)) <= index)
	  {
	    ++n;
	    binomialCoefficient = static_cast<UnsignedLong>(round(newBinomialCoefficient));
	    // Using integer arithmetic, the computation should have been implemented this way to avoid truncation:
	    // newBinomialCoefficient = (newBinomialCoefficient * (n + dimension)) / n;
	    // but for large n it should lead to an overflow.
	    // Instead, we use floating point arithmetic
	    newBinomialCoefficient *= 1.0 + static_cast<NumericalScalar>(dimension) / static_cast<NumericalScalar>(n);
	  }
	return n;
      }

      /* The bijective association between an integer and a set of indices.
	 First, we look for the total degree of the associated polynom. Then, we
	 compute recursively the complement by looping over the degree of the remainder.
	 For a given index I, we have:
	 I = Binomial(n_1, d) + ... + Binomial(n_{d-1}, 1)
	 where Binomial(n_1, d_1) is 
       */
      EnumerateFunction::Indices EnumerateFunction::operator() (const UnsignedLong index) const
      {
	Indices result(dimension_, 0);
	if (index == 0) return result;
	UnsignedLong binomialCoefficient(0);
	UnsignedLong degree(findBinomialCoefficient(index, dimension_, binomialCoefficient));
	// Loop over the dimension of the remaining polynomial
	UnsignedLong currentIndex(index);
	for (UnsignedLong i = 0; i < dimension_ - 1; ++i)
	  {
	    // Early exit if the remaining polynomial is constant
	    if (currentIndex <= binomialCoefficient)
	      {
		result[i] = degree;
		return result;
	      }
	    currentIndex -= binomialCoefficient;
	    const UnsignedLong remainingDegree(findBinomialCoefficient(currentIndex, dimension_ - i - 1, binomialCoefficient));
	    result[i] = degree - remainingDegree;
	    degree = remainingDegree;
	  }
	result[dimension_ - 1] = degree;
	return result;
      }

      /* The inverse of the association */
      UnsignedLong EnumerateFunction::inverse(const Indices & indices) const
      {
	const UnsignedLong size(indices.getSize());
	if (size == 1) return indices[0];
	UnsignedLong totalDegree(0);
	for (UnsignedLong i = 0; i < size; ++i) totalDegree += indices[i];
	UnsignedLong result(getStrateCumulatedCardinal(totalDegree));

	return result;
      }
      
      /* The cardinal of the given strate */
      UnsignedLong EnumerateFunction::getStrateCardinal(const UnsignedLong strateIndex) const
      {
	return static_cast<UnsignedLong>(round(exp(SpecFunc::LnGamma(dimension_ + strateIndex) - SpecFunc::LnGamma(dimension_) - SpecFunc::LnGamma(strateIndex + 1))));
      }
      
      /* The cardinal of the cumulated strate above or equal to the given strate */
      UnsignedLong EnumerateFunction::getStrateCumulatedCardinal(const UnsignedLong strateIndex) const
      {
	return static_cast<UnsignedLong>(round(exp(SpecFunc::LnGamma(dimension_ + strateIndex + 1) - SpecFunc::LnGamma(dimension_ + 1) - SpecFunc::LnGamma(strateIndex + 1))));
      }
 

      /* Dimension accessor */
      void EnumerateFunction::setDimension(const UnsignedLong dimension)
      {
	dimension_ = dimension;
      }

      UnsignedLong EnumerateFunction::getDimension() const
      {
	return dimension_;
      }


      /* Method save() stores the object through the StorageManager */
      void EnumerateFunction::save(const StorageManager::Advocate & adv) const
      {
	PersistentObject::save(adv);
      }


      /* Method load() reloads the object from the StorageManager */
      void EnumerateFunction::load(const StorageManager::Advocate & adv)
      {
	PersistentObject::load(adv);
      }


    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
