//                                               -*- C++ -*-
/**
 *  @file  Secant.cxx
 *  @brief Implementation class of the scalar nonlinear solver based on
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: Secant.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include "Secant.hxx"
#include "Log.hxx"
#include <cmath>

namespace OpenTURNS {

  namespace Base {

    namespace Solver {

      typedef Secant::NumericalMathFunction::NumericalPoint NumericalPoint;
      typedef Common::Log                                   Log;

      /**
       * @class Secant
       *
       * This class is an interface for the 1D nonlinear Secants
       */

      CLASSNAMEINIT(Secant);

      /** Default constructor */
      Secant::Secant():
	SolverImplementation()
      {
	// Nothing to do
      }

      /** Parameter constructor */
      Secant::Secant(const NumericalScalar absoluteError,
		     const NumericalScalar relativeError,
		     const UnsignedLong maximumFunctionEvaluation):
	SolverImplementation(absoluteError, relativeError, maximumFunctionEvaluation)
      {
	// Nothing to do
      }

      /** Virtual constructor */
      Secant * Secant::clone() const
      {
	return new Secant(*this);
      }

      /** String converter */
      String Secant::__repr__() const
      {
	OSS oss;
	oss << "class=" << Secant::GetClassName()
	    << " derived from " << SolverImplementation::__repr__();
	return oss;
      }

      /** Solve attempt to find one root to the equation function(x) = value in [infPoint, supPoint] */
      NumericalScalar Secant::solve(const NumericalMathFunction & function,
				    const NumericalScalar value,
				    NumericalScalar infPoint,
				    NumericalScalar supPoint)
        throw(InternalException, InvalidDimensionException)
      {
	if ((function.getInputNumericalPointDimension() != 1) || (function.getOutputNumericalPointDimension() != 1))
	  {
	    throw InvalidDimensionException(HERE) << "Error: secant requires a scalar function, here input dimension=" << function.getInputNumericalPointDimension() << " and output dimension=" << function.getOutputNumericalPointDimension();
	  }
	UnsignedLong maximumFunctionEvaluation(getMaximumFunctionEvaluation());
	if (maximumFunctionEvaluation < 2)
	  {
	    throw InternalException(HERE) << "Error: secant needs to evaluate the function at least two times, here maximumFunctionEvaluation=" << maximumFunctionEvaluation;
	  }
	/* We take into account the fact that we use 2 function calls when using the other solve method */
	setMaximumFunctionEvaluation(maximumFunctionEvaluation - 2);
	NumericalScalar root(solve(function, value, infPoint, supPoint, function(NumericalPoint(1, infPoint))[0], function(NumericalPoint(1, supPoint))[0]));
	setMaximumFunctionEvaluation(maximumFunctionEvaluation);
	setUsedFunctionEvaluation(getUsedFunctionEvaluation() + 2);
	return root;
      }

      /** Solve attempt to find one root to the equation function(x) = value in [infPoint, supPoint] given function(infPoint) and function(supPoint) with the secant method */
      NumericalScalar Secant::solve(const NumericalMathFunction & function,
                                    const NumericalScalar value,
				    NumericalScalar infPoint,
				    NumericalScalar supPoint,
				    NumericalScalar infValue,
				    NumericalScalar supValue)
        throw(InternalException, InvalidDimensionException)
      {
	if ((function.getInputNumericalPointDimension() != 1) || (function.getOutputNumericalPointDimension() != 1))
	  {
	    throw InvalidDimensionException(HERE) << "Error: secant method requires a scalar function, here input dimension=" << function.getInputNumericalPointDimension() << " and output dimension=" << function.getOutputNumericalPointDimension();
	  }
	/* We transform function(x) = value into function(x) - value = 0 */
	infValue -= value;
	supValue -= value;
	if (infPoint > supPoint)
	  {
	    NumericalScalar tmp(infPoint);
	    infPoint = supPoint;
	    supPoint = tmp;
	    tmp = infValue;
	    infValue = supValue;
	    supValue = tmp;
	  }
	if (infValue * supValue >= 0.0)
	  {
	    throw InternalException(HERE) << "Error: Brent method requires that the function takes different signs at the endpoints of the given starting interval, here f(infPoint) - value=" << infValue << " and f(supPoint) - value=" << supValue;
	  }
	NumericalScalar oldPoint(infPoint);
	NumericalScalar currentPoint(supPoint);
	NumericalScalar oldValue(infValue);
	NumericalScalar currentValue(supValue);
	UnsignedLong usedFunctionEvaluation(0);
	UnsignedLong maximumFunctionEvaluation(getMaximumFunctionEvaluation());
	NumericalScalar length(fabs(currentPoint - oldPoint));
	NumericalScalar newPoint(0.0);
	while ((length > getAbsoluteError()) && (length > getRelativeError() * fabs(newPoint)) && (usedFunctionEvaluation < maximumFunctionEvaluation))
	  {
	    // Check if the secant method can be used for this iteration
	    if (oldValue == currentValue)
	      {
		// Secant method cannot be used. Use bisection instead.
		newPoint = 0.5 * (supPoint + infPoint);
		length = 0.5 * (supPoint - infPoint);
	      }
	    else
	      {
		// Secant method can be used.
		newPoint = (oldPoint - oldValue * (oldPoint - currentPoint) / (oldValue - currentValue));
		length = fabs(newPoint - currentPoint);
		// Check if the secant candidat is admissible.
                if ((newPoint > supPoint) || (newPoint < infPoint))
	          {
		    newPoint = 0.5 * (infPoint + supPoint);
                    length = 0.5 * (supPoint - infPoint);
	          }
	      }
	    // Evaluate the function at the new point, and update the sequence of points
	    NumericalScalar newValue(function(NumericalPoint(1, newPoint))[0] - value);
	    usedFunctionEvaluation++;
	    oldPoint = currentPoint;
	    oldValue = currentValue;
	    currentPoint = newPoint;
	    currentValue = newValue;
	    // Update also the bounds of the bracketing interval
	    if (newValue * infValue >= 0.0)
	      {
		infPoint = newPoint;
		infValue = newValue;
	      }
	    else
	      {
		supPoint = newPoint;
		supValue = newValue;
	      }
	  } // while
	setUsedFunctionEvaluation(usedFunctionEvaluation);
	// Perform one last inexpensive step to increase precision
	if (oldValue != currentValue)
	  {
	return oldPoint - oldValue * (oldPoint - currentPoint) / (oldValue - currentValue);
	  }
	else
	  {
	    return 0.5 * (infPoint + supPoint);
	  }
      }

    } /* namespace Solver */
  } /* namespace Base */
} /* namespace OpenTURNS */
