//                                               -*- C++ -*-
/**
 *  @file  Bisection.cxx
 *  @brief Implementation class of the scalar nonlinear solver based on
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: Bisection.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include "Bisection.hxx"
#include <cmath>

namespace OpenTURNS {

  namespace Base {

    namespace Solver {

      typedef Bisection::NumericalMathFunction::NumericalPoint NumericalPoint;

      /**
       * @class Bisection
       *
       * This class is an interface for the 1D nonlinear Bisections
       */

      CLASSNAMEINIT(Bisection);

      /** Default constructor */
      Bisection::Bisection():
	SolverImplementation()
      {
	// Nothing to do
      }

      /** Parameter constructor */
      Bisection::Bisection(const NumericalScalar absoluteError,
			   const NumericalScalar relativeError,
			   const UnsignedLong maximumFunctionEvaluation):
	SolverImplementation(absoluteError, relativeError, maximumFunctionEvaluation)
      {
	// Nothing to do
      }

      /** Virtual constructor */
      Bisection * Bisection::clone() const
      {
	return new Bisection(*this);
      }

      /** String converter */
      String Bisection::__repr__() const
      {
	OSS oss;
	oss << "class=" << Bisection::GetClassName()
	    << " derived from " << SolverImplementation::__repr__();
	return oss;
      }

      /** Solve attempt to find one root to the equation function(x) = value in [infPoint, supPoint] */
      NumericalScalar Bisection::solve(const NumericalMathFunction & function,
				       const NumericalScalar value,
				       NumericalScalar infPoint,
				       NumericalScalar supPoint)
        throw(InternalException, InvalidDimensionException)
      {
	if ((function.getInputNumericalPointDimension() != 1) || (function.getOutputNumericalPointDimension() != 1))
	  {
	    throw InvalidDimensionException(HERE) << "Error: bisection requires a scalar function, here input dimension=" << function.getInputNumericalPointDimension() << " and output dimension=" << function.getOutputNumericalPointDimension();
	  }
	UnsignedLong maximumFunctionEvaluation(getMaximumFunctionEvaluation());
	if (maximumFunctionEvaluation < 2)
	  {
	    throw InternalException(HERE) << "Error: bisection needs to evaluate the function at least two times, here maximumFunctionEvaluation=" << maximumFunctionEvaluation;
	  }
	/* We take into account the fact that we use 2 function calls when using the other solve method */
	setMaximumFunctionEvaluation(maximumFunctionEvaluation - 2);
	NumericalScalar root(solve(function, value, infPoint, supPoint, function(NumericalPoint(1, infPoint))[0], function(NumericalPoint(1, supPoint))[0]));
	setMaximumFunctionEvaluation(maximumFunctionEvaluation);
	setUsedFunctionEvaluation(getUsedFunctionEvaluation() + 2);
	return root;
      }

      /** Solve attempt to find one root to the equation function(x) = value in [infPoint, supPoint] given function(infPoint) and function(supPoint) with the bisection method */
      NumericalScalar Bisection::solve(const NumericalMathFunction & function,
				       const NumericalScalar value,
				       NumericalScalar infPoint,
				       NumericalScalar supPoint,
				       NumericalScalar infValue,
				       NumericalScalar supValue)
        throw(InternalException, InvalidDimensionException)
      {
	if ((function.getInputNumericalPointDimension() != 1) || (function.getOutputNumericalPointDimension() != 1))
	  {
	    throw InvalidDimensionException(HERE) << "Error: bisection method requires a scalar function, here input dimension=" << function.getInputNumericalPointDimension() << " and output dimension=" << function.getOutputNumericalPointDimension();
	  }
	UnsignedLong usedFunctionEvaluation(0);
	UnsignedLong maximumFunctionEvaluation(getMaximumFunctionEvaluation());
	/* We transform function(x) = value into function(x) - value = 0 */
	infValue -= value;
	supValue -= value;
	if (infValue * supValue >= 0.0)
	  {
	    throw InternalException(HERE) << "Error: bisection method requires that the function takes different signs at the endpoints of the given starting interval, here f(infPoint)=" << infValue << ", f(supPoint)=" << supValue << " and value=" << value;
	  }
	NumericalScalar middlePoint(0.5 * (infPoint + supPoint));
	NumericalScalar length(fabs(infPoint - supPoint));
	while ((length > getAbsoluteError()) && (length > getRelativeError() * fabs(middlePoint)) && (usedFunctionEvaluation < maximumFunctionEvaluation))
	  {
	    middlePoint = 0.5 * (infPoint + supPoint);
	    NumericalScalar middleValue(function(NumericalPoint(1, middlePoint))[0] - value);
	    usedFunctionEvaluation++;
	    // If the function takes a value at middle on the same side of value that at left
	    if (middleValue * infValue >= 0.0)
	      {
		infPoint = middlePoint;
		infValue = middleValue;
	      }
	    else
	      {
		supPoint = middlePoint;
		supValue = middleValue;
	      }
	    length = fabs(infPoint - supPoint);
	  } // while
	setUsedFunctionEvaluation(usedFunctionEvaluation);
	return 0.5 * (infPoint + supPoint);
      }

    } /* namespace Solver */
  } /* namespace Base */
} /* namespace OpenTURNS */
