//                                               -*- C++ -*-
/**
 *  @file  NearestPointAlgorithm.hxx
 *  @brief NearestPointAlgorithm implements an algorithm for finding the 
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: NearestPointAlgorithm.hxx 1262 2009-05-28 12:47:53Z dutka $
 */
#ifndef OPENTURNS_NEARESTPOINTALGORITHM_HXX
#define OPENTURNS_NEARESTPOINTALGORITHM_HXX

#include "OTprivate.hxx"
#include "TypedInterfaceObject.hxx"
#include "Threadable.hxx"
#include "NumericalMathFunction.hxx"
#include "NearestPointAlgorithmImplementation.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Optim
    {

      /**
       * @class NearestPointAlgorithm
       * NearestPointAlgorithm implements an interface to an algorithm for finding the 
       * point of a manifold defined implicitely the nearest to the origin
       */

      class NearestPointAlgorithm
	: public Common::TypedInterfaceObject<NearestPointAlgorithmImplementation>,
	  public Common::Threadable
      {

	CLASSNAME;
      public:

	typedef NearestPointAlgorithmImplementation::NumericalPoint    NumericalPoint;
	typedef Func::NumericalMathFunction                            NumericalMathFunction;
	typedef Common::Pointer<NearestPointAlgorithmImplementation>   Implementation;
	typedef NearestPointAlgorithmImplementation::InternalException InternalException;
#ifndef SWIG
	typedef NearestPointAlgorithmImplementation::Result            Result;
#else
	typedef NearestPointAlgorithmImplementationResult              Result;
#endif

	/** Default constructor */
	explicit NearestPointAlgorithm();

	/** Constructor from an implementation */
	NearestPointAlgorithm(const NearestPointAlgorithmImplementation & implementation);

	/** Constructor from a Pointer to an implementation */
	NearestPointAlgorithm(const Implementation & p_implementation);

	/** Constructor with parameters */
	NearestPointAlgorithm(const NumericalMathFunction & levelFunction);

	/** Performs the actual computation. Must be overloaded by the actual optimisation algorithm */
	virtual void run()
          throw(InternalException);

	/** Starting point accessor */
	NumericalPoint getStartingPoint() const;

	/** Starting point accessor */
	void setStartingPoint(const NumericalPoint & startingPoint);

	/** Level function accessor */
	NumericalMathFunction getLevelFunction() const;

	/** Level function accessor */
	void setLevelFunction(const NumericalMathFunction & levelFunction);

	/** Level value accessor */
	NumericalScalar getLevelValue() const;

	/** Level value accessor */
	void setLevelValue(const NumericalScalar levelValue);

	/** Result accessor */
	Result getResult() const;

	/** Result accessor */
	void setResult(const Result & result);

	/** Maximum iterations number accessor */
	UnsignedLong getMaximumIterationsNumber() const;

	/** Maximum iterations number accessor */
	void setMaximumIterationsNumber(const UnsignedLong maximumIterationsNumber);

	/** Maximum absolute error accessor */
	NumericalScalar getMaximumAbsoluteError() const;

	/** Maximum absolute error accessor */
	void setMaximumAbsoluteError(const NumericalScalar maximumAbsoluteError);

	/** Maximum relative error accessor */
	NumericalScalar getMaximumRelativeError() const;

	/** Maximum relative error accessor */
	void setMaximumRelativeError(const NumericalScalar maximumRelativeError);

	/** Maximum residual error accessor */
	NumericalScalar getMaximumResidualError() const;

	/** Maximum residual error accessor */
	void setMaximumResidualError(const NumericalScalar maximumResidualError);

	/** Maximum constraint error accessor */
	NumericalScalar getMaximumConstraintError() const;

	/** Maximum constraint error accessor */
	void setMaximumConstraintError(const NumericalScalar maximumConstraintError);

	/** String converter */
	virtual String __repr__() const;

	/** Verbose accessor */
	Bool getVerbose() const;
	void setVerbose(const Bool verbose);

      } ; /* class NearestPointAlgorithm */

    } /* namespace Optim */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_NEARESTPOINTALGORITHM_HXX */
