//                                               -*- C++ -*-
/**
 * @file  QuadraticNumericalMathEvaluationImplementation.cxx
 * @brief Class for a quadratic numerical math function implementation
 *        of the form y = constant + <linear, x-c> + <<quadratic, x-c>, x-c>
 *        where constant is a dim(y) numerical point, c a dim(x) numerical
 *        point, linear is a dim(x) by dim(y) matrix, quadratic is a
 *        dim(x) by dim(x) by dim(y) symmetric tensor and <linear, x>
 *        means Transpose(linear).x, <quadratic, x> means
 *        Transpose_kj(quadratic).x
 *
 * (C) Copyright 2005-2006 EADS
 *
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 *
 *
 * \author $LastChangedBy: dutka $
 * \date   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 */

#include "QuadraticNumericalMathEvaluationImplementation.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Func {

      CLASSNAMEINIT(QuadraticNumericalMathEvaluationImplementation);

      static Common::Factory<QuadraticNumericalMathEvaluationImplementation> RegisteredFactory("QuadraticNumericalMathEvaluationImplementation");

      /* Default constructor */
      QuadraticNumericalMathEvaluationImplementation::QuadraticNumericalMathEvaluationImplementation()
	: NumericalMathEvaluationImplementation()
      {
	// Nothing to do
      }

      /* Parameter constructor f(x) = constant + linear.x + 1/2 x'.quadratic.x */
      QuadraticNumericalMathEvaluationImplementation::QuadraticNumericalMathEvaluationImplementation(const NumericalPoint & center,
												     const NumericalPoint & constant,
												     const Matrix & linear,
												     const SymmetricTensor & quadratic)
        throw(InvalidDimensionException) :
        NumericalMathEvaluationImplementation(),
        center_(center),
        constant_(constant),
	linear_(linear.transpose()),
	quadratic_(quadratic)
      {
	/* Check if the dimension of the constant term is compatible with the linear and quadratic terms */
        if ((constant.getDimension() != linear.getNbColumns()) || (constant.getDimension() != quadratic.getNbSheets())) throw InvalidDimensionException(HERE) << "Constant term dimension is incompatible with the linear term or with the quadratic term";
	/* Check if the dimension of the center is compatible with the linear and quadratic terms */
        if ((center.getDimension() != linear.getNbRows()) || (center.getDimension() != quadratic.getNbRows())) throw InvalidDimensionException(HERE) << "Center term dimension is incompatible with the linear term or with the quadratic term";
	Description description(0);
	for (UnsignedLong i = 0; i < center.getDimension(); ++i) description.add(OSS() << "x" << i);
	for (UnsignedLong i = 0; i < constant.getDimension(); ++i) description.add(OSS() << "y" << i);
	setDescription(description);
      }

      /* Virtual constructor */
      QuadraticNumericalMathEvaluationImplementation * QuadraticNumericalMathEvaluationImplementation::clone() const
      {
	return new QuadraticNumericalMathEvaluationImplementation(*this);
      }

      /* Comparison operator */
      Bool QuadraticNumericalMathEvaluationImplementation::operator ==(const QuadraticNumericalMathEvaluationImplementation & other) const
      {
	return ((quadratic_ == other.getQuadratic()) && (linear_ == other.getLinear()) && (constant_ == other.getConstant()) && (center_ == other.getCenter()));
      }
  
      /* String converter */
      String QuadraticNumericalMathEvaluationImplementation::__repr__() const {
	OSS oss;
	oss << "class=" << QuadraticNumericalMathEvaluationImplementation::GetClassName()
	    << " name=" << getName()
	    << " center=" << center_.__repr__()
            << " constant=" << constant_.__repr__()
            << " linear=" << linear_.transpose()
            << " quadratic=" << quadratic_;
	return oss;
      }
  
      /* Accessor for the center */
      QuadraticNumericalMathEvaluationImplementation::NumericalPoint QuadraticNumericalMathEvaluationImplementation::getCenter() const
      {
	return center_;
      }

      /* Accessor for the constant term */
      QuadraticNumericalMathEvaluationImplementation::NumericalPoint QuadraticNumericalMathEvaluationImplementation::getConstant() const
      {
	return constant_;
      }

      /* Accessor for the linear term */
      QuadraticNumericalMathEvaluationImplementation::Matrix QuadraticNumericalMathEvaluationImplementation::getLinear() const
      {
	return linear_.transpose();
      }

      /* Accessor for the quadratic term */
      QuadraticNumericalMathEvaluationImplementation::SymmetricTensor QuadraticNumericalMathEvaluationImplementation::getQuadratic() const
      {
	return quadratic_;
      }

      /* Here is the interface that all derived class must implement */
	
      /* Operator () */
      QuadraticNumericalMathEvaluationImplementation::NumericalPoint QuadraticNumericalMathEvaluationImplementation::operator() (const NumericalPoint & in) const
	throw(InvalidArgumentException, InternalException)
      {
	if ((in.getDimension() != linear_.getNbColumns()) || (in.getDimension() != quadratic_.getNbRows())) throw InvalidArgumentException(HERE) << "Invalid input dimension";
	/* We don't have a true linear algebra with tensors, so we must perform the tensor/vector product by hand */
	const NumericalPoint delta(in - center_);
	NumericalPoint value(constant_ + linear_ * delta);
	/* As we don't have a sheet extractor yet, we can't use the following code */
	const UnsignedLong sheetNumber(quadratic_.getNbSheets());
	for(UnsignedLong index = 0; index < sheetNumber; ++index)
	  {
	    value[index] += 0.5 * NumericalPoint::dot(delta, quadratic_.getSheet(index) * delta);
	  }
	++callsNumber_;
	return value;
      }

      /* Accessor for input point dimension */
      UnsignedLong QuadraticNumericalMathEvaluationImplementation::getInputNumericalPointDimension() const
	throw(InternalException)
      {
	return center_.getDimension();
      }
      
      /* Accessor for output point dimension */
      UnsignedLong QuadraticNumericalMathEvaluationImplementation::getOutputNumericalPointDimension() const
	throw(InternalException)
      {
	return constant_.getDimension();
      }
      
      /* Method save() stores the object through the StorageManager */
      void QuadraticNumericalMathEvaluationImplementation::save(const StorageManager::Advocate & adv) const
      {
	NumericalMathEvaluationImplementation::save(adv);
	adv.writeValue(center_, StorageManager::MemberNameAttribute, "center_");
	adv.writeValue(constant_, StorageManager::MemberNameAttribute, "constant_");
	adv.writeValue(linear_, StorageManager::MemberNameAttribute, "linear_");
	adv.writeValue(quadratic_, StorageManager::MemberNameAttribute, "quadratic_");
      }

      /* Method load() reloads the object from the StorageManager */
      void QuadraticNumericalMathEvaluationImplementation::load(const StorageManager::Advocate & adv)
      {
	NumericalMathEvaluationImplementation::load(adv);
	StorageManager::List objList = adv.getList(StorageManager::ObjectEntity);
	objList.readValue(center_,  StorageManager::MemberNameAttribute, "center_");
	objList.readValue(constant_,  StorageManager::MemberNameAttribute, "constant_");
	objList.readValue(linear_, StorageManager::MemberNameAttribute, "linear_");
	objList.readValue(quadratic_, StorageManager::MemberNameAttribute, "quadratic_");
      }

    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
