//                                               -*- C++ -*-
/**
 *  @file  Study.hxx
 *  @brief Study keeps all PersistentObjects in a file
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: Study.hxx 1262 2009-05-28 12:47:53Z dutka $
 */
#ifndef OPENTURNS_STUDY_HXX
#define OPENTURNS_STUDY_HXX

#include <iostream>              // for std::ostream
#include <map>                   // for std::map
#include "Object.hxx"
#include "Pointer.hxx"
#include "StorageManager.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Common
    {

      /* Forward declarations */
      class InterfaceObject;
      class StorageManager;

      /**
       * @class Study
       *
       * Study keeps all PersistentObjects in a file
       */

      class Study
	: public Object
      {
	CLASSNAME;

      public:

	/** An enumaration to designate the desired map */
	enum Map { MAINMAP, GUIMAP, TUIMAP, LAST /* Do NOT use LAST tag */ };

	/* Some typedefs for easy reading */
	typedef Common::Pointer<PersistentObject>   MapElement;
	typedef std::map<Id, MapElement>            ObjectMap;
	typedef std::map<Map, ObjectMap>            MapCollection;
	typedef Common::Pointer<StorageManager>     StorageManagerImplementation;

	/**
	 * Default constructor
	 */
	explicit Study();

	/**
	 * String converter
	 * This method shows human readable information on the
	 * internal state of an object. It is used when streaming
	 * the object or for user information.
	 */
	virtual String __repr__() const;


	/** This method saves the study through the storage manager */
	void save();

	/** This method reloads the study from the storage manager */
	void load();



	/** Storage manager accessor */
	void setStorageManager(const StorageManager & smgr);
	StorageManagerImplementation getStorageManager() const;

	/** Add a PersistentObject to the study */
	void add(const InterfaceObject & io, Map map = MAINMAP);
#ifndef SWIG
	/** Add a PersistentObject to the study (any map) */
	void add(const PersistentObject * po, Map map = MAINMAP);
#endif
	/** Add a PersistentObject to the study (any map) */
	void add(const PersistentObject & po, Map map = MAINMAP);

	/** Remove a PersistentObject from the study */
	void remove(const InterfaceObject & io, Map map = MAINMAP);

	/** Query if object is stored in study */
	Bool hasObject(Id id) const;

	/** Get object whose id is given */
	MapElement getObject(Id id) const;

	/** Fill an object with one got from study */
	void fillObject(PersistentObject & po, const String & name) const;
	void fillObject(InterfaceObject  & io, const String & name) const;
	void fillObject(PersistentObject & po, const Id id) const;
	void fillObject(InterfaceObject  & io, const Id id) const;


      protected:

	/** Get object whose class and name are given */
	MapElement getObjectByName(const String & className, const String & name) const;

	/** Add a PersistentObject to the study (any map) */
	void add(const PersistentObject * po, ObjectMap & map);

	/** Add a PersistentObject to the study (any map) */
	void add(const InterfaceObject & io, ObjectMap & map);

	/** Remove a PersistentObject from the study (any map) */
	void remove(const InterfaceObject & io, ObjectMap & map);

	/** This method purges the study from the reloaded objects that are tagged unvisible */
	void cleanUnvisibleObject();

	/** This method performs the translation of ids after a study load */
	void translateId();


      private:

	/** Assignment */
	Study & operator =(const Study & other);

	/** A map of maps designated by their enum tag */
	MapCollection maps;

	/** The StorageManager that drives the load/store operations */
	StorageManagerImplementation p_storageManager_;

      }; /* class Study */


    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_STUDY_HXX */
