#! /usr/bin/env python
# -*- coding: utf-8 -*-

#   OpenTeacher
#   depends on: python-qt4
#
#   This program is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation, either version 3 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program.  If not, see <http://www.gnu.org/licenses/>.

import words
import errors

import xml.dom.minidom
import urllib2

class RequestXml(xml.dom.minidom.Document):
    """The RequestXml class is used to create xml understood by the WRTS-api. The
       xml is used to send wordLists to WRTS."""
    def __init__(self, wordList):
        xml.dom.minidom.Document.__init__(self)
        self.wordList = wordList

    def createXml(self):
        #Check if enough metadata is supplied, otherwise raise an error.
        if not (self.wordList.titleSet() and self.wordList.questionLanguageSet() and self.wordList.answerLanguageSet()):
            raise errors.NotEnoughMetadataError()

        #create the root-element
        listDom = self.createElement("list")
        self.appendChild(listDom)

        #create a words-element
        self.wordsDom = self.createElement("words")
        listDom.appendChild(self.wordsDom)

        #append a title, question language, and answer language element
        self.titleDom = self.createElement("title")
        self.titleDom.appendChild(self.createTextNode(self.wordList.title))
        listDom.appendChild(self.titleDom)

        self.questionLanguageDom = self.createElement("lang-a")
        self.questionLanguageDom.appendChild(self.createTextNode(self.wordList.questionLanguage))
        listDom.appendChild(self.questionLanguageDom)

        self.answerDom = self.createElement("lang-b")
        self.answerDom.appendChild(self.createTextNode(self.wordList.answerLanguage))
        listDom.appendChild(self.answerDom)

        for word in self.wordList:
            #create word-element
            wordDom = self.createElement("word")

            #append question
            questionWordDom = self.createElement("word-a")
            questionWordDom.appendChild(self.createTextNode(word.question))
            wordDom.appendChild(questionWordDom)

            #append answer (and second answer if one)
            answerWordDom = self.createElement("word-b")
            answer = word.answer
            if word.secondAnswerSet():
                answer += u"; %s" % word.secondAnswer
            answerWordDom.appendChild(self.createTextNode(answer))
            wordDom.appendChild(answerWordDom)

            #append word to wordsElement
            self.wordsDom.appendChild(wordDom)
        return self.toxml(encoding="UTF-8")

class WrtsConnection:
    """This class is used to keep a connection with WRTS. It stores authenticationdata and offers some
       methods which make it easy to get some data without the need of remembering the URL.

       Methods:
           login(email, password)
           exportWordList(wordList)
           importWordLIst(url)

        Properties:
            lists - gets all lists from the open account
            loggedIn - tells if the connection keeps valid (working) credentials of the user"""

    class HeadRequest(urllib2.Request):
        """This class is used to let urllib2 perform a HEAD-request."""
        def get_method(self):
            return "HEAD"

    def __init__(self):
        self.loggedIn = False

    def logIn(self, email, password):
        """Creates a connection to WRTS, with the authenticationdata inside it. Raises possibly a WrtsConnectionError/WrtsLoginError"""

        #Create connection
        unencoded = u"%s:%s" % (email, password)
        #to UTF-8, because we need binary
        unencoded = str(unencoded.encode("utf-8"))
        encoded = unencoded.encode("base64").strip()

        self._opener = urllib2.build_opener()
        self._opener.addheaders = [("Authorization", "Basic %s" % encoded),
                                   ("User-Agent", "OpenTeacher")]

        #Try loading the api; if not logged in it won't work, and raises a WrtsLoginError
        self._openUrl("http://www.wrts.nl/api", "HEAD")

    @property
    def lists(self):
        """Get all wrts-lists; returns a WrtsListParser instance."""
        xmlDom = self._openUrl("http://www.wrts.nl/api/lists")
        return ListsParser(xmlDom)

    def exportWordList(self, wordList):
        """Exports a wordList to WRTS, fully automatic after your login. Throws WrtsConnectionError"""
        #Create the xml-document and set the wordlist
        requestXml = RequestXml(wordList)

        #Send a POST request, with as body the xml
        self._openUrl("http://www.wrts.nl/api/lists", "POST", requestXml.createXml(), {"Content-Type": "application/xml"})

    def importWordList(self, url):
        """Downloads a WRTS wordlist from URL and parses it into a WordList object. Throws WrtsConnectionError/WrtsLoginError"""
        xml = self._openUrl(url)
        wordListParser = WordListParser(xml)

        #return the wordList
        return wordListParser.wordList

    def _openUrl(self, url, method="GET", body=None, additionalHeaders=None):
        """Open an url, and return the response as a xml.dom.minidom.Document. Can raise a WrtsConnectionError/WrtsConnectionError"""
        #If additionalHeaders not defined, they're set empty
        if not additionalHeaders:
            additionalHeaders = {}
        #Create a request object
        if method == "HEAD":
            request = self.HeadRequest(url, headers=additionalHeaders)
        elif method == "GET":
            request = urllib2.Request(url, headers=additionalHeaders)
        elif method == "POST":
            request = urllib2.Request(url, body, additionalHeaders)
        #Send it
        try:
            response = self._opener.open(request)
        except urllib2.HTTPError, e:
            if e.code == 401:
                #Not logged in
                self.loggedIn = False
                #Tell the user he/she isn't authorized.
                raise errors.WrtsLoginError()
            else:
                #Unknown status (most likely an error): not logged in
                self.loggedIn = False

                #Show for debugging:
                try:
                    print e.code, e.reason
                except AttributeError:
                    print e.code

                #But because it doesn't make sense to break the program for a WRTS error, show a nice error:
                raise errors.WrtsConnectionError()
        except urllib2.URLError, e:
            #Something wrong with the connection
            self.loggedIn = False
            #show for debugging
            print e
            #Show a nice error to the user.
            raise errors.WrtsConnectionError()

        #If no errors during request
        self.loggedIn = True

        if method == "HEAD":
            #HEAD never sends a body, so xml processing doesn't make sense.
            return

        return xml.dom.minidom.parse(response)

class ListsParser:
    """This class parses a wrts-api page: the lists-page. It can return the titles
       of the lists as a python list with unicode strings, and it an get the url of the
       corresponding wordList if you give the index of that title"""
    def __init__(self, listDom):
        unfilteredTitleDom = listDom.getElementsByTagName("title")
        self._titleDom = []
        #Filter titleDom for groups
        for title in unfilteredTitleDom:
            if title.parentNode.hasAttribute("href"):
                self._titleDom.append(title)

    @property
    def lists(self):
        #Create a list to store the titles in.
        lists = []
        #Add titles to the earlier created list
        for title in self._titleDom:
            #Get the title (which can - too bad - be empty at WRTS)
            try:
                lists.append(title.childNodes[0].nodeValue)
            except IndexError:
                lists.append(u"")
        #Return the titles
        return lists

    def getWordListUrl(self, index):
        """Gets the right node from the titleDom (selected by index),
           gets the parentNode of it, which has an attribute 'href',
           which is the needed url."""
        return self._titleDom[index].parentNode.getAttribute("href")

class WordListParser:
    """This class parses a wordlist from the WRTS-API into a WordList-instance."""
    def __init__(self, listDom):
        self.listDom = listDom

    @property
    def wordList(self):
        #Create a new WordList instance
        wordList = words.WordList()
        #Read title; sometimes the element is empty, so the except clauses are needed.
        try:
            title = self.listDom.getElementsByTagName("title")[0].childNodes[0].nodeValue
        except IndexError:
            title = u""

        #see title
        try:
            questionLanguage = self.listDom.getElementsByTagName("lang-a")[0].childNodes[0].nodeValue
        except IndexError:
            questionLanguage = u""

        #see title
        try:
            answerLanguage = self.listDom.getElementsByTagName("lang-b")[0].childNodes[0].nodeValue
        except IndexError:
            answerLanguage = u""

        #Set metadata inside the wordList
        wordList.title = title
        wordList.questionLanguage = questionLanguage
        wordList.answerLanguage = answerLanguage

        #Get the words
        wordsDom = self.listDom.getElementsByTagName("word")
        #Loop through the words in the xml
        for wordDom in wordsDom:
            #Create a Word-instance
            word = words.Word()

            #Read the question
            try:
                question = wordDom.getElementsByTagName("word-a")[0].childNodes[0].nodeValue
            except IndexError:
                question = u""

            #Read the answer
            try:
                answer = wordDom.getElementsByTagName("word-b")[0].childNodes[0].nodeValue
            except IndexError:
                answer = u""

            #Set the question inside the word.
            word.question = question

            #Try to find a second answer
            if "," in answer:
                splittedAnswer = answer.split(",", 1)
            elif ";" in answer:
                splittedAnswer = answer.split(";", 1)

            #Look if a second answer was found
            try:
                splittedAnswer
            except NameError:
                #No! Only add an answer.
                word.answer = answer
            else:
                #Yes! Add both the first and second answer
                word.answer = splittedAnswer[0]
                word.secondAnswer = splittedAnswer[1]
                #Remove splittedAnswer so it's not used in the anywhere further in the loop.
                del splittedAnswer

            # Add the current edited word to the wordList instance
            wordList.addWord(word)

        #And finally, return the wordList
        return wordList
