/*
 *  Copyright (c) 2009-2010 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Source.h"

using namespace OpenShiva;

#include <fstream>

#include <GTLCore/String.h>

#include "Debug.h"
#include "Lexer_p.h"
#include "LightParser_p.h"
#include "MetadataLexer_p.h"
#include "MetadataParser_p.h"
#include "Metadata.h"
#include "GTLCore/CompilationMessage.h"

#include "GTLCore/Metadata/Factory_p.h"
#include <GTLCore/CompilationMessages.h>
#include <GTLCore/SharedPointer.h>

struct Source::Private : public GTLCore::SharedPointerData
{
  Private() : metadata(0), metadataCompilationFailed(false), uptodate(false), type( InvalidSource ) {}
  Private(const Private& _rhs) : SharedPointerData(), name(_rhs.name), source(_rhs.source), metadata(0), metadataCompilationFailed(false), uptodate(false), type( InvalidSource ) {}
  
  void compileMetaData();
  void update();
  GTLCore::String name;
  GTLCore::String source;
  Metadata* metadata;
  bool metadataCompilationFailed;
  GTLCore::CompilationMessages compilationErrors;
  bool uptodate; ///< Hold wether name and type have been extracted from source
  SourceType type;
  ImageType outputImageType;
  std::vector<ImageType> inputImageTypes;
};


// ---- Library::Private ---- //

void Source::Private::compileMetaData()
{
  metadata = 0;
  std::istringstream iss(source);
  MetadataLexer* lexer = new MetadataLexer(&iss);
  MetadataParser parser( lexer, "" );
  metadata = parser.parse();
  compilationErrors = parser.errorMessages();
  metadataCompilationFailed = not compilationErrors.errors().empty();
  GTL_DEBUG( compilationErrors.errors().size() );
  GTL_DEBUG( parser.errorMessages().errors().size() );
  if( compilationErrors.errors().size() != 0 )
  {
    GTLCore::Metadata::Factory::deleteEntry( metadata );
    metadata = 0;
  }
}

void Source::Private::update()
{
  if(uptodate) return;
  std::istringstream iss(source);
  Lexer lexer( &iss );
  LightParser parser(&lexer);
  parser.parse();
  name = parser.name();
  type = parser.sourceType();
  outputImageType = parser.outputImageType();
  inputImageTypes = parser.inputImageTypes();
  if( type != Library)
  {
    switch( inputImageTypes.size() )
    {
      case 0:
        type = GeneratorKernel;
        break;
      case 1:
        type = FilterKernel;
        break;
      default:
        type = CompositionKernel;
        break;
    }
  }
  uptodate = true;
}

Source::Source() : d(new Private)
{
  d->ref();
}

GTL_SHARED_DATA(Source)

GTLCore::String Source::name() const
{
  d->update();
  return d->name;
}

GTLCore::String Source::source() const
{
  return d->source;
}

void Source::setSource( const GTLCore::String& _source)
{
  deref();
  GTLCore::Metadata::Factory::deleteEntry( d->metadata );
  d->metadata = 0;
  d->metadataCompilationFailed = false;
  d->uptodate = false;
  d->source = _source;
}

void Source::loadFromFile(const GTLCore::String& _fileName)
{
  GTLCore::String source = "";
  std::ifstream in;
  in.open(_fileName.c_str() );
  if(not in)
  {
    SHIVA_DEBUG( "Impossible to open file " << _fileName );
    return;
  }
  std::string str;
  std::getline(in,str);
  while ( in ) {
    source += str;
    source += "\n";
    std::getline(in,str);
  }
  setSource(source);
}

const Metadata* Source::metadata() const
{
  if(not d->metadata and not d->metadataCompilationFailed)
  {
    d->compileMetaData();
  }
  return d->metadata;
}

GTLCore::CompilationMessages Source::metadataCompilationMessages() const
{
  return d->compilationErrors;
}

Source::SourceType Source::sourceType() const
{
  d->update();
  return d->type;
}

Source::ImageType Source::outputImageType() const
{
  return d->outputImageType;
}

Source::ImageType Source::inputImageType(int idx) const
{
  if( idx >= 0 and (std::size_t)idx < d->inputImageTypes.size())
  {
    return d->inputImageTypes[idx];
  }
  return InvalidImage;
}

int Source::countInputImages() const
{
  return d->inputImageTypes.size();
}
