/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "ParameterEntry.h"

#include "../Debug.h"
#include "../Value.h"
#include "../Type.h"
#include "../TypesManager.h"

#include "TextEntry.h"
#include "ValueEntry.h"

using namespace GTLCore;
using namespace GTLCore::Metadata;

/**
  * Possible widget types.
  */
enum WidgetType {
  CheckBoxWidget, ///< checkbox (for booleans)
  IntegerWidget, ///< integer
  IntegerVectorWidget, ///< vector of integers
  FloatWidget, ///< float
  FloatVectorWidget, ///< vector of floats
  CurveWidget, ///< curve
  RgbColorWidget, ///< color selector
  RgbaColorWidget ///< color selector with RGBA
};

struct ParameterEntry::Private {
  Private() : type(0) {}
  Value minimumValue;
  Value maximumValue;
  Value defaultValue;
  const Type* type;
  GTLCore::String description;
  WidgetType widgetType;
};

ParameterEntry::ParameterEntry( const GTLCore::String& _name, const std::list< const Entry* >& _entries ) : Group( _name, _entries ), d(new Private)
{
  // Type
  const Entry* typeEntry = entry( "type" );
  bool widgetTypeSet = false;
  if( typeEntry )
  {
    const TextEntry* v = typeEntry->asTextEntry();
    GTL_ASSERT( v );
    if( v->text() == "int" )
    {
      d->type = Type::Integer32;
      d->widgetType = IntegerWidget;
      widgetTypeSet = true;
    } else if( v->text() == "float" ) {
      d->type = Type::Float32;
      d->widgetType = FloatWidget;
      widgetTypeSet = true;
    } else if( v->text() == "bool" ) {
      d->type = Type::Boolean;
      d->widgetType = CheckBoxWidget;
      widgetTypeSet = true;
    } else if( v->text() == "curve" ) {
      d->type = TypesManager::getArray( TypesManager::getVector( Type::Float32, 2 ) );
      d->widgetType = CurveWidget;
      widgetTypeSet = true;
    } else if( v->text() == "color" ) {
      d->type = Type::Color;
      d->widgetType = RgbaColorWidget;
      widgetTypeSet = true;
    } else if( v->text() == "float2" ) {
      d->type = TypesManager::getVector( Type::Float32, 2 );
      d->widgetType = FloatVectorWidget;
      widgetTypeSet = true;
    } else if( v->text() == "float3" ) {
      d->type = TypesManager::getVector( Type::Float32, 3 );
      d->widgetType = FloatVectorWidget;
      widgetTypeSet = true;
    } else if( v->text() == "float4" ) {
      d->type = TypesManager::getVector( Type::Float32, 4 );
      d->widgetType = FloatVectorWidget;
      widgetTypeSet = true;
    }
  }
  // DefaultValue
  const Entry* defaultEntry = entry( "defaultValue" );
  if( defaultEntry )
  {
    const ValueEntry* v = defaultEntry->asValueEntry();
    GTL_ASSERT( v );
    d->defaultValue = v->value();
    if( not d->type )
    {
      d->type = v->value().type();
    }
  }
  // MinValue
  const Entry* minEntry = entry( "minValue" );
  if( minEntry )
  {
    const ValueEntry* v = minEntry->asValueEntry();
    GTL_ASSERT( v );
    d->minimumValue = v->value();
    if( not d->type )
    {
      d->type = v->value().type();
    }
  }
  // MaxValue
  const Entry* maxEntry = entry( "maxValue" );
  if( maxEntry )
  {
    const ValueEntry* v = maxEntry->asValueEntry();
    GTL_ASSERT( v );
    d->maximumValue = v->value();
    if( not d->type )
    {
      d->type = v->value().type();
    }
  }
  // Description
  const Entry* descriptionEntry = entry( "description" );
  if( descriptionEntry )
  {
    const TextEntry* v = descriptionEntry->asTextEntry();
    GTL_ASSERT( v );
    d->description = v->text();
  }
  GTL_ASSERT( d->type );
  if( not minEntry )
  {
    if( d->type == Type::Integer32 )
    {
      d->minimumValue.setInt32( 0 );
    } else if( d->type == Type::Float32 )
    {
      d->minimumValue.setFloat32( 0.0 );
    }
  }
  if( not maxEntry )
  {
    if( d->type == Type::Integer32 )
    {
      d->maximumValue.setInt32( 100 );
    } else if( d->type == Type::Float32 )
    {
      d->maximumValue.setFloat32( 1.0 );
    }
  }
  if( not defaultEntry )
  {
    if( d->type == Type::Integer32 )
    {
      d->defaultValue.setInt32( 50 );
    } else if( d->type == Type::Float32 )
    {
      d->defaultValue.setFloat32( 0.5 );
    } else if( d->type == TypesManager::getArray( TypesManager::getVector( Type::Float32, 2 ) ) )
    {
      std::vector< GTLCore::Value > v1; v1.push_back( 0.0f ); v1.push_back( 0.0f );
      std::vector< GTLCore::Value > v2; v2.push_back( 1.0f ); v2.push_back( 1.0f );
      std::vector< GTLCore::Value > v3; v3.push_back( v1 ); v3.push_back( v2 );
      d->defaultValue.setArray( v3, d->type );
    } else if( d->type == TypesManager::getVector( Type::Float32, 3 ) )
    {
      std::vector< GTLCore::Value > v1; v1.push_back( 0.0f ); v1.push_back( 0.0f ); v1.push_back( 0.0f );
      d->defaultValue.setArray( v1, d->type );
    }
  }
  // Set the widget type if not set
  if( not widgetTypeSet )
  {
    switch( d->type->dataType() )
    {
      case Type::BOOLEAN:
        d->widgetType = CheckBoxWidget;
        break;
      case Type::INTEGER32:
        d->widgetType = IntegerWidget;
        break;
      case Type::FLOAT32:
        d->widgetType = FloatWidget;
        break;
      case Type::ARRAY:
        d->widgetType = CurveWidget;
        break;
      case Type::VECTOR:
        d->widgetType = FloatVectorWidget;
        break;
      default:
        GTL_ABORT("Unsupported datatype");
    }
  }
}

ParameterEntry::~ParameterEntry()
{
  delete d;
}

GTLCore::Value ParameterEntry::minimumValue() const
{
  return d->minimumValue;
}

GTLCore::Value ParameterEntry::maximumValue() const
{
  return d->maximumValue;
}

GTLCore::Value ParameterEntry::defaultValue() const
{
  return d->defaultValue;
}

const GTLCore::Type* ParameterEntry::type() const
{
  return d->type;
}

GTLCore::String ParameterEntry::description() const
{
  return d->description;
}

const ParameterEntry* ParameterEntry::asParameterEntry() const
{
  return this;
}
