/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "MetadataParser_p.h"

#include "GTLCore/ErrorMessage.h"
#include "GTLCore/Macros_p.h"
#include "GTLCore/Type.h"
#include "GTLCore/TypesManager.h"
#include "GTLCore/Value.h"

#include "GTLCore/Metadata/TextEntry.h"
#include "GTLCore/Metadata/ValueEntry.h"
#include "GTLCore/Metadata/Factory_p.h"
#include "GTLCore/Metadata/Group.h"
#include "GTLCore/Metadata/ParameterEntry.h"

#include "Debug.h"
#include "MetadataLexer_p.h"
#include "Metadata.h"

using namespace OpenShiva;

struct MetadataParser::Private {
  MetadataLexer* lexer;
  GTLCore::Token token;
  std::list<GTLCore::ErrorMessage> errorMessages;
  GTLCore::String fileName;
};

MetadataParser::MetadataParser( MetadataLexer* _lexer, const GTLCore::String& fileName) : d(new Private)
{
  d->lexer = _lexer;
  d->fileName = fileName;
}

MetadataParser::~MetadataParser()
{
  delete d->lexer;
  delete d;
}

OpenShiva::Metadata* MetadataParser::parse()
{
  const GTLCore::Metadata::Entry* version = 0;
  const GTLCore::Metadata::Group* infoList = 0;
  const GTLCore::Metadata::Group* parametersList = 0;
  getNextToken() ;
  if( currentToken().type == GTLCore::Token::INFERIOR )
  {
    getNextToken();
    while( currentToken().type != GTLCore::Token::SUPPERIOR and currentToken().type != GTLCore::Token::END_OF_FILE )
    {
      if( currentToken().type == GTLCore::Token::IDENTIFIER )
      {
        SHIVA_DEBUG( currentToken().string );
        GTLCore::String str = currentToken().string;
        getNextToken();
        isOfType( currentToken(), GTLCore::Token::COLON );
        getNextToken();
        if( str == "version" )
        {
          version = parseValueEntry("version");
        } else if( str == "info" ) {
          const GTLCore::Metadata::Entry* entry = parseGroupOrParameterEntry("info", false);
          if( entry )
          {
            infoList = entry->asGroup();
            GTL_ASSERT( infoList );
          }
        } else if( str == "parameters" ) {
          const GTLCore::Metadata::Entry* entry = parseGroupOrParameterEntry("parameters", true);
          if( entry )
          {
            parametersList = entry->asGroup();
            validateParameters( parametersList);
            GTL_ASSERT( parametersList );
          }
        } else {
          GTL_DEBUG("Unexpected");
          reportUnexpected( currentToken() );
          getNextToken();
        }
      } else {
        GTL_DEBUG("Unexpected");
        reportUnexpected( currentToken() );
        getNextToken();
      }
    }
  }
  return new Metadata( version, infoList, parametersList );
}

void MetadataParser::getNextToken()
{
  d->token = d->lexer->nextToken();
}

const GTLCore::Token& MetadataParser::currentToken()
{
  return d->token;
}

void MetadataParser::reportError( const GTLCore::String& errMsg, const GTLCore::Token& token )
{
  SHIVA_DEBUG( errMsg );
  d->errorMessages.push_back( GTLCore::ErrorMessage( errMsg, token.line, d->fileName ) );
}

void MetadataParser::reportUnexpected( const GTLCore::Token& token )
{
  reportError("Unexpected: " + GTLCore::Token::typeToString( token.type ), token );
  getNextToken();
}

const std::list<GTLCore::ErrorMessage>& MetadataParser::errorMessages() const
{
  return d->errorMessages;
}

bool MetadataParser::isOfType( const GTLCore::Token& token, GTLCore::Token::Type type )
{
  if( token.type == type )
  {
    return true;
  } else {
    reportError("Expected " + GTLCore::Token::typeToString(type) + " before " + GTLCore::Token::typeToString(token.type)  + ".", token);
    return false;
  }
}

const GTLCore::Metadata::ParameterEntry* MetadataParser::parseParameterEntry(const GTLCore::String& name)
{
  GTL_ABORT("oh");
  return 0;
}

const GTLCore::Metadata::TextEntry* MetadataParser::parseTextEntry( const GTLCore::String& name)
{
  GTL_DEBUG( "parseTextEntry" );
  if( currentToken().type == GTLCore::Token::STRING_CONSTANT or currentToken().type == GTLCore::Token::IDENTIFIER )
  {
    GTLCore::String v = currentToken().string;
    getNextToken();
    if( isOfType( currentToken(), GTLCore::Token::SEMI ) )
    {
      getNextToken();
      return GTLCore::Metadata::Factory::createTextEntry( name, v );
    }
  } else {
    reportUnexpected( currentToken() );
  }
  return 0;
}

GTLCore::Value MetadataParser::parseCoumpoundValue()
{
  std::vector< GTLCore::Value > values;
  enum { UNKNOWN, ARRAY, FLOAT } coumpoundType = UNKNOWN;
  if( isOfType( currentToken(), GTLCore::Token::STARTBRACE ) )
  {
    getNextToken();
    if( currentToken().type == GTLCore::Token::STARTBRACE )
    {
      coumpoundType = ARRAY;
    } else {
      coumpoundType = FLOAT;
    }
    while(true )
    {
      if( coumpoundType == ARRAY and isOfType( currentToken(), GTLCore::Token::STARTBRACE ) )
      {
        values.push_back( parseCoumpoundValue() );
        getNextToken();
      } else if( coumpoundType == FLOAT )
      {
        if( currentToken().type == GTLCore::Token::FLOAT_CONSTANT )
        {
          values.push_back( currentToken().f );
        } else if( currentToken().type == GTLCore::Token::INTEGER_CONSTANT )
        {
          values.push_back( (float)currentToken().i );
        } else {
          GTL_DEBUG("Unexpected");
          reportUnexpected( currentToken() );
        }
        getNextToken();
      } else {
        GTL_DEBUG("Unexpected");
        reportUnexpected( currentToken() );
        return values;
      }
      if( currentToken().type == GTLCore::Token::ENDBRACE )
      {
        const GTLCore::Type* type = 0;
        if( coumpoundType == FLOAT )
        {
          type = GTLCore::TypesManager::getVector( GTLCore::Type::Float, values.size() );
        }
        return GTLCore::Value(values, type);
      }
      if( not isOfType( currentToken(), GTLCore::Token::COMA ) )
      {
        return GTLCore::Value();
      }
      getNextToken();
    }
    getNextToken();
  }
  return GTLCore::Value();
}

const GTLCore::Metadata::ValueEntry* MetadataParser::parseValueEntry(const GTLCore::String& name)
{
  GTL_DEBUG( "parseValueEntry" );
  GTLCore::Value val;
  bool neg = false;
  if( currentToken().type == GTLCore::Token::MINUS )
  {
    neg = true;
    getNextToken();
  }
  if( currentToken().type == GTLCore::Token::FLOAT_CONSTANT )
  {
    if(neg)
    {
      val.setFloat( -currentToken().f );
    } else {
      val.setFloat( currentToken().f );
    }
  } else if( currentToken().type == GTLCore::Token::INTEGER_CONSTANT )
  {
    if(neg)
    {
      val.setInt32( -currentToken().i );
    } else {
      val.setInt32( currentToken().i );
    }
  } else if( currentToken().type == GTLCore::Token::STARTBRACE )
  {
    val = parseCoumpoundValue();
  } else {
    GTL_DEBUG("Unexpected");
    reportUnexpected( currentToken() );
    getNextToken();
    return 0;
  }
  getNextToken();
  if( isOfType( currentToken(), GTLCore::Token::SEMI ) )
  {
    getNextToken();
    return GTLCore::Metadata::Factory::createValueEntry( name, val );
  }
  getNextToken();
  return 0;
}

const GTLCore::Metadata::Entry* MetadataParser::parseGroupOrParameterEntry( const GTLCore::String& _name, bool _parameter)
{
  bool isAParameter = false;
  GTL_DEBUG( "parseListEntry" );
  std::list< const GTLCore::Metadata::Entry*> entries;
  if( isOfType( currentToken(), GTLCore::Token::INFERIOR ) )
  {
    getNextToken();
    while( currentToken().type != GTLCore::Token::SUPPERIOR and currentToken().type != GTLCore::Token::END_OF_FILE )
    {
      if( isOfType( currentToken(), GTLCore::Token::IDENTIFIER ) )
      {
        GTLCore::String str = currentToken().string;
        getNextToken();
        if( isOfType( currentToken(), GTLCore::Token::COLON ) )
        {
          getNextToken();
          const GTLCore::Metadata::Entry* entry = 0;
          if( currentToken().type == GTLCore::Token::INFERIOR )
          {
            entry = parseGroupOrParameterEntry( str, _parameter );
          } else if( not _parameter or str == "description" or str == "label" ) {
            GTL_DEBUG("t: " << str);
            entry = parseTextEntry( str );
          } else {
            GTL_DEBUG("t: " << str);
            if( str == "type" )
            {
              const GTLCore::Metadata::TextEntry* entryType = parseTextEntry( str );
              if( entryType )
              {
                if( entryType->text() == "int" or entryType->text() == "float"
                    or entryType->text() == "curve" or entryType->text() == "rgb"
                    or entryType->text() == "rgba" )
                {
                  isAParameter = true;
                  entry = entryType;
                } else {
                  GTLCore::Metadata::Factory::deleteEntry( entryType );
                }
              }
            } else if( str == "minValue" or str == "maxValue" or str == "defaultValue" )
            {
              isAParameter = true;
              entry = parseValueEntry( str );
            } else {
              GTL_DEBUG("Unexpected");
              reportUnexpected( currentToken() );
            }
          }
          if( entry )
          {
            entries.push_back( entry );
          }
        } else {
          break;
        }
      } else {
        break;
      }
    }
    getNextToken();
    if( isOfType( currentToken(), GTLCore::Token::SEMI ) )
    {
      getNextToken();
      if( isAParameter )
      {
        GTL_ASSERT( _parameter );
        return GTLCore::Metadata::Factory::createParameterEntry( _name, entries);
      } else {
        return GTLCore::Metadata::Factory::createGroup( _name, entries);
      }
    } else {
      foreach( const GTLCore::Metadata::Entry* entry, entries)
      {
        GTLCore::Metadata::Factory::deleteEntry( entry );
      }
      return 0;
    }
  }
  getNextToken();
  return 0;
}

void MetadataParser::validateParameters(const GTLCore::Metadata::Group* arg1)
{
  foreach(const GTLCore::Metadata::Entry* entry, arg1->entries())
  {
    if(const GTLCore::Metadata::ParameterEntry* pEntry = entry->asParameterEntry())
    {
      using GTLCore::Metadata::ParameterEntry;
      foreach(const GTLCore::Metadata::Entry* cEntry, pEntry->entries())
      {
        if(const GTLCore::Metadata::ValueEntry* vEntry = cEntry->asValueEntry())
        {
          const GTLCore::Value& val = vEntry->value();
          const GTLCore::Type* type = val.type();
          switch(pEntry->widgetType())
          {
            case ParameterEntry::CheckBoxWidget:
            case ParameterEntry::IntegerWidget:
            case ParameterEntry::FloatWidget:
              if(not type->isNumber())
              {
                reportError( pEntry->name() + "'s " + vEntry->name() + " requires a number.", currentToken());
              }
              break;
            case ParameterEntry::RgbColorWidget:
              if(type->dataType() != GTLCore::Type::VECTOR or val.asArray()->size() != 3)
              {
                reportError( pEntry->name() + "'s " + vEntry->name() + " requires a vector of three numbers.", currentToken());                
              }
              break;
            case ParameterEntry::RgbaColorWidget:
              if(type->dataType() != GTLCore::Type::VECTOR or val.asArray()->size() != 4)
              {
                reportError( pEntry->name() + "'s " + vEntry->name() + " requires a vector of four numbers.", currentToken());                
              }
              break;
            default:
              break;
          }
        }
      }
    } else if(const GTLCore::Metadata::Group* gEntry = entry->asGroup())
    {
      validateParameters(gEntry);
    }
  }
}
