/*
 *  Copyright (c) 2008-2009 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "PixelDescription.h"

#include "Debug.h"
#include "Type.h"

using namespace GTLCore;


struct PixelDescription::Private {
  std::vector<const Type*> channelsType;
  std::vector< std::size_t > channelPositions;
  int bitsSize;
  void initChannelPositions();
  int alphaPos;
};

void PixelDescription::Private::initChannelPositions()
{
  for( std::size_t i = 0; i < channelsType.size(); ++i )
  {
    channelPositions.push_back( i );
  }
  GTL_ASSERT( channelPositions.size() == channelsType.size() );
}

PixelDescription::PixelDescription( const Type* channelType_, int channels_) : d(new Private)
{
  GTL_ASSERT( channels_ > 0);
  GTL_ASSERT( channelType_->bitsSize() > 0 );
  d->bitsSize = channels_ * channelType_->bitsSize();
  d->channelsType.reserve(channels_);
  for(int i = 0; i< channels_; ++i)
  {
    d->channelsType.push_back( channelType_ );
  }
  d->initChannelPositions();
  d->alphaPos = -1;
}

PixelDescription::PixelDescription( const Type* channelType_, int channels_, int alphaPos_) : d(new Private)
{
  GTL_ASSERT( channels_ > 0);
  GTL_ASSERT( channelType_->bitsSize() > 0 );
  d->bitsSize = channels_ * channelType_->bitsSize();
  d->channelsType.reserve(channels_);
  for(int i = 0; i< channels_; ++i)
  {
    d->channelsType.push_back( channelType_ );
  }
  d->initChannelPositions();
  d->alphaPos = alphaPos_;
}

PixelDescription::PixelDescription( const std::vector<const Type* >& channelsType_) : d(new Private)
{
  GTL_ASSERT( channelsType_.size() > 0);
  d->channelsType = channelsType_;
  d->bitsSize = 0;
  for( std::vector<const Type*>::iterator it = d->channelsType.begin();
       it != d->channelsType.end(); ++it)
  {
    GTL_ASSERT( (*it)->bitsSize() > 0 );
    d->bitsSize += (*it)->bitsSize();
  }
  d->initChannelPositions();
  d->alphaPos = -1;
}

PixelDescription::PixelDescription( const std::vector<const Type* >& channelsType_, int _alphaPos) : d(new Private)
{
  GTL_ASSERT( channelsType_.size() > 0);
  d->channelsType = channelsType_;
  d->bitsSize = 0;
  for( std::vector<const Type*>::iterator it = d->channelsType.begin();
       it != d->channelsType.end(); ++it)
  {
    GTL_ASSERT( (*it)->bitsSize() > 0 );
    d->bitsSize += (*it)->bitsSize();
  }
  d->initChannelPositions();
  d->alphaPos = _alphaPos;
}

PixelDescription::PixelDescription( const PixelDescription& rhs) : d(new Private(*rhs.d))
{
}

PixelDescription& PixelDescription::operator=(const PixelDescription& rhs)
{
  *d = *rhs.d;
  return *this;
}

PixelDescription::~PixelDescription()
{
  delete d;
}

const std::vector<const Type*>& PixelDescription::channelTypes() const
{
  return d->channelsType;
}

std::size_t PixelDescription::channels() const
{
  return d->channelsType.size();
}

bool PixelDescription::sameTypeChannels() const
{
  return hasSameTypeChannels();
}

bool PixelDescription::hasSameTypeChannels() const
{
  const Type* _type = *d->channelsType.begin();
  for( std::vector<const Type*>::const_iterator it = d->channelsType.begin();
       it != d->channelsType.end(); ++it)
  {
    if( _type != *it )
      return false;
  }
  return true;
}

int PixelDescription::bitsSize() const
{
  return d->bitsSize;
}

bool PixelDescription::operator<( const PixelDescription& _rhs ) const
{
  if( d->bitsSize == _rhs.d->bitsSize )
  {
    if( d->channelsType.size() == _rhs.d->channelsType.size() )
    {
      for(unsigned int i = 0; i < d->channelsType.size(); ++i)
      {
        if( d->channelsType[i] != _rhs.d->channelsType[i] )
        {
          return d->channelsType[i] < _rhs.d->channelsType[i];
        }
      }
      return false;
    } else {
      return (d->channelsType.size() < _rhs.d->channelsType.size());
    }
    return false;
  } else
  {
    return (d->bitsSize < _rhs.d->bitsSize );
  }
}

bool PixelDescription::operator==( const PixelDescription& _rhs ) const
{
  if( d->channelsType.size() != _rhs.d->channelsType.size()
      and d->bitsSize != _rhs.d->bitsSize )
  {
    return false;
  } else {
    for(unsigned int i = 0; i < d->channelsType.size(); ++i)
    {
      if( d->channelsType[i] != _rhs.d->channelsType[i]
          or d->channelPositions[i] != _rhs.d->channelPositions[i] )
      {
        return false;
      }
    }
    return true;
  }
}

bool PixelDescription::operator!=( const PixelDescription& _rhs ) const
{
  return not ( *this == _rhs );
}

const std::vector< std::size_t >& PixelDescription::channelPositions() const
{
  return d->channelPositions;
}

void PixelDescription::setChannelPositions( const std::vector< std::size_t >& _positions )
{
  GTL_ASSERT( _positions.size() == channels() );
  d->channelPositions = _positions;
}

int PixelDescription::alphaPos() const
{
  return d->alphaPos;
}
