/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _CONVERT_CENTER_H_
#define _CONVERT_CENTER_H_

#include <utility>
#include <GTLCore/Export.h>

namespace GTLCore {
  namespace AST {
    class ConvertExpression;
    class Expression;
  }
  class ConvertExpressionFactory;
  class Type;
  /**
   * @internal
   * This class contains informations about all the possible conversion.
   * @ingroup GTLCore
   */
  class GTLCORE_EXPORT ConvertCenter {
    public:
      ConvertCenter();
      ~ConvertCenter();
    public:
      enum ConversionQuality {
          NONEEDED_CONVERSION, ///< indicate that no conversion is needed
          NOT_CONVERTIBLE, ///< indicates that there is no conversion between the two type
          LOSSLESS_CONVERSION, ///< indicates that the conversion is lossless (ie from integer to float)
          LOSS_CONVERSION ///< indicates there is a risk of losing information in the conversion (ie from float to integer)
      };
      /**
       * @return the conversion quality between two types.
       */
      ConversionQuality conversionQuality( const GTLCore::Type* _srcType, const GTLCore::Type* _dstType ) const;
    public:
      /**
       * Convert a value to a given type.
       * If the type of the \a value is allready \a _dstType , the function will return \a value .
       */
      AST::Expression* createConvertExpression( AST::Expression* value, const GTLCore::Type* _dstType ) const;
      /**
       * Create the conversions for binary operations:
       *  - first autoconvert type that needs to be autoconverted.
       *  - then convert the two given values to the highest priority type
       */
      std::pair<AST::Expression*, AST::Expression*> createConvertExpressions( AST::Expression* value1, AST::Expression* value2 ) const;
      void addConvertExpressionFactory( ConvertExpressionFactory* );
      void addAutoConversion( const GTLCore::Type* _srcType, const GTLCore::Type* _dstType );
      /**
       * @return the type to which \a _srcType needs to be converted
       */
      const GTLCore::Type* autoConvertType( const GTLCore::Type* _srcType ) const;
      /**
       */
    private:
      struct Private;
      Private* const d;
  };
  /**
   * @internal
   * This class allows to create the \ref AST::ConvertExpression that can convert between two different types.
   * @ingroup GTLCore
   */
  class GTLCORE_EXPORT ConvertExpressionFactory {
    public:
      ConvertExpressionFactory( );
      virtual ~ConvertExpressionFactory();
      /**
       * Create the \ref AST::ConvertExpression that can convert the given @p value to the given @p _dstType
       */
      virtual AST::ConvertExpression* create( AST::Expression* value, const GTLCore::Type* _dstType ) const = 0;
      /**
       * @return true if this \ref ConvertExpressionFactory can convert between @p srcType and @p dstType
       */
      virtual bool canConvertBetween( const GTLCore::Type* srcType, const GTLCore::Type* dstType) const = 0;
      /**
       * @return the conversion quality between two types.
       */
      virtual ConvertCenter::ConversionQuality conversionQuality( const GTLCore::Type* _srcType, const GTLCore::Type* _dstType ) const = 0;
    private:
      struct Private;
      Private* const d;
  };

}

#endif
