/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _GTLCORE_ABSTRACT_IMAGE_H_
#define _GTLCORE_ABSTRACT_IMAGE_H_

#include <GTLCore/Export.h>

namespace GTLCore {
  class Buffer;
  class PixelDescription;
  class Region;
  /**
   * Base class of Images. Reimplement the virtual functions to give access to
   * your own image data.
   * 
   * Alternatively you can use \ref BufferImage (with any \ref GTLCore::Buffer ) or
   * \ref Image .
   * 
   * @ingroup GTLCore
   */
  class GTLCORE_EXPORT AbstractImage {
    public:
      enum SamplingMode {
        Nearest,
        Linear
      };
      /**
       * Iterator allows to access Image data as buffer. A \ref BufferImage
       * will return an iterator that gives access to a single buffer,
       * while a titled image will give an iterator that after each call to \ref next
       * will give access to a different tile.
       */
      class ConstIterator {
        protected:
          ConstIterator();
        public:
          virtual ~ConstIterator();
        public:
          virtual bool next() = 0;
          virtual bool hasNext() const = 0;
          /**
           * @return the current buffer.
           */
          virtual const Buffer* current() const = 0;
          /**
           * @return the area covered by the current buffer.
           */
          virtual Region area() const = 0;
      };
      /**
       * Iterator allows to access Image data as buffer. A \ref BufferImage
       * will return an iterator that gives access to a single buffer,
       * while a titled image will give an iterator that after each call to \ref next
       * will give access to a different tile.
       *
       * This Iterator gives access to a non const buffer.
       */
      class Iterator : public ConstIterator {
        protected:
          Iterator();
        public:
          virtual ~Iterator();
        public:
          using ConstIterator::current;
          virtual Buffer* current() = 0;
      };
    public:
      AbstractImage( const GTLCore::PixelDescription& _pixelDescription );
      virtual ~AbstractImage();
    public:
      /**
       * @param _x
       * @param _y
       * @return a pointer to the pixel at coordinates (_x,_y)
       * 
       * If coordinates (_x,_y) are outside the image, it is still
       * expected for the \ref AbstractImage to return a valid pointer
       * to some default pixel value.
       */
      virtual char* data( int _x, int _y ) = 0;
      virtual const char* data( int _x, int _y ) const = 0;
      const GTLCore::PixelDescription& pixelDescription() const;
      /**
       * Compare two images over the \param _region .
       * 
       * @return the number of different pixel (or -1 if pixelDescription are
       *         different)
       */
      int compare( const AbstractImage* _image, const GTLCore::Region& _region) const;
      virtual ConstIterator* createIterator() const = 0;
      virtual Iterator* createIterator() = 0;
    protected:
      int pixelSize() const;
    private:
      struct Private;
      Private* const d;
  };
};


#endif
