/*###############################################################################
# Linux Management Providers (LMP), IP provider package
# Copyright (C) 2007 Ilsoo Byun <widepis@etri.re.kr ,widepis@empal.com>
#
# This program is being developed under the "OpenDRIM" project.
# The "OpenDRIM" project web page: http://opendrim.sourceforge.net
# The "OpenDRIM" project mailing list: opendrim@googlegroups.com
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; version 2
# of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#################################################################################

#################################################################################
# To contributors, please leave your contact information in this section
# AND comment your changes in the source code.
#
# Modified by 2008 Guillaume BOTTEX, ETRI <guillaumebottex@etri.re.kr, guillaumebottex@gmail.com>
###############################################################################*/

#include "OpenDRIM_IPProtocolEndpointAccess.h"
#include "Common/IP_Common.h"

static const string systemCreationClassName = "OpenDRIM_ComputerSystem";
static const string creationClassName = "OpenDRIM_IPProtocolEndpoint";
static string systemName;

int IP_OpenDRIM_IPProtocolEndpoint_load(const CMPIBroker* broker, string& errorMessage) {
	_E_;
	CF_assert(CF_getSystemName(systemName, errorMessage));
	_L_;
	return OK;
}

int IP_OpenDRIM_IPProtocolEndpoint_unload(string& errorMessage) {
	_E_;
	// Nothing
	_L_;
	return OK;
}

int IP_OpenDRIM_IPProtocolEndpoint_retrieve(const CMPIBroker* broker, const CMPIContext* ctx, vector<OpenDRIM_IPProtocolEndpoint>& result, const char** properties, string& errorMessage, const string& discriminant) {
	_E_;
	vector<string> eth_names;
	CF_assert(CF_getEthernetPortNames(eth_names, errorMessage));

	//Initialize an instance
	OpenDRIM_IPProtocolEndpoint instance;
	instance.setSystemCreationClassName(systemCreationClassName);
	instance.setSystemName(systemName);
	instance.setCreationClassName(creationClassName);

	for(unsigned int i=0;i<eth_names.size();i++)
	{
		if (!CF_startsWith(eth_names[i], "eth"))
			continue;

		instance.setName(eth_names[i]);
		if (discriminant == "ei") {
			CF_assert(IP_OpenDRIM_IPProtocolEndpoint_populate(instance, errorMessage));
		}
		result.push_back(instance);
	}
	_L_;
	return OK;
}

int IP_OpenDRIM_IPProtocolEndpoint_getInstance(const CMPIBroker* broker, const CMPIContext* ctx, OpenDRIM_IPProtocolEndpoint& instance, const char** properties, string& errorMessage) {
	_E_;
	vector<string> if_names;
	vector<string>::size_type index;

	CF_assert(CF_getEthernetPortNames(if_names, errorMessage));

	if (instance.SystemCreationClassName == systemCreationClassName &&
			instance.CreationClassName == creationClassName &&
			instance.SystemName == systemName &&
			CF_foundInList(instance.Name,if_names,index))
	{
		CF_assert(IP_OpenDRIM_IPProtocolEndpoint_populate(instance, errorMessage));
	}
	else
	{
		errorMessage = "No instance";
		return NOT_FOUND;
	}
	_L_;
	return OK;
}

int IP_OpenDRIM_IPProtocolEndpoint_setInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_IPProtocolEndpoint& newInstance, const OpenDRIM_IPProtocolEndpoint& oldInstance, const char** properties, string& errorMessage) {
	_E_;
	_L_;
	return NOT_SUPPORTED;
}

int IP_OpenDRIM_IPProtocolEndpoint_createInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_IPProtocolEndpoint& instance, string& errorMessage) {
	_E_;
	_L_;
	return NOT_SUPPORTED;
}

int IP_OpenDRIM_IPProtocolEndpoint_deleteInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_IPProtocolEndpoint& instance, string& errorMessage) {
	_E_;
	_L_;
	return NOT_SUPPORTED;
}

int IP_OpenDRIM_IPProtocolEndpoint_RequestStateChange(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_IPProtocolEndpoint& instance, unsigned int& returnValue, const OpenDRIM_IPProtocolEndpoint_RequestStateChange_In& in, OpenDRIM_IPProtocolEndpoint_RequestStateChange_Out& out, string& errorMessage) {
	_E_;
	_L_;
	return OK;
}

int IP_OpenDRIM_IPProtocolEndpoint_BroadcastReset(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_IPProtocolEndpoint& instance, unsigned int& returnValue, string& errorMessage) {
	_E_;
	_L_;
	return OK;
}

int IP_OpenDRIM_IPProtocolEndpoint_populate(OpenDRIM_IPProtocolEndpoint& instance, string& errorMessage) {
	_E_;
	/*
	 * Properties to fill from profile
	 * + Mandatory:
	 * [X] SystemCreationClassName [KEY]
	 * [X] CreationClassName       [KEY]
	 * [X] SystemName              [KEY]
	 * [X] Name                    [KEY]
	 * [X] NameFormat
	 * [X] ProtocolIFType
	 * [X] RequestedState
	 * [X] EnabledState
	 * [X] ElementName
	 * [X] AddressOrigin
	 * + Conditional:
	 * [X] IPv4Address
	 * [X] SubnetMask
	 * + Other:
	 * [X] TimeOfLastStateChange
	 * [X] EnabledDefault
	 */

	ifstream ifs;
	string ip;
	string subnetmask;
	unsigned int enabledState;

	instance.setNameFormat("network-interface-name");
	instance.setProtocolIFType(IFTYPE_IPv4);
	instance.setRequestedState(RS_NOT_APPLICABLE);

	if(instance.ProtocolIFType==IFTYPE_IPv4)
		instance.setElementName(instance.Name+" IPv4 Protocol Endpoint");

	string ifconfig_str, temp;
	CF_assert(CF_runCommand("/sbin/ifconfig "+instance.Name, ifconfig_str, temp, errorMessage));
	vector<string> ifconfig_parsed;
	CF_splitText(ifconfig_parsed, ifconfig_str, '\n');

	CF_getIfEnabledState(instance.Name,enabledState ,errorMessage);
	instance.setEnabledState(enabledState);

	// EnabledDefault
	// From DMTF Schema: By default, the element is "Enabled"
	instance.setEnabledDefault(ED_ENABLE);

	// AddressOrigin
	// set to default value
	instance.setAddressOrigin(AO_UNKNOWN);

	CF_getIP(instance.Name,ip,errorMessage);
	if(ip!="")
		instance.setIPv4Address(ip);

	CF_getSubNetMask(instance.Name,subnetmask,errorMessage);
	if(subnetmask!="")
		instance.setSubnetMask(subnetmask);

	string device_path;
	string line;

	if (!CF_isExist(DEVICE_BASE_PATH))
	{
		device_path = DEBIAN_DEVICE_FILE;
		ifs.open(device_path.c_str(), ifstream::in);

		if(!ifs)
		{
			errorMessage = "Cannot open file: "+device_path;
			return FAILED;
		}

		bool foundAuto=false;
		while (getline(ifs, line))
		{
			vector<string> tokens;

			CF_splitText(tokens,line," ");

			if(tokens.size()>1 && tokens[1]==instance.Name && tokens[0]=="iface")
			{
				if(tokens[3]=="static")
					instance.setAddressOrigin(AO_STATIC);
				else if(tokens[3]=="dhcp")
					instance.setAddressOrigin(AO_DHCP);

				if(ip=="" || subnetmask=="")
				{
  				while(line!="")
  				{
  					getline(ifs, line);
  					line=CF_trimText(line);
  					CF_splitText(tokens,line," ");

  					if(tokens[0]=="address" && ip=="")
  						instance.setIPv4Address(tokens[1]);
  					else if(tokens[0]=="netmask" && subnetmask=="")
  						instance.setSubnetMask(tokens[1]);
  				}
				}
			}
			else if(tokens.size()>1 && tokens[1]==instance.Name && tokens[0]=="auto")
			{
				instance.setEnabledDefault(ED_ENABLE);
				foundAuto=true;
			}

		}
		ifs.close();

		if(!foundAuto)
			instance.setEnabledDefault(ED_DISABLE);

	}
	else
	{
		device_path = DEVICE_BASE_PATH+"ifcfg-"+instance.Name;
		ifs.open(device_path.c_str(), ifstream::in);

		if(!ifs)
		{
			errorMessage = "Cannot open file: "+device_path;
			return OK;
		}

		bool foundBootProto=false;
		while (getline(ifs, line))
		{
			int idx = line.find_first_of('=');
			string name = CF_trimText(line.substr(0, idx));
			string value = CF_trimText(line.substr(idx+1));

			if (name == "BOOTPROTO" && !foundBootProto) // AddressOrigin
			{
				if (value == "none" || value == "static")
					instance.setAddressOrigin(AO_STATIC);
				else if (value == "dhcp")
					instance.setAddressOrigin(AO_DHCP);
				else if (value == "bootp")
					instance.setAddressOrigin(AO_BOOTP);
				else
					instance.setAddressOrigin(AO_UNKNOWN);
				foundBootProto=true;
			}
			else if (name == "ONBOOT")	// EnabledDefault
			{
				if (value == "yes")
					instance.setEnabledDefault(ED_ENABLE);
				else
					instance.setEnabledDefault(ED_DISABLE);
			}

		}

		if(!foundBootProto)
			instance.setAddressOrigin(AO_STATIC);

		ifs.clear();
		ifs.seekg (0, ios::beg);

		while (getline(ifs, line))
		{
			int idx = line.find_first_of('=');
			string name = CF_trimText(line.substr(0, idx));
			string value = CF_trimText(line.substr(idx+1));

			if(ip=="" && instance.AddressOrigin==AO_STATIC)
			{
					if (name == "IPADDR")
						instance.setIPv4Address(value);
			}

			if(subnetmask=="" && instance.AddressOrigin==AO_STATIC)
			{
					if (name == "NETMASK")
						instance.setSubnetMask(value);
			}
		}
		ifs.close();
	}

	//TimeOfLastStateChange
	string modifiedTime;
	CF_assert(CF_lastModified(device_path, modifiedTime, errorMessage));
	instance.setTimeOfLastStateChange(modifiedTime);
	_L_;
	return OK;
}

