/* tag: openbios forth environment, executable code
 *
 * Copyright (C) 2003 Patrick Mauritz, Stefan Reinauer
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */

#include "openbios/config.h"
#include "openbios/bindings.h"
#include "asm/types.h"
#include "dict.h"
#include "openbios/kernel.h"
#include "openbios/stack.h"
#include "openbios/drivers.h"
#include "openbios/pci.h"
#include "sys_info.h"
#include "openbios.h"
#include "relocate.h"

void boot(void);

static unsigned char intdict[256 * 1024];

#ifdef CONFIG_DRIVER_PCI
static const pci_arch_t default_pci_host = {
    .name = "Intel,i440FX",
    .vendor_id = PCI_VENDOR_ID_INTEL,
    .device_id = PCI_DEVICE_ID_INTEL_82441,
    .io_base = 0x1000,
};
#endif

static void init_memory(void)
{
	/* push start and end of available memory to the stack
	 * so that the forth word QUIT can initialize memory
	 * management. For now we use hardcoded memory between
	 * 0x10000 and 0x9ffff (576k). If we need more memory
	 * than that we have serious bloat.
	 */

	PUSH(0x10000);
	PUSH(0x9FFFF);
}

static void
arch_init( void )
{
	void setup_timers(void);

	modules_init();
#ifdef CONFIG_DRIVER_PCI
        arch = &default_pci_host;
	ob_pci_init();
#endif
#ifdef CONFIG_DRIVER_IDE
	setup_timers();
	ob_ide_init("/pci/isa", 0x1f0, 0x3f4, 0x170, 0x374);
#endif
#ifdef CONFIG_DRIVER_FLOPPY
	ob_floppy_init("/isa", "floppy0", 0x3f0, 0);
#endif
#ifdef CONFIG_XBOX
	init_video(phys_to_virt(0x3C00000), 640, 480, 32, 2560);
	node_methods_init();
#endif
	device_end();
	bind_func("platform-boot", boot );
}

int openbios(void)
{
	extern struct sys_info sys_info;
#ifdef CONFIG_DEBUG_CONSOLE
#ifdef CONFIG_DEBUG_CONSOLE_SERIAL
	uart_init(CONFIG_SERIAL_PORT, CONFIG_SERIAL_SPEED);
#endif
	/* Clear the screen.  */
	cls();
#endif

        collect_sys_info(&sys_info);

	dict=intdict;
	load_dictionary((char *)sys_info.dict_start,
			(unsigned long)sys_info.dict_end -
			(unsigned long)sys_info.dict_start);

	relocate(&sys_info);

#ifdef CONFIG_DEBUG_CONSOLE_VGA
	video_init();
#endif
#ifdef CONFIG_DEBUG_BOOT
	printk("forth started.\n");
	printk("initializing memory...");
#endif

	init_memory();

#ifdef CONFIG_DEBUG_BOOT
	printk("done\n");
#endif

	PUSH_xt( bind_noname_func(arch_init) );
	fword("PREPOST-initializer");

	PC = (ucell)findword("initialize-of");

	if (!PC) {
		printk("panic: no dictionary entry point.\n");
		return -1;
	}
#ifdef CONFIG_DEBUG_DICTIONARY
	printk("done (%d bytes).\n", dicthead);
	printk("Jumping to dictionary...\n");
#endif

	enterforth((xt_t)PC);

	return 0;
}
