MODULE AllModules;

IMPORT
  (* Make Framework
     ---------------------------------------------------------------------- *)
  OOC:Make,
  OOC:Make:TranslateToC,
  OOC:Make:WriteMainFileC,
  OOC:Make:TranslateToIA32,
  OOC:Make:WriteMainFileAssembler,
  OOC:Make:LinkProgramC,
  
(* Miscellaneous Modules
   ------------------------------------------------------------------------ *)
  OOC:Error,
    (* Handles error lists for the compiler. It is specialization of 
       @omodule{Msg}.  *)
    
    
(* Scanner
   ------------------------------------------------------------------------ *)
  OOC:Scanner:Symbol,
    (* Defines symbolic names for the symbols recognized by the parser.
       Provides mappings from symbol id to symbol name and symbol
       representation.  *)
       
  OOC:Scanner:InputBuffer,
    (* Provides the buffering mechanism for the scanner module.  *)
    
  ScannerBuilder := OOC:Scanner:Builder,
    (* Builder interface for the scanner module.  The scanner calls factory
       methods from this module to create symbol objects.  *)
       
  OOC:Scanner,
    (* This is the module that does the actual scanning work.  It reads a
       file and calls the factory methods of the builder to create a symbol
       instance for every token it recognizes.  *)
       
       
  OOC:Scanner:BasicList,
    (* This builder module creates a list of symbols from the scanner's data.
       This list is subsequently fed to @oproc{OOC:Parser.Module}.  *)
       
  OOC:Scanner:Identity,
    (* Debugging builder: Writes symbols unchanged to the output channel.  *)
    
  OOC:Scanner:Statistics,
    (* Displays some statistics about the symbols used in the source text.  *)
    
  ScannerXML := OOC:Scanner:XML,
    (* Debugging builder: Converts list of symbols generated by the scanner
       into an XML document.  Every symbol becomes an element, with the
       symbol's attributes appearing in the element's start tag.  *)
       
  
(* Abstract Syntax Tree
   ------------------------------------------------------------------------ *)
  OOC:AST,
    (* Defines the abstract builder class that is used by the parser to
       register parsed objects.  For every production of the Oberon-2 syntax,
       there exists a corresponding factory method.  *)
  
  OOC:AST:ExtTree,
    (* Concrete implementation of the builder interfaces defined by 
       @omodule{OOC:AST}.  It creates an abstract syntax tree that is
       a one-to-one representation of the data passed that is passed to 
       the builder by the parser.  *)
  
  OOC:AST:ExtTree:CreateSymTab,
    (* Creates the symbol table from the abstract syntax tree defined by
       module @omodule{OOC:AST:ExtTree}.  Uses the factory methods of
       @omodule{OOC:SymbolTable:Builder} to populate the symbol table.  *)
  
  ExtTreeXML := OOC:AST:ExtTree:XML,
    (* Writes an abstract syntax tree into an XML document that describes the
       structure of the syntax tree.  The only information lost is formatting,
       comments, and pragmas.  *)
  
  OOC:AST:ExtTree:StupidPP,
    (* A simple pretty printer: It writes the abstract syntax tree to the
       output channel, omitting any comments.  The output is the complete
       contents of the source module, using a very rigid indentation and
       formatting style.  *)


(* Parser
   ------------------------------------------------------------------------ *)
  OOC:Parser,
    (* Parses an Oberon-2 module and creates an abstract syntax tree by
       calling the appropriate builder methods of @otype{OOC:AST.Builder}.
       This module exports a single procedure, @oproc{OOC:Parser.Module},
       whose result is the root of the constructed syntax tree.  *)


(* Symbol Table
   ------------------------------------------------------------------------ *)
  Sym := OOC:SymbolTable,
    (* Defines the entries and structure of the standard symbol table.
       Provides the basic namespace interface, and the basic infrastructure
       for storage and retrieval of persistent symbol table data.  *)
  
  OOC:SymbolTable:XML,
    (* Writes a symbol table to an XML document.  *)
  
  OOC:SymbolTable:GetClass,
    (* Determines the class name for a given item.  This is mainly used
       to structure documentation automatically generated from sources
       files.  *)
       
  SymbolTableBuilder := OOC:SymbolTable:Builder,
    (* Defines a builder class with factory methods for the symbol table
       entries defined in @omodule{Sym := OOC:SymbolTable}.  *)
  
  OOC:SymbolTable:Predef,
    (* Defines symbol table of pseudo module with predefined entities,
       like @samp{CHAR} or @samp{COPY}.  *)
  
  OOC:SymbolTable:TypeRules,
    (* Defines the type rules of the language.  *)

  OOC:SymbolTable:Namespace,
    (* This module defines the module, procedure, and record namespaces.  It
       also implements semantic checks on these namespaces.  *)
       
  OOC:SymbolTable:CreateNamespace,
    (* Converts a raw symbol table, as derived from the module by the parser,
       into a fully fledged symbol table with namespace information, and
       replaces type names with the actual type.  Any error in a module's
       declarations are reported by this module.  *)

  
  OOC:Repository,
    (* Defines the repository data structure, and the operations on it to
       locate a module's files and to create new files in it.  *)
  RepFileSystem := OOC:Repository:FileSystem,
    (* A repository implementation using the local file system.  *)
  
  OOC:Config:Pragmas,
  OOC:Config:StdPragmas,
  
  OOC:Config:Repositories,
    (* Holds the configuration for the set of repositories accessible
       to the compiler.  Locates files in the configured repositories.  *)
  ConfigRepFileSystem := OOC:Config:FileSystem,
    (* Parses the data from the symbol file that describes a file
       system based repository.  *)
  OOC:Config:CCompiler,
  OOC:Config:Assembler,
  
  OOC:Auxiliary:ParseModule,

  
  (* Intermediate Code Representation (aka IR)
     ---------------------------------------------------------------------- *)
  OOC:IR,
  (* High-level code intermediate representation.  This module defines classes
     for the various statements and expressions in Oberon-2, together with a
     builder class to create instances from them.  All semantic checks are done
     in this module: once a module's IR code has been completely build without
     any errors, further processing of the code must @emph{not} cause any error
     message to be emitted by the compiler.  *)

  OOC:IR:ConstFold,
  (* Constant folding on IR expression.  For example, this module reduces
     the expression on the right hand side of a constant declaration to a
     single scalar value.  *)

  IRXML := OOC:IR:XML,
  (* Write an IR data structure as an XML document.  *)

  OOC:IR:VisitAll,
  (* A vistor traversing the whole IR data structure.  *)

  
  (* SSA Back-end
     ---------------------------------------------------------------------- *)
  OOC:SSA,
  (* Defines the SSA datastructure: results, instructions, operands, and the
     like.  Also provides the basic operations that oeprate on the elements of
     a procedure's SSA representation.  *)
  
  OOC:SSA:Opcode,
  (* Defines the opcodes of SSA instructions.  The documentation of the
     opcodes can also be found here.  *)
  
  OOC:SSA:Opnd,
  (* Defines symbolic names for SSA operand classes.  *)

  OOC:SSA:Result,
  (* Defines symbolic names for SSA result classes.  *)
  
  OOC:SSA:IRtoSSA,
  (* Converts the IR of a procedure into (hopefully equivalent) SSA
     representation.  *)

  OOC:SSA:Aliasing,
  (* Calculates aliasing between instructions.  That is, this module provides
     the algorithms to determine if two instructions operate or potentially
     operate on the same set of memory locations.  *)
  
  OOC:SSA:Destore,
  (* The initial translation to SSA code turns all accesses to variables to
     memory read and write.  This module inspects these reads and writes and
     tries to avoid writing values to memory that can be passed through
     registers in the final assembler code), and reading values from memory
     that are available from registers.  (As far as SSA is concerned, scalar
     values passed between instructions roughly correspond to values passed
     through registers.)  *)

  OOC:SSA:Blocker,
  (* Imposes a block structure on a flat list of SSA instructions.  The result
     resembles a guarded command structure like it is used in GSA.  This module
     is used by two different kinds of clients.  First, SSA visualization
     modules use it to improve readability of the rendered SSA code.  Second,
     the code scheduler takes ``blocked'' SSA as input to create basic blocks
     and jumps&branches between them.  *)

  OOC:SSA:Schedule,
  (* Generates basic block representation and schedules instructions.  This
     transforms an SSA representation that is free of explicit control flow
     information into a representation that uses the lower-level concept of
     basic blocks and explicit jumps&branches between them.  *)
  
  OOC:SSA:IGraph,
  (* Manipulation of interference graph of values' life ranges.  Used by the
     allocator module below.  *)
  
  OOC:SSA:Allocator,
  (* A register allocator for the C back-end.  When writing C code, local
     variables of a procedure are replaced with numbered @code{register}
     variables as far as possible.  *)
  
  OOC:SSA:WriteC,
  (* Translates SSA instructions to C.  Input is the IR of a procedure body.
     This is converted to SSA, any transformations are applied to the SSA
     result, which, after scheduling of instructions and register allocation,
     is written out as C  *)
     
  SSAXML := OOC:SSA:XML,
  (* Writes out the SSA representation of a procedure body as an XML
     document.  *)
  

  (* Test Programs
     ---------------------------------------------------------------------- *)
  TestScanner, TestParser, TestSymTab, TestConfigSections,
  TestConfigCmdLine, TestConfigEnv, TestConfigSimple, TestInterfaceGen,
  TestTexinfo, TestCompile,
  
  RunTests;
  (* Implements the XML based test framework.  Used for language
     compliance tests, and to check results of executables.  *)

  (* Library Modules
     ----------------------------------------------------------------------

     Note: Library modules should be added to this list as they are moved into
     the ooc2 source tree under lib/src/ and have been successfully tested with
     the new compiler.  At the moment, this requires that a decent test
     coverage of the provided functions exist, and that the SSA back-end does
     not fail on them.  Because ooc2 must be build with oo2c v1 and its core
     library for the forseeable future, the modules listed here are not used
     directly by ooc2.  The list is therefore purely informal.

     Ascii
     CharClass
     Strings
     
     ----  *)

END AllModules.
