/* OpenCP Module Player
 * copyright (c) '94-'98 Niklas Beisert <nbeisert@physik.tu-muenchen.de>
 *
 * Interface routines for Audio CD player
 *
 * revision history: (please note changes here)
 *  -nb980510   Niklas Beisert <nbeisert@physik.tu-muenchen.de>
 *    -first release
 *  -ss040907   Stian Skjelstad <stian@nixia.no>
 *    -removed cdReadDir
 *    -Linux related changes
 */

#include "config.h"
#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <linux/cdrom.h>
#include "types.h"
#include "cdaudio.h"
#include "boot/plinkman.h"
#include "cpiface/cpiface.h"
#include "dev/deviplay.h"
#include "dev/devisamp.h"
#include "filesel/pfilesel.h"
#include "filesel/mdb.h"
#include "stuff/compat.h"
#include "stuff/poutput.h"
#include "stuff/sets.h"

#include <curses.h>

void _splitpath(const char *src, char *drive, char *path, char *file, char *ext); /* defined in stuff/compat.h but this header clashed with linux/cdrom.h */

static unsigned long cdpTrackStarts[CDROM_LEADOUT+1];
static unsigned char cdpFirstTrack;
static unsigned char cdpPlayMode; /* 1 = disk, 0 = track */
static unsigned char cdpTrackNum; /* current track, calculated in GStrings */
static int cdpViewSectors; /* ??? view-option */
static FILE *cdpDrive; /* nasty */
static unsigned long basesec; /* current start... usually a cdpTrack[n] value */
static unsigned long length; /* and the length of it */
static signed long newpos; /* skip to */
static unsigned char setnewpos; /* and the fact we should skip */
static char vdev[8];

static int16_t speed;
static char finespeed=8;
static uint32_t pausefadestart;
static uint8_t pausefaderelspeed;
static int8_t pausefadedirect;

static void startpausefade(void)
{
	if (pausefadedirect)
	{
		if (pausefadedirect<0)
			plPause=1;
		pausefadestart=2*dos_clock()-DOS_CLK_TCK-pausefadestart;
	} else
		pausefadestart=dos_clock();

	if (plPause)
	{
		plChanChanged=1;
		plPause=0;
		cdRestart(fileno(cdpDrive));
		pausefadedirect=1;
	} else
		pausefadedirect=-1;
}

static void normalize(void)
{
	mcpNormalize(0);
	speed=set.speed;
	cdSetSpeed(speed);
}

static void dopausefade(void)
{
	int16_t i;
	if (pausefadedirect>0)
	{
		i=((int32_t)dos_clock()-pausefadestart)*64/DOS_CLK_TCK;
		if (i<0)
			i=0;
		if (i>=64)
		{
			i=64;
			pausefadedirect=0;
		}
	} else {
		i=64-((int32_t)dos_clock()-pausefadestart)*64/DOS_CLK_TCK;
		if (i>=64)
			i=64;
		if (i<=0)
		{
			i=0;
			pausefadedirect=0;
			plPause=1;
			cdPause(fileno(cdpDrive));
			plChanChanged=1;
			cdSetSpeed(speed);
			return;
		}
	}
	pausefaderelspeed=i;
	cdSetSpeed(speed*i/64);
}
static char *gettimestr(unsigned long s, char *time)
{
	unsigned char csec, sec, min;
	csec=(s%75)*4/3;
	time[8]=0;
	time[7]='0'+csec%10;
	time[6]='0'+csec/10;
	sec=(s/75)%60;
	time[5]=':';
	time[4]='0'+sec%10;
	time[3]='0'+sec/10;
	min=s/(75*60);
	time[2]=':';
	time[1]='0'+min%10;
	time[0]='0'+min/10;
	return time;
}

static void cdaDrawGStrings(uint16_t (*buf)[CONSOLE_MAX_X])
{
	int i;
	char timestr[9];

	struct cdStat stat;

	cdGetStatus(fileno(cdpDrive), &stat);


	memset(buf[0]+80, 0, (plScrWidth-80)*sizeof(uint16_t));
	memset(buf[1]+80, 0, (plScrWidth-80)*sizeof(uint16_t));
	memset(buf[2]+80, 0, (plScrWidth-80)*sizeof(uint16_t));

	writestring(buf[0], 0, 0x09, "  mode: ..........  ", 20);
      	writestring(buf[0], 8, 0x0F, "cd-audio", 10);
	for (i=1; i<=cdpTrackNum; i++)
		if (stat.position<cdpTrackStarts[i])
			break;

	writestring(buf[0], 20, 0x09, "playmode: .....  status: .......", plScrWidth-20);
	writestring(buf[0], 30, 0x0F, cdpPlayMode?"disk ":"track", 5);
	writestring(buf[0], 45, 0x0F, (stat.error)?"  error":(stat.paused)?" paused":"playing", 7);
	writestring(buf[0], 52, 0x0F, "                 speed: ...%", 28);
	_writenum(buf[0], 76, 0x0F, stat.speed*100/256, 10, 3);

	writestring(buf[1], 0, 0x09, "drive: ....... start:   :..:..  pos:   :..:..  length:   :..:..  size: ...... kb", plScrWidth);
	writestring(buf[1], 7, 0x0F, vdev, 7); /* VERY TODO.. can we fit more data on this line??? */
	if (cdpViewSectors)
	{
		writenum(buf[1], 22, 0x0F, cdpTrackStarts[0], 10, 8, 0);
		writenum(buf[1], 37, 0x0F, stat.position-cdpTrackStarts[0], 10, 8, 0);
		writenum(buf[1], 55, 0x0F, cdpTrackStarts[cdpTrackNum+1]-cdpTrackStarts[0], 10, 8, 0);
	} else {
		writestring(buf[1], 22, 0x0F, gettimestr(cdpTrackStarts[0]+150, timestr), 8);
		writestring(buf[1], 37, 0x0F, gettimestr(stat.position-cdpTrackStarts[0], timestr), 8);
		writestring(buf[1], 55, 0x0F, gettimestr(cdpTrackStarts[cdpTrackNum+1]-cdpTrackStarts[0], timestr), 8);
	}
	_writenum(buf[1], 71, 0x0F, (cdpTrackStarts[cdpTrackNum+1]-cdpTrackStarts[0])*147/64, 10, 6);

	writestring(buf[2], 0, 0x09, "track: ..      start:   :..:..  pos:   :..:..  length:   :..:..  size: ...... kb", plScrWidth);
	_writenum(buf[2], 7, 0x0F, i-1+cdpFirstTrack, 10, 2);
	if (cdpViewSectors)
	{
		writenum(buf[2], 22, 0x0F, cdpTrackStarts[i-1]+150, 10, 8, 0);
		writenum(buf[2], 37, 0x0F, stat.position-cdpTrackStarts[i-1], 10, 8, 0);
		writenum(buf[2], 55, 0x0F, cdpTrackStarts[i]-cdpTrackStarts[i-1], 10, 8, 0);
	} else {
		writestring(buf[2], 22, 0x0F, gettimestr(cdpTrackStarts[i-1]+150, timestr), 8);
		writestring(buf[2], 37, 0x0F, gettimestr(stat.position-cdpTrackStarts[i-1], timestr), 8);
		writestring(buf[2], 55, 0x0F, gettimestr(cdpTrackStarts[i]-cdpTrackStarts[i-1], timestr), 8);
	}
	_writenum(buf[2], 71, 0x0F, (cdpTrackStarts[i]-cdpTrackStarts[i-1])*147/64, 10, 6);
}

static int cdaProcessKey(uint16_t key)
{
	int i;
	switch (key)
	{
		case KEY_ALT_K:
			cpiKeyHelp('p', "Start/stop pause with fade");
			cpiKeyHelp('P', "Start/stop pause with fade");
			cpiKeyHelp(KEY_CTRL_P, "Start/stop pause");
			cpiKeyHelp('t', "Toggle sector view mode");
			cpiKeyHelp(KEY_DOWN, "Jump back (small)");
			cpiKeyHelp(KEY_UP, "Jump forward (small)");
			cpiKeyHelp(KEY_LEFT, "Jump back");
			cpiKeyHelp(KEY_RIGHT, "Jump forward");
			cpiKeyHelp(KEY_HOME, "Jump to start of track");
			cpiKeyHelp('<', "Jump track back");
			cpiKeyHelp('>', "Jump track forward");
			cpiKeyHelp(KEY_F9, "Pitch speed down");
			cpiKeyHelp(KEY_F11, "Pitch speed down");
			cpiKeyHelp(KEY_F10, "Pitch speed up");
			cpiKeyHelp(KEY_F12, "Pitch speed up");
			if (plrProcessKey)
				plrProcessKey(key);
			return 0;
		case 'p': case 'P':
			startpausefade();
			break;
		case KEY_CTRL_P:
			plPause=!plPause;
			if (plPause)
				cdPause(fileno(cdpDrive));
			else
				cdRestart(fileno(cdpDrive));

			break;
		case 't':
			cdpViewSectors=!cdpViewSectors;
			break;
/*		case 0x2000:
			cdpPlayMode=1;
			setnewpos=0;
			basesec=cdpTrackStarts[0];
			length=cdpTrackStarts[cdpTrackNum]-basesec;
			//    strcpy(name, "DISK");
			break;*/
		case KEY_DOWN:
			newpos-=75;
			setnewpos=1;
			break;
		case KEY_UP:
		    	newpos+=75;
			setnewpos=1;
			break;
		case KEY_LEFT:
			newpos-=75*10;
			setnewpos=1;
			break;
		case KEY_RIGHT:
			newpos+=75*10;
			setnewpos=1;
			break;
		case KEY_HOME:
			if (!cdpPlayMode)
			{
				newpos=0;
				setnewpos=1;
				break;
			}
			for (i=1; i<=cdpTrackNum; i++)
				if (newpos<(cdpTrackStarts[i]-basesec))
					break;
			newpos=cdpTrackStarts[i-1]-basesec;
			setnewpos=1;
			break;
/* TODO-keys	case 0x8D00: //ctrl-up
			newpos-=60*75;
			setnewpos=1;
			break;
		case 0x9100: //ctrl-down
			newpos+=60*75;
			setnewpos=1;
			break;*/
		case '<' /* CTRL_LEFT TODO-keys */:
			if (!cdpPlayMode)
				break;
			for (i=2; i<=cdpTrackNum; i++)
				if (newpos<(cdpTrackStarts[i]-basesec))
					break;
			newpos=cdpTrackStarts[i-2]-basesec;
			setnewpos=1;
			break;
		case '>' /* CTRL_RIGH TODO-keys */:
			if (!cdpPlayMode)
				break;
			for (i=1; i<=cdpTrackNum; i++)
				if (newpos<(cdpTrackStarts[i]-basesec))
					break;
			newpos=cdpTrackStarts[i]-basesec;
			setnewpos=1;
			break;
/* TODO-keys	case 0x7700: //ctrl-home
			newpos=0;
			setnewpos=1;
			break;*/
		case KEY_F9:
		case KEY_F11:
			if ((speed-=finespeed)<16)
				speed=16;
			cdSetSpeed(speed);
			break;
		case KEY_F10:
		case KEY_F12:
			if ((speed+=finespeed)>2048)
				speed=2048;
			cdSetSpeed(speed);
			break;
		default:
			if (smpProcessKey)
			{
				int ret=smpProcessKey(key);
				if (ret==2)
					cpiResetScreen();
				if (ret)
					return 1;
			}
			if (plrProcessKey)
			{
				int ret=plrProcessKey(key);
				if (ret==2)
					cpiResetScreen();
				if (ret)
					return 1;
			}
			return 0;
	}
	return 1;
}

static int cdaLooped(void)
{
	struct cdStat stat;

	if (pausefadedirect)
		dopausefade();

	cdSetLoop(fsLoopMods);
	
	cdIdle();

	cdGetStatus(fileno(cdpDrive), &stat);

	/*
	if (status->error&STATUS_ERROR)
		return 1;
	*/

	if (stat.looped)
		return 1;
	
	if (setnewpos)
	{
		if (newpos<0)
			newpos=0;
		if (newpos>=length)
		{
			if (fsLoopMods)
				newpos=0;
			else
				return 1;
		}
		cdPause(fileno(cdpDrive));
		cdRestartAt(fileno(cdpDrive), basesec+newpos /*, length-newpos*/);
		if (plPause)
			cdPause(fileno(cdpDrive));
		setnewpos=0;
	} else
		newpos=stat.position-basesec;
	
	return 0;
}

static void cdaCloseFile(void)
{
	cdStop(fileno(cdpDrive));
}

static int cdaOpenFile(const char *path, struct moduleinfostruct *info, FILE *file)
{
	char name[NAME_MAX+1];
	char ext[NAME_MAX+1];
	unsigned char tnum;

	cdpDrive=file;

	_splitpath(path, 0, 0, name, ext);

	if (!strcmp(name, "DISK"))
		tnum=0xFF;
	else
		if (!memcmp(name, "TRACK", 5)&&isdigit(name[5])&&isdigit(name[6])&&(strlen(name)==7))
			tnum=(name[5]-'0')*10+(name[6]-'0');
		else
			return -1;

	if (!cdIsCDDrive(fileno(cdpDrive)))
		return -1;

	cdpTrackNum=cdGetTracks(fileno(cdpDrive), cdpTrackStarts, &cdpFirstTrack, CDROM_LEADOUT);

	if (tnum!=0xFF)
	{
		if ((tnum<cdpFirstTrack)||(tnum>(cdpFirstTrack+cdpTrackNum)))
			return -1;
		cdpPlayMode=0;
		basesec=cdpTrackStarts[tnum-cdpFirstTrack];
		length=cdpTrackStarts[tnum-cdpFirstTrack+1]-basesec;
	} else {
		if (!cdpTrackNum)
			return -1;
		cdpPlayMode=1;
		basesec=cdpTrackStarts[0];
		length=cdpTrackStarts[cdpTrackNum+1]-basesec;
	}

	newpos=0;
	setnewpos=1;
	plPause=0;

	plIsEnd=cdaLooped;
	plProcessKey=cdaProcessKey;
	plDrawGStrings=cdaDrawGStrings;

	/*  cdLockTray(cdpDrive, 1); */

	strncpy(vdev, info->comment, 8);
	vdev[7]=0;

	if (cdPlay(fileno(cdpDrive), basesec, length))
		return -1;

	normalize();

	return 0;
}

struct cpifaceplayerstruct cdaPlayer = {cdaOpenFile, cdaCloseFile};
char *dllinfo = "";
struct linkinfostruct dllextinfo = {"playcda", "OpenCP CDA Player (c) 1995-04 Niklas Beisert, Tammo Hinrichs, Stian Skjelstad", DLLVERSION, 0};
