(*****************************************************************************

  Liquidsoap, a programmable audio stream generator.
  Copyright 2003-2007 Savonet team

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details, fully stated in the COPYING
  file at the root of the liquidsoap distribution.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *****************************************************************************)

(** API to various lastfm protocols *)

(** Records for login and client *)
type client = { client : string ; version : string }
type login = { user : string ; password : string }

module Audioscrobbler :
sig

  (** Audioscrobbler is the submission protocol as described at
    
     http://www.audioscrobbler.net/development/protocol/ *)

  (** {2 Types} *)

  (** source of the track *)
  type source = User | Broadcast | Recommendation | Lastfm | Unknown

  (** possible rating for the track *)
  type rating = Love | Ban | Skip

  (** song submission type *)
  type song = { artist : string; track: string; time: float option; 
                source : source option; rating : rating option ;
                length : float option ; album : string option ; 
                trackauth : string option ; tracknumber : int option; 
                musicbrainzid : string option }

  (** various errors *)
  type error = Http of string | Banned | Badauth | Badtime
              | Failed of string | UnknownError of string | Success
              | Internal of string
  exception Error of error
  
  (** Get meaning of Error e *)
  val string_of_error : error -> string
  
  (** {2 Basic API} 
     
     Using this API, all requests are done in one single step *)
  
  (** [do_np client login song]
     Execute a nowplaying request 
     with authentification *)
  val do_np : client -> login -> song -> unit

  (** [do_submit client login song]
      Execute a nowplaying request 
      with authentification *)
  val do_submit : client -> login -> song list -> unit

  (** {2 Advanced API} 
     
     This API is for advanced usages.
     
     You may use it this way:
     {ol {- handshake : initiate session }
      {- np/submit: execute request } }

    The module will cache session informations and avoid redundant
    requests, so you might always call handshake. *)

  (** [handshake client login] 
     Open session, returns session ID *)
  val handshake : client -> login -> string

  (** [np sessionID track]
     Execute a nowplaying request *)
   val np : string -> song -> unit

  (** [submit sessionID tracks]
     Execute a submit request *)
   val submit : string -> song list -> unit

end

module Radio : 
sig

(** API for using lastfm radios 

    No documentation avaible for now... *) 

  (** {2 Types} *)

  (** Type for track datas 
   
    A track is a list of "field","value" metadatas and an uri *)
  type track = (string * string) list * string
  
  (** Various errors *)
  type error = Http of string | Init of string | Adjust of string*string | Playlist | Empty
  exception Error of error
  
  (** Get meaning of Error e *)
  val string_of_error : error -> string

  (** {2 Basic API} *)
  
  (** [get uri] performs whole process and
      outputs a list of metadatas,uri
      from given lastfm uri *)
  val get : string -> track list

  (** {2 Advanced API} 

     Using this API you shall call: 
    {ol {- parse: get required parts of the uri}
    {- init: initiate a session}
    {- adjust: adjust station } }
    Then you can use any of the following:
    {ul {- url : return the url for a xspf playlist }
    {- playlist: return the raw xml content of the playlist}
    {- tracks : returns the parsed playlist.}}
    After each of those calls, you shall use *only* one
    of the songs from the playlist.
  
    The module will cache session informations and avoid redundant
    requests, so you might always call init and adjust.

    If you call url or playlist, and anything went bad, you have to call [clear] to 
    remove cached data about this session. *)

  (** [parse uri] parse the given lastfm:// uri
    * returns login option,(station,options option) *)
  val parse : string -> (login option)*(string*string option)
  
  (** [init login] initiate lastfm session
    * Returns the session id *)
  val init : login option -> string
  
  (** [adjust id station] adjusts lastfm station 
    * for given session ID *)
  val adjust : string -> string -> unit

  (** [url id options] returns the url of the playlist *)
  val url : string -> string option -> string  
  
  (** [playlist id options] returns the raw xml content of the playlist *)
  val playlist : string -> string option -> string

  (** [tracks id options] 
    * returns a list of metadatas,uri  *)
  val tracks : string -> string option -> track list
  
  (** [clear id] closes and clear all 
    * informations about the given session ID *)
  val clear : string -> unit

end
