(*****************************************************************************

  Liquidsoap, a programmable audio stream generator.
  Copyright 2003-2007 Savonet team

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details, fully stated in the COPYING
  file at the root of the liquidsoap distribution.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *****************************************************************************)

(* lastfm protocol API for ocaml *)

exception Http of string

(* Records for client *)
type client = { client : string ; version : string }
type login = { user : string ; password : string }

let request ?(post="") ?(headers=[]) ?(port=80) ~host req =
let call = match post with
   | "" -> new Http_client.get_call
  	 | _ -> new Http_client.post_call
  in
  let pipeline = new Http_client.pipeline in
  let http_headers = call#request_header `Base in
  let body = call#request_body in
  call#set_request_uri (Printf.sprintf "http://%s:%d%s" host port req) ;
  let headers = ("User-agent",
      Printf.sprintf "ocaml-lastfm/%s" Constants.version) 
      :: headers
  in
  http_headers#set_fields headers ;
  begin
    match post with
  | "" -> ()
  | _ -> begin
  	         body#set_value post ; 
  		 call#set_request_body body ;
  		 http_headers#update_field 
  		   "Content-length" 
  		   (string_of_int (String.length post));
  	       end
  end ;
  call#set_request_header http_headers ;
  pipeline#add call ;
  pipeline#run () ;
  call#response_body#value

module Audioscrobbler=
  struct

    (* See http://www.audioscrobbler.net/development/protocol/
     * For protocol description *)

    (* Data types *)
    type source = User | Broadcast | Recommendation | Lastfm | Unknown
    type rating = Love | Ban | Skip

    (* song submission type *)
    type song = { artist : string; track: string; time: float option; 
                  source : source option; rating : rating option ;
                  length : float option ; album : string option ;
                  trackauth : string option ; tracknumber : int option; 
                  musicbrainzid : string option }

    type error = Http of string | Banned | Badauth | Badtime
                | Failed of string | UnknownError of string | Success
                | Internal of string
    exception Error of error
    
    let string_of_error e = 
      match e with
        | Http s -> Printf.sprintf "http connection failed: %s" s
        | Banned -> "banned client"
        | Badauth -> "wrong login/password"
        | Badtime -> "wrong timestamp, check your clock"
        | Failed s -> Printf.sprintf "failure: %s" s
        | UnknownError s -> Printf.sprintf "unknown error: %s" s
        | Internal s -> Printf.sprintf  "erreur interne: %s" s
        | Success -> "success!"

    let error_of_response s = 
      try
        let parse s = 
           let regexp = Pcre.regexp "FAILED\\s([^\\r\\n]*)" in
           let sub = Pcre.exec ~rex:regexp s in
           Failed (Pcre.get_substring sub 1)
        in
	let test (p,e) = 
	  try
	    ignore(Pcre.exec ~pat:p s) ;
	    raise (Error e)
	  with
	    | Not_found -> ()
	in
	let values = [("OK",Success);
	              ("BANNED",Banned);
		      ("BADAUTH",Badauth);
		      ("BADTIME",Badtime)]
        in
	try
	  List.iter test values ;
          parse s
	with 
	  | Error e -> e
      with
        | Not_found -> UnknownError "unrecognized response code"

    (* Protocol constants *)
    let version = "1.2"
    let base_port = 80
    let base_host = "post.audioscrobbler.com"
    let handshake_req = Printf.sprintf "/?hs=true&p=%s&c=%s&v=%s&u=%s&t=%s&a=%s" version
    (* sessions contains (user,pass) => id *)
    let sessions = Hashtbl.create 1
    (* urls contains id => (np_url,submit_url) *)
    let urls = Hashtbl.create 1
    let raise e = raise (Error e)
    (* Wrapper for main request *)
    let request = try request with e -> raise (Http (Printexc.to_string e))
    let arg_value_string x = 
      match x with 
        | Some e -> e
        | None -> ""
    let arg_value_int x = 
      match x with 
        | Some e -> string_of_int e
        | None -> ""
    let arg_value_float x =
      match x with
        | Some e -> Printf.sprintf "%.0f" e
        | None -> ""


    let clear sessionid =
        let keys = Hashtbl.fold (fun a b r -> if b = sessionid then a::r else r)
                      sessions []
        in
        ignore (List.map (fun x -> Hashtbl.remove sessions x) keys) ;
        Hashtbl.remove urls sessionid

    let handshake client login = 
      let client,version,user,pass = 
         client.client,client.version,
         login.user,login.password
      in
      try 
        Hashtbl.find sessions (user,pass)
      with
        | Not_found ->
           let timestamp = Printf.sprintf "%.0f" (Unix.time ()) in
           let pass_digest = Digest.string pass in
           let token = Digest.string((Digest.to_hex pass_digest) ^ timestamp) in
           let req = handshake_req client version user timestamp (Digest.to_hex token) in
           let ans = request ~host:base_host ~port:base_port req in
           let state,id,v = 
             try
               let lines = Pcre.split ~pat:"[\r\n]+" ans in
               match lines with
                 | state :: id :: [a; b] -> state,id,(a,b)
                 | _ -> raise (error_of_response ans)
             with
               | Not_found -> raise (error_of_response ans)
           in
           match error_of_response state with
             | Success -> Hashtbl.replace sessions (user,pass) id; 
                          Hashtbl.replace urls id v; id
             | e -> raise e

    let audioscrobbler_post id base_url values = 
      let url = Neturl.parse_url base_url in
      let host = Neturl.url_host url in
      let port = Neturl.url_port url in
      let req = String.concat "/" (Neturl.url_path url) in
      let args = List.map (fun (a,b) -> 
                             Printf.sprintf "%s=%s" 
                                a (Netencoding.Url.encode b) )
                            values
      in
      let post = String.concat "&" args in
      let headers = [("Content-type","application/x-www-form-urlencoded")] in
      let ans = request ~post:post ~headers:headers ~host:host ~port:port req in
      match error_of_response ans with
        | Success -> ()
        | e -> clear id; raise e

    let np id song = 
      let url,_ = 
        try 
          Hashtbl.find urls id
        with
          | Not_found -> raise (Internal (Printf.sprintf "No session data for session ID %s" id))
      in
      let values = [("s",id);("a",song.artist);("t",song.track);
                    ("album",arg_value_string song.album);
                    ("l",arg_value_float song.length);
                    ("n",arg_value_int song.tracknumber);
                    ("m",arg_value_string song.musicbrainzid)] 
      in
      audioscrobbler_post id url values

     let submit id songs = 
      let _,url = 
        try 
          Hashtbl.find urls id
        with
          | Not_found -> raise (Internal (Printf.sprintf "No session data for session ID %s" id))
      in
      let count = ref 0 in
      let args = ref [("s",id)] in
      let get_arg = fun s -> Printf.sprintf "%s[%d]" s !count in
      let add_arg song =
        let ar,tr,t,s,r,l,al,n,m,x =
           song.artist, song.track,song.time,
           song.source, song.rating,
           song.length, song.album,
           song.tracknumber, song.musicbrainzid,
           song.trackauth
        in
        let l = match l with
                  | None when s <> Some User -> ""
                  | Some s -> Printf.sprintf "%.0f" s
                  | None -> raise (Internal "length required when source is User")
        in
        let t = match t with
                  | Some t -> Printf.sprintf "%.0f" t
                  | None -> raise (Internal "time required when source is User")
        in
        let x = match x with
                  | Some x -> x
                  | None when s = Some Lastfm -> raise (Internal "trackauth required when source is Lastfm")
                  | None -> ""
        in
        let s =  match s with 
                   | Some User -> "P"
                   | Some Broadcast -> "P"
                   | Some Recommendation -> "E"
                   | Some Lastfm -> "L"
                   | Some Unknown -> "U"
                   | None -> raise (Internal "source field is required for submit")
        in
        let r =  match r with 
                   | Some Love -> "L"
                   | Some Ban when s = "L" -> "B"
                   | Some Skip when s = "L" -> "S"
                   | None -> ""
                   | _ -> raise (Internal "bad rating value (ban and skip are for lastfm sources only)")
        in
        args := [(get_arg "a",ar);(get_arg "t",tr);(get_arg "i",t);
                 (get_arg "o",s ^ x);(get_arg "r",r);(get_arg "l",l);
                 (get_arg "b",arg_value_string al);(get_arg "n",arg_value_int n);
                 (get_arg "m",arg_value_string m)] @ (!args)
     in
     List.iter add_arg songs ;
     audioscrobbler_post id url (!args)

    let do_np client login song = 
      let id = handshake client login in
      np id song

    let do_submit client login songs = 
      let id = handshake client login in
      submit id songs


  end

module Radio=
  struct

    (* Type for track datas 
     * A track is a list of "field","value" metadatas
     * and an uri *)
    type track = (string * string) list * string

    type error = Http of string | Init of string | Adjust of string*string | Playlist | Empty
    exception Error of error
    
    let string_of_error e = 
      match e with
        | Http s -> Printf.sprintf "http connection failed: %s" s
        | Init s -> Printf.sprintf "could not open session:\n%s" s
        | Adjust (s,s') -> Printf.sprintf "could not adjust station to %s:\n%s\nIs the URI valid ?" s s'
        | Playlist -> "error while parsinf the playlist"
        | Empty -> "no files available"
    
    let raise e = raise (Error e)

    (* Some constant for the protocol *)
    let host = "ws.audioscrobbler.com"
    let port = 80
    let sessions = Hashtbl.create 1
    let stations = Hashtbl.create 1
    let anon_req = "//1.0/webclient/xmlrpc.php"
    let anon_post = "<methodCall><methodName>getSession</methodName><params /></methodCall>"
    let anon_handshake = Printf.sprintf "/1.0/radio/webclient/handshake.php?sessionKey=%s&user=%s"
    let registered_handshake = Printf.sprintf "/radio/handshake.php?username=%s&passwordmd5=%s"
    let station_set = Printf.sprintf "/radio/adjust.php?session=%s&url=%s"
    let anon_split_rex = Pcre.regexp "<value><string>(.+)</string></value>\n<value><string>(.+)</string></value>"

    let playlist_req id opt = 
      let opt = 
        match opt with 
          | None -> ""
          | Some s -> Printf.sprintf "&%s" s
      in
      Printf.sprintf "/radio/xspf.php?sk=%s%s&desktop=1" id opt

    (* Wrapper for main request *)
    let request = try request ~port:port ~host:host with e -> raise (Http (Printexc.to_string e))

    
    (* Some parsing functions *)
    
    let handshake_rex = Pcre.regexp "session=([0-9a-z]+).*"
    let parse_handshake s = 
        try
          let sub = Pcre.exec ~rex:handshake_rex s in
          Pcre.get_substring sub 1
        with
          | Not_found -> raise (Init s)
    
    let adjust_pat = "response=OK"
    let check_adjust s =
        Pcre.pmatch ~pat:adjust_pat s

    let opt_split_rex = Pcre.regexp "^([^?]+)\\?(.+)$"
    let opt_parse s =
        try
          let sub = Pcre.exec ~rex:opt_split_rex s in
          Pcre.get_substring sub 1, Some (Pcre.get_substring sub 2)
        with
          | Not_found -> s,None
    
    let auth_split_rex = Pcre.regexp "^lastfm://([^:]+):([^@]+)@(.+)$"
    let parse uri = 
        try
          let sub = Pcre.exec ~rex:auth_split_rex uri in
          let user,password = Pcre.get_substring sub 1,
                              Pcre.get_substring sub 2
          in
          (Some { user = user ; password = password }),
                 opt_parse (Pcre.get_substring sub 3)
        with
          | Not_found -> None,opt_parse uri
    
    let anon_parse s = 
       try
         let content = Xml.parse_string s in
         let rec get_auth content = 
           match content with
             | Xml.Element("data",_,
                   [Xml.Element("value",_,[Xml.Element("string",_,[Xml.PCData x])]);
            	Xml.Element("value",_,[Xml.Element("string",_,[Xml.PCData y])])]) :: l
    	          -> x,y
             | Xml.Element(_,_,l) :: l' -> get_auth (l@l')
    	 | _ -> raise (Init s)
        in
        get_auth [content]
       with
         | _ -> raise (Init s)
    
    (* Core stuff.. *)
    
    let clear sessionid =
        let keys = Hashtbl.fold (fun a b r -> if b = sessionid then a::r else r)
                      sessions []
        in
        ignore (List.map (fun x -> Hashtbl.remove sessions x) keys) ;
        Hashtbl.remove stations sessionid
    
    let anon_session () = 
      let headers = [("Content-Type","text/xml")] in
      let ret = request ~post:anon_post ~headers:headers anon_req in
      anon_parse ret
    
    let init login = 
      match login with
        | None -> let user,id = anon_session () in
                  let ret = request 
                      (anon_handshake id (Netencoding.Url.encode user)) 
                  in
                  parse_handshake ret
       | Some login -> 
          let user,password = login.user,login.password in
          try
            Hashtbl.find sessions (user,password)
          with
            | Not_found -> 
                    let password = Digest.to_hex (Digest.string password) in
                    let ret = request (registered_handshake 
    	                      (Netencoding.Url.encode user) password)
                    in
                    let sessionid = parse_handshake ret in
                    Hashtbl.replace sessions (user,password) sessionid;
                    sessionid
      
    
    let adjust sessionid req = 
      try 
        assert (Hashtbl.find stations sessionid = req)
      with 
        | _ ->
        let http_req = station_set sessionid (Netencoding.Url.encode req)
        in
        let ret = request http_req in
        if check_adjust ret then
           Hashtbl.replace stations sessionid req
        else
           begin
             Hashtbl.remove stations sessionid; 
             clear sessionid;
             raise (Adjust (req,ret))
           end
    
    let tracks sessionid opt = 
      try
        let req = playlist_req sessionid opt in
        let playlist = request req in
        Xmlplaylist.tracks playlist
      with
        | Xmlplaylist.Error e -> clear sessionid; raise Playlist 
        | Error e -> clear sessionid; raise e
    
    let get uri = 
      let login,(station,options) = parse uri in
      let id = init login in
      adjust id station;
      tracks id options
    
    let url id opt = 
     Printf.sprintf "http://%s:%d%s" host port (playlist_req id opt)

    let playlist id opt = 
      request (playlist_req id opt)

end

