/*
 * Jamie Guinan, December 1998
 *
 * Fan daemon.  Turns the fan off, then waits for the temperature
 * to (almost) reach the high thermostat setting and turns it on, and
 * waits for it to reach the low thermostat setting and turns it off.
 *
 * This seems to work even when run as non-root.
 *
 * Note that you need at least therm.h from the kernel sources to
 * compile this.
 *
 * Copyright:  Free for redistribution, modifications welcome, as 
 *    long as credits maintained.
 *
 * Warranty: None whatsoever.
 *
 * 2005/11/20 Woody Suwalski <woody@netwinder.org>
 *  - added fan event logging via syslog
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include <syslog.h>
/* #include "/usr/src/linux/arch/arm/drivers/char/therm.h" */
#include <asm/therm.h>

#define THERMDEV "/dev/temperature"

#define SLEEPSECONDS 60*3


int temperature, last_temperature;
/* struct THERM_SETTING ts; */
struct therm ts;


int get_temperature()
{
  int therm_fd;
  therm_fd = open(THERMDEV, O_RDONLY);
  if (therm_fd < 0)
    {
      return -1;
    }
  ioctl(therm_fd, CMD_GET_TEMPERATURE, &temperature);
  close(therm_fd);
  return 0;
}

int get_thermostates()
{
  int therm_fd;
  therm_fd = open(THERMDEV, O_RDONLY);
  if (therm_fd < 0)
    {
      return -1;
    }
  ioctl(therm_fd, CMD_GET_THERMOSTATE2, &ts);
  close(therm_fd);
  return 0;
}

int set_fan(int cntrl)
{
  int therm_fd;
  therm_fd = open(THERMDEV, O_RDONLY);
  if (therm_fd < 0)
    {
      return -1;
    }
  ioctl(therm_fd, CMD_SET_FAN, &cntrl);
  close(therm_fd);
  return 0;
}

void quit()
{
  /* On signal or error, turn fan on before exit. */
  set_fan(1);
  exit(-1);
}


void sig_handle(int n)
{
  quit();
}

int main(int argc, char *argv[])
{
  
  void usage()
    {
      printf("Usage: %s\n", argv[0]);
    }

  if (argc != 1)
    {
      usage();
      return -1;
    }

  /* Background thyself. */
  switch (fork())
    {
    case 0:
      /* Child. */
      break;
    case -1:
      /* Error. */
      perror("fork");
      exit(-1);
      break;
    default:
      /* Parent. */
      exit(0);
    }

  /* Handle a few common signals. */
  signal(SIGINT, sig_handle);
  signal(SIGQUIT, sig_handle);
  signal(SIGHUP, sig_handle);
  signal(SIGABRT, sig_handle);
  signal(SIGILL, sig_handle);
  signal(SIGTERM, sig_handle);

  last_temperature = 0;
  openlog("fand", LOG_CONS | LOG_PID, LOG_DAEMON);

  /* No controlling terminal. */
  setsid();

  while (1)
    {
      if (get_temperature() != 0)
	{
	  quit();
	}
	
      if (get_thermostates() != 0)
	{
	  quit();
	}
	  
      /* printf("temperature=%d, ts.hi=%d ts.lo=%d\n", temperature, ts.hi, ts.lo); */
      
      if (last_temperature == 0)
	{
	  set_fan(0);
	}
      
      if (temperature != last_temperature)
	{
	  /* 
	   * Use one less than ts.hi so that it jumps into "hairdryer" mode
	   * instead of "screaming-fighter-jet-takeoff" mode.
	   */
	  if (temperature >= (ts.hi-1))
	    {
	      syslog(LOG_NOTICE, "Fan ON: %dC", ts.hi);
	      if (set_fan(1) != 0)
		{
		  quit();
		}
	    }
	  else if (temperature <= ts.lo)
	    {
	      syslog(LOG_NOTICE, "Fan OFF: %dC", ts.lo);
	      if (set_fan(0) != 0)
		{
		  quit();
		}
	    }
	  last_temperature = temperature;
	}

      sleep (SLEEPSECONDS);
    }

  return 0;
}

