/* $Id: nws_cmd.c,v 1.24 2005/06/10 20:52:22 graziano Exp $ */

#include "config_nws.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>
#if TIME_WITH_SYS_TIME
#	include <sys/time.h>
#	include <time.h>
#else
#	if HAVE_SYS_TIME_H
#		include <sys/time.h>
#	else
#		include <time.h>
#	endif
#endif
#include "math.h"

#include "diagnostic.h"
#include "messages.h"
#include "protocol.h"
#include "strutil.h"
#include "osutil.h"
#include "skills.h"
#include "clique_protocol.h"
#include "register.h"

#define NWSAPI_SHORTNAMES 1
#include "nws_api.h" 

#include "host_protocol.h"

typedef enum {DEAD, HEALTHY, SICK, UNRESPONSIVE} HostStatii;

/* local function that prints the status of a host (in cookie) */
static void
PrintStatus(	struct host_cookie *cookie,
		HostStatii stat,
		HostInfo *info,
		NsInfo *memory) {
	const char *TYPES[] = {"forecaster", "memory", "name_server", "sensor", "proxy"};
	const char *STATII[] = {"dead", "healthy", "sick", "unresponsive"};

	if (cookie == NULL) {
		return;
	}

	printf("%s", HostCImage(cookie));
	/* if we have the info, we don't need stat */
	if (info != NULL) {
		printf(" (%s)", info->registrationName);
		printf(" %s", TYPES[info->hostType]);
		printf(" registers with %s", info->nameServer);
		if (memory != NULL && memory->nameServer[0] != '\0') {
			printf(" uses %s as memory", memory->nameServer);
		}
	}
	printf(": %s\n", STATII[stat]);
}

static int
FillCookie(	struct host_cookie *cookie,
		const char *host,
		HostInfo *info,
		NsInfo *memInfo,
		double timeout) {
	/* sanity check */
	if (cookie == NULL || host == NULL || info == NULL) {
		ABORT("FillCookie: NULL parameter\n");
	}

	Host2Cookie(host, GetDefaultPort(SENSOR_HOST), cookie);
	if (!ConnectToHost(cookie, NULL)) {
		PrintStatus(cookie, DEAD, NULL, NULL);
		return 0;
	}
	if (!GetHostInfo(cookie, info, timeout)) {
		PrintStatus(cookie, UNRESPONSIVE, NULL, NULL);
		return 0;
	}

	/* we are now trying to get the memory the sensor is using: older
	 * sensor will lead to an error message */
	if (memInfo) {
		memInfo->nameServer[0] = '\0';
		if (!HostMemoryInfo(cookie, memInfo , timeout)) {
			INFO("Perhaps talking to old sensor?\n");
		}
	}

	return 1;
}

static void
BurpXML(	FILE *f,
		const char *src,
		const char *dst,
		const char *res,
		const char *opts,
		ForecastCollection *col) {

	/* sanity check */
	if (src == NULL || res == NULL || col == NULL) {
		ERROR("BurpXML: NULL parameters!\n");
		return;
	}

	/* let's print the seriesForecast stuff */
	fprintf(f, "   <seriesForecast>\n");
	fprintf(f, "\t<source>%s</source>\n", src);
	if (dst != NULL && dst[0] != '\0') {
		fprintf(f, "\t<destination>%s</destination>\n", dst);
	}
	fprintf(f, "\t<resource>%s</resource>\n", res);
	if (opts != NULL) {
		fprintf(f, "\t<options>%s</options>\n", opts);
	}

	/* if timestamp == 0 we don't have resonable values */
	if (col->measurement.timeStamp == 0) {
		fprintf(f, "   </seriesForecast>\n");
		return;
	}

	/* now let's print the forecastCollection */
	fprintf(f, "\t<forecastCollection>\n");
	fprintf(f, "\t\t<measurement>\n");
	fprintf(f, "\t\t\t<timestamp>%.0f</timestamp>\n", col->measurement.timeStamp);
	fprintf(f, "\t\t\t<value>%.9f</value>\n", col->measurement.measurement);
	fprintf(f, "\t\t</measurement>\n");
	fprintf(f, "\t\t<forecast minimize=\"absolute\">\n");
	fprintf(f, "\t\t\t<value>%.9f</value>\n", col->forecasts[MAE_FORECAST].forecast);
	fprintf(f, "\t\t\t<error>%.9f</error>\n", col->forecasts[MAE_FORECAST].error);
	fprintf(f, "\t\t\t<method>%s</method>\n", MethodName(col->forecasts[MAE_FORECAST].methodUsed));
	fprintf(f, "\t\t</forecast>\n");
	fprintf(f, "\t\t<forecast minimize=\"square\">\n");
	fprintf(f, "\t\t\t<value>%.9f</value>\n", col->forecasts[MSE_FORECAST].forecast);
	fprintf(f, "\t\t\t<error>%.9f</error>\n", col->forecasts[MSE_FORECAST].error);
	fprintf(f, "\t\t\t<method>%s</method>\n", MethodName(col->forecasts[MSE_FORECAST].methodUsed));
	fprintf(f, "\t\t</forecast>\n");
	fprintf(f, "\t</forecastCollection>\n");
	fprintf(f, "   </seriesForecast>\n");
}

static void
PrintForecast(	FILE *f,
		const char *src,
		const char *dst,
		const char *res,
		const char *opts,
		ForecastCollection *col,
		int title) {
	/* let's see if we need to just write the title */
	if (title) {
		fprintf(f, "%-15s ", "Resource");
		fprintf(f, "%-15s ", "Source");
		if (dst != NULL) {
			fprintf(f, "%-15s ", "Dest");
		}
		fprintf(f, "%-12s ", "Time");
		fprintf(f, "%-12s ", "Measurement");
		fprintf(f, "%-14s ", "Forecast (MAE)");
		fprintf(f, "%-11s ", "Error (MAE)");
		fprintf(f, "%-30s ", "Methor (MAE)");
		fprintf(f, "%-14s ", "Forecast (MSE)");
		fprintf(f, "%-11s ", "Error (MSE)");
		fprintf(f, "%-30s\n", "Method (MSE)");
		return;
	}

	/* print the enciladas */
	fprintf(f, "%-15s ", res);
	fprintf(f, "%-15s ", src);
	if (dst != NULL) {
		fprintf(f, "%-15s ", dst);
	}
	if (opts != NULL) {
		fprintf(f, "%s", opts);
	}

	/* if timestamp == 0 we don't have resonable values */
	if (col->measurement.timeStamp == 0) {
		fprintf(f, "N/A\n");
		return;
	}

	/* now let's print the forecastCollection */
	fprintf(f, "%-12.0f ", col->measurement.timeStamp);
	fprintf(f, "%10.4f ", col->measurement.measurement);
	fprintf(f, "%12.4f ", col->forecasts[MAE_FORECAST].forecast);
	fprintf(f, "%9.4f ", col->forecasts[MAE_FORECAST].MAE_error);
	fprintf(f, "%-30s ", MethodName(col->forecasts[MAE_FORECAST].methodUsed));
	fprintf(f, "%12.4f ", col->forecasts[MSE_FORECAST].forecast);
	fprintf(f, "%9.4f ", col->forecasts[MSE_FORECAST].error);
	fprintf(f, "%-30s\n", MethodName(col->forecasts[MSE_FORECAST].methodUsed));
}

/* work around our broken DNS */
static void
DamnDNS(	char *src,
		char *dst,
		const char *resource,
		char *clique,
		ForecastCollection *f) {
	ObjectSet series;
	char filter[256],
	     *name, 
	     s[MAX_HOST_NAME], 
	     d[MAX_HOST_NAME];
	int state;

	/* first of all let's initialize the forecast */
	f->measurement.timeStamp = 0;
	strcpy(s, src);
	strcpy(d, dst);

	for (state = 0; state < 3; state++) {
		sprintf(filter, "&(host=%s)(target=%s)(activity=%s)(resource=%s)", s, d, clique, resource);
		if (GetObjects(filter, &series)) {
			name = NwsAttributeValue_r(FindNwsAttribute(series, "name"));
			if (name) {
				GetForecast(name, f);
				FREE(name);
				state = 5;
			}
		}
		FreeObjectSet(&series);

		/* now, thank to our screwed up DNS, we need to add .*
		 * all over the freaking place, but we need to do it one
		 * host at a time to avoid failure in case of good DNS! */
		name = strchr(s, ':');
		if (!name) {
			/* there is no ':' so we just add * */
			strcat(s, ".*");
		} else {
			/* we need to add ".*" before the ':' */
			memmove(name + 2, name, strlen(name) + 1);
			*name = '.';
			*(name + 1) = '*';
		}
		name = strchr(d, ':');
		if (!name) {
			/* there is no ':' so we just add * */
			strcat(d, ".*");
		} else {
			/* we need to add ".*" before the ':' */
			memmove(name + 2, name, strlen(name) + 1);
			*name = '.';
			*(name + 1) = '*';
		}

		switch (state) {
		case 0:
			/* we do only the src here */
			strcpy(d, dst);
			break;
		case 1:
			/* we do only the dst here */
			strcpy(s , src);
			break;
		case 2:
			/* we do both */
			break;
		}
	}
}

/* very simple very stupid function to print the child-of in the
 * dependency tree between cliques */
static void
PrintChild(	int *tree,
		char **cliques,
		int howMany,
		int father,
		int *supers) {
	int i, p;

	/* let's print the top-level */
	for (p = 0, i = 0; i < howMany; i++) {
		if (tree[i + father*howMany] == 1) {
			if (p == 0) {
				p = 1;
				printf("children of %s:", cliques[father]);
			}
			printf(" %s", cliques[i]);
		}
	}
	if (p) {
		printf(".\n");
	}

	for (i = 0; p && i < howMany; i++) {
		for (p = 0; p < howMany; p++) {
			if (supers[p] == -1 || supers[p] == i) {
				break;
			}
		}
		if (supers[p] != -1) {
			continue;
		}
		if (tree[i + father*howMany] == 1) {
			PrintChild(tree, cliques, howMany, i, supers);
		}
	}
}

#define SWITCHES "s:P:t:rVv:f:XN:SGT:"

static void
usage() {
	printf("\nUsage: nws_cmd [OPTIONS] command host [command-options]\n");
	printf("Control the hosts for the Network Weather Service\n");
	printf("\nOPTIONS can be:\n");
	printf("\t-t timeout          specify a timeout\n");
	printf("\t-X                  print result from proxy cmd in XML\n");
	printf("\t-N nameserver       the nameserver to use\n");
	printf("\t-P proxy            the proxy to use\n");
	printf("\t-f file             read host and command options from #file#\n");
	printf("\t-s size,buffer,msg  instruct ping to use these values\n");
	printf("\t-r                  relax host name matching rules\n");
	printf("\t-T time             discard data older then #time# seconds\n");
	printf("\t-v level            verbose level (up to 5)\n");
	printf("\t-V                  print version\n");
	printf("\nand commands can be:\n");
	printf("\ttest                query #host#: accepts multiple hosts as options\n");
	printf("\tskill               ask #host# to run a remote skill (just once)\n");
	printf("\tping                probe band & lat to the #host#s\n");
	printf("\ttime                translate seconds from epoch to friendlier time\n");
	printf("\tproxy               the first options is the resource, the reminder are the hosts\n");
	printf("\tproxyClique         the first options is the resource, then the clique name\n");
	printf("\tpath                the first 2 options are the clique or hosts, then the level\n");
	printf("Report bugs to <nws@nws.cs.ucsb.edu>.\n\n");
}

int
main(		int argc,
		char *argv[]) {
	char tmp[255], *commandLine, *ptr, *command;
	struct host_cookie cookie;
	HostSpec NS;
	int c, sizes[3], z, inXML, superClique, gnuplot;
	double tout, valid;
	unsigned short verbose;
	extern char *optarg;
	extern int optind;
	HostInfo info;
	NsInfo memInfo;
	FILE *optionsFile;

	/* deafult values */
	commandLine = strdup("");;
	optionsFile = NULL;
	tout = -1;
	valid = 0;
	verbose = 2;
	superClique = gnuplot = 0;
	sizes[0] = 256;		/* experiment size */
	sizes[1] = 32;		/* buffer size */
	sizes[2] = 16;		/* message size */
	inXML = 0;
	NS.machineName[0] = '\0';

	while((c = getopt(argc, argv, SWITCHES)) != EOF) {
		switch(c) {
		case 'r':
			RelaxHostNameCheck(1);
			break;

		case 'G':
			gnuplot = 1;
			break;

		case 'S':
			superClique = 1;
			break;

		case 'X':
			inXML = 1;
			break;

		case 'T':
			valid = strtod(optarg, NULL);
			break;

		case 'P':
			Host2Cookie(optarg, DefaultHostPort(PROXY_HOST), &cookie);
			NS = *MakeHostSpec(cookie.name, cookie.port);
			if (!UseProxy(&NS)) {
				WARN("Failed to set the proxy\n");
			}
			break;

		case 'N':
			Host2Cookie(optarg, DefaultHostPort(NAME_SERVER_HOST), &cookie);
			NS = *MakeHostSpec(cookie.name, cookie.port);
			if (!UseNameServer(&NS)) {
				WARN("Failed to set nameserver\n");
			}
			break;

		case 't':
			tout = strtod(optarg, NULL);
			break;

		case 'f':
			optionsFile = fopen(optarg, "r");
			if (optionsFile == NULL) {
				printf("Cannot open file %s\n", optarg);
				exit(1);
			}
			break;

		case 's':
			/* Parse the comma-delimited size list, allowing
			 * each element to default if unspecified (e.g.
			 * "32", "10,,5" and ",,8" are all legal). */
			command = optarg;
			for (z = 0; z < 3 && *command != '\0'; z++) {
				if (*command != ',') {
					if (GETTOK(tmp, command, ",", (const char **)(&command))) {
						sizes[z] = (int)strtol(tmp, NULL, 10);
					}
				}
				if (*command != '\0') {
					command++;
				}
			}
			break;

		case 'v':
			verbose = (unsigned short)atol(optarg);
			break;

		case 'V':
			printf("nws_cmd for NWS version %s", VERSION);
#ifdef HAVE_PTHREAD_H
			printf(", with thread support");
#endif
#ifdef WITH_DEBUG
			printf(", with debug support");
#endif
			printf("\n\n");
			exit(0);
			break;

		case '?':
			if (optopt == 'v') {
				/* using basic level */
				verbose = 2;
				break;
			}
			/* non recognized options: printing help */

		default:
			usage();
			exit(1);
			break;
		}
	}

	/* set the verbose level */
	SetDiagnosticLevel(verbose, stderr, stderr);

	/* now the next word should be the command */
	if (optind >= argc) {
		usage();
		exit(1);
	} 

	/* this is the command */
	command = strdup(argv[optind++]);
	if (command == NULL) {
		ABORT("out of memory\n");
	}

	/* now let's collect all the host/comamnds from command line/file */
	for (; optind < argc; optind++) {
		commandLine = REALLOC(commandLine, strlen(commandLine) + strlen(argv[optind]) + 2, 1);
		strcat(commandLine, argv[optind]);
		strcat(commandLine, "\n");
	}
	if (optionsFile != NULL) {
		/* read the file now */
		while (fgets(tmp, sizeof(tmp), optionsFile) != NULL) {
			/* remove empty lines */
			if (strlen(tmp) <= 1) {
				continue;
			}
			commandLine = REALLOC(commandLine, strlen(commandLine) + strlen(tmp) + 2, 1);
			strcat(commandLine, tmp);
			strcat(commandLine, "\n");
		}
		fclose(optionsFile);
	}
	if (commandLine[0] == '\0' && strcmp(command, "tree")) {
		ABORT("There is no host/options specified\n");
		exit(1);
	}
	ptr = commandLine;

	if (strcmp(command, "test") == 0) {
		/* run a test command */
		for (; GETWORD(tmp, ptr, (const char **)&ptr); ) {
			/* try to connect to the host */
			if (FillCookie(&cookie, tmp, &info, &memInfo, tout)) {
				if (info.healthy) {
					PrintStatus(&cookie, HEALTHY, &info, &memInfo);
				} else {
					PrintStatus(&cookie, SICK, &info, &memInfo);
				}
				DisconnectHost(&cookie);
			}
		}
	} else if (strcmp(command, "testSeries") == 0) {
		Object seriesobject;
		ObjectSet seriesset, activityset;
		int optcount, filterLen;
		char *filter, *value, *name;

		for (optcount = 0; GETWORD(tmp, ptr, (const char **)&ptr); ) {
			optcount++;
			filterLen = 40 + strlen(tmp);
			filter = malloc(sizeof(char) * filterLen);
			sprintf(filter, "(&(objectclass=nwsSeries)(name=%s*))", tmp);
			/* let's get the series */
			if(!GetObjectsTimeout(filter, &seriesset, tout)) {
				ABORT("error getting objects\n");
			}
			FREE(filter);

			/* nothing back: we don't know about this series */
			if (strlen(seriesset) <= 1) {
				FreeObjectSet(&seriesset);
				printf("%s: unknown\n", tmp);
				continue;
			} 

			/* let's gp through all the series we found */
			ForEachObject(seriesset, seriesobject) {
				name = NwsAttributeValue_r(FindNwsAttribute(seriesobject, "name"));
				value = NwsAttributeValue_r(FindNwsAttribute(seriesobject, "activity"));
				if (value == NULL) {
					printf("%s: stale\n", name);
					FREE(name);
					continue;
				}

				filterLen = strlen(value) + 44;
				filter = malloc(sizeof(char) * filterLen);
				sprintf(filter, "(&(objectclass=nwsActivity)(name=%s))", value);
		 
				if (!GetObjectsTimeout(filter, &activityset, tout)) {
					ABORT("error getting objects\n");
				}
				FREE(filter);

				if (strlen(activityset) > 1) {
					printf("%s: current\n", name);
				} else {
					printf("%s: orphan\n", name);
				}
				FREE(name);
				FREE(value);
				FreeObjectSet(&activityset);
			}
		}
		if (!optcount) {
			ABORT("Need to specify seriesname as option\n");
		}
	} else if (strcmp(command, "skill") == 0) {
		char *opts;
		char **objs;
		int len;
		double d[2];

		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need a host\n");
			exit(1);
		}
		if (!FillCookie(&cookie, tmp, &info, NULL, tout)) {
			exit(2);
		}
		/* turns whatever into options */
		opts = NULL;
		len = 0;
		for (; GETWORD(tmp, ptr, (const char **)&ptr); ) {
			opts = (char*)REALLOC(opts, len + strlen(tmp) + 2, 1);
			if (len != 0) {
				strcat(opts, "\t");
				len++;
			} else {
				opts[0] = '\0';
			}
			strcat(opts, tmp);
			len += strlen(tmp);
		}
		if (len == 0) {
			printf("I need at least a skill\n");
			exit(1);
		}

		/* simple check: skillName gotta be here */
		if (strstr(opts, "skillName") == NULL) {
			printf("no attribute \"skillName\" found!\n");
			exit(2);
		}

		if (!RemoteUseSkill(&cookie, opts, tout, &objs, d, &len)) {
			printf("failed to use skill on %s\n", HostCImage(&cookie));
			exit(2);
		}
		for (len--; len >= 0; len--) {
			printf("%s %s: %.2f\n", HostCImage(&cookie), objs[len], d[len]);
			free(objs[len]);
		}
		free(opts);
		DisconnectHost(&cookie);
	} else if (strcmp(command, "ping") == 0) {
		/* let's try to ping an host */
		SkillResult *res = NULL;
		int len, j;
		char opts[255];

		for (; GETWORD(tmp, ptr, (const char **)&ptr); ) {
			/* let's fill in the options: target ... */
			sprintf(opts, "target:%s", tmp);
			/* size, buffer and message sizes in kB */
			strcat(opts, "\tsize:");
			sprintf(opts + strlen(opts), "%d", sizes[0]);
			strcat(opts, "\tbuffer:");
			sprintf(opts + strlen(opts), "%d", sizes[1]);
			strcat(opts, "\tmessage:");
			sprintf(opts + strlen(opts), "%d", sizes[2]);

			/* let's print the sizes and the target */
			printf("(%dk,%dk,%dk) ", sizes[0], sizes[1], sizes[2]);
			printf("to %s: ", tmp);

			UseSkill(tcpMessageMonitor, opts, tout, &res, &len);
			for(j = 0; j < len; j++) {
				printf("%s: ", ResourceName(res[j].resource));
				if (!res[j].succeeded) {
			              printf("failed ");
				      continue;
				}
				printf("%.4f %s ", res[j].measurement, ResourceLabel(res[j].resource));
			} 
			printf("\n");
			FreeSkillResults(len, &res);
		}
	} else if (strcmp(command, "time") == 0) {
		time_t t;
		char s[256];

		for (; GETWORD(tmp, ptr, (const char **)&ptr); ) {
			t = (int)strtol(tmp, (char **)NULL, 10);
			strftime(s, 256, "%d-%m-%Y/%H:%M:%S", localtime(&t));
			printf("%s\n", s);
		}
		fflush(stdout);
	} else if (!strcmp(command, "proxy") || !strcmp(command, "proxyClique")) {
		char *hosts, *res, host1[MAX_HOST_NAME], host2[MAX_HOST_NAME];
		const char *ptr1, *ptr2;
		int howMany, j, interMachine, len;
		ForecastCollection *col;
		int mode;

		/* then is the resource */
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			ABORT("I need a resource\n");
		}
		res = strdup(tmp);
		if (res == NULL) {
			ABORT("out of memory\n");
		}
		if (IsResourceValid(res) == -1) {
			ABORT1("Unknown resource (%s)\n", res);
		}

		/* let's see if we have source/dest machines */
		interMachine = IntermachineResource(res);

		/* turns the targets into a comma-separated list */
		hosts = NULL;
		len = 0;
		for (c = 0; GETWORD(tmp, ptr, (const char **)&ptr); c++) {
			hosts = (char *)realloc(hosts, len + strlen(tmp) + 2);
			if (hosts == NULL) {
				ABORT("out of memory\n");
			}
			if (len > 0) {
				hosts[len] = ',';
				len++;
			}
			memcpy(hosts + len, tmp, strlen(tmp) + 1);
			len += strlen(tmp);
			c++;
		}
		if (c == 0) {
			ObjectSet objs;
			Object obj;

			INFO("You didn't specify a target: I'll look for it\n");
			if (!strcmp(command, "proxyClique")) {
				/* look for cliques */
				if (!GetObjects("(&(objectclass=nwsActivity)(controlName=clique))", &objs)) {
					ABORT("Cannot retrieve data from the nameserver\n");
				}
			} else {
				/* look for hosts */
				if (!GetObjects("(objectclass=nwsHost)", &objs)) {
					ABORT("Cannot retrieve data from the nameserver\n");
				}
			}
			ForEachObject(objs, obj) {
				char *name;

				name = NwsAttributeValue_r(FindNwsAttribute(obj, "name"));
				if (name == NULL) {
					continue;
				}
				hosts = (char *)realloc(hosts, len + strlen(name) + 2);
				if (hosts == NULL) {
					ABORT("out of memory\n");
				}
				if (len > 0) {
					hosts[len] = ',';
					len++;
				}
				memcpy(hosts + len, name, strlen(name) + 1);
				len += strlen(name);
				c++;
				FREE(name);
			}
			FreeObjectSet(&objs);
			if (c == 0) {
				ABORT("Cannot find targets!\n");
			}
		}

		/* now we do different things if we need to retrieve data
		 * or just tells the proxy to start fetching */
		howMany = 0;

		/* set the default mode for the virtualization */
		mode = MODE_MIN;
		if (strcmp(res, NWSAPI_DEFAULT_LATENCY_RESOURCE) == 0) {
			mode = MODE_MAX;
		}

		/* let's call the proxy */
		if (!strcmp(command, "proxy")) {
			if (superClique) {
				howMany = GetVirtualForecasts(res, NULL, hosts, mode, 10, valid, &col, tout);
			} else {
				howMany = GetAllForecasts(res, NULL, hosts, valid, &col, tout);
			}
		} else {
			char *tmpHosts;

			if (superClique) {

				howMany = GetCliquesForecasts(res, hosts, &tmpHosts, mode, 10, valid, &col, tout);
			} else {
				howMany = GetCliqueForecasts(res, hosts, &tmpHosts, valid, &col, tout);
			}
			if (howMany > 0) {
				FREE(hosts);
				hosts = tmpHosts;
			}
		}
		if (howMany == 0) {
			ABORT("Error in getting the forecast!\n");
		}

		/* if we have intermachine experiment we have
		 * sqrt() of the machines */
		if (interMachine) {
			howMany = sqrt(howMany);
		}

		/* damn XML ... */
		if (inXML) {
			fprintf(stdout, "<seriesCollection>\n");
		} else if (!gnuplot) {
			/* let's print title */
			PrintForecast(stdout, 
					NULL, 
					(interMachine) ? "yep" : NULL, 
					NULL,
					NULL,
					NULL,
					1);
		}

		/* now let's go through all of the forecasts */
		for (c=0, ptr1=hosts; GETTOK(host1, ptr1, ",", &ptr1); c++) {
			/* only interMachine resources (band/lat)
			 * returns matrices */
			if (!interMachine && c > 0) {
				break;
			}
			for (j = 0, ptr2 = hosts; GETTOK(host2, ptr2, ",", &ptr2); j++) {
				if (!interMachine) {
					strcpy(host1, host2);
					host2[0] = '\0';
				} else if (c == j && !gnuplot) {
					/* skip the same hosts on a 2
					 * host resource */
					continue;
				}

				if (inXML) {
					BurpXML(stdout, 
						host1,
						host2,
						res,
						NULL,
						&col[j + c*howMany]);
				} else if (gnuplot) {
					double value;

					value = 0;
					if (col[j + c*howMany].measurement.timeStamp) {
						value = col[j + c*howMany].forecasts[0].forecast;
					}

					printf("%d %d %.6f (%s -> %s)\n", c, j, value, host1, host2);
					printf("%d %d %.6f (%s -> %s)\n\n", c, j+1, value, host1, host2);
					printf("%d %d %.6f (%s -> %s)\n", c+1, j, value, host1, host2);
					printf("%d %d %.6f (%s -> %s)\n\n", c+1, j+1, value, host1, host2);
				} else {
					PrintForecast(stdout, 
						host1,
						host2,
						res,
						NULL,
						&col[j + c*howMany],
						0);
				}
			}
		}
		if (inXML) {
			fprintf(stdout, "</seriesCollection>\n");
		} else {
			fprintf(stdout, "\n");
		}
		FREE(col);
		FREE(res);
		FREE(hosts);
	} else if (strcmp(command, "path") == 0) {
		ObjectSet path, objs;
		Object obj;
		char *src, *dst;
		int level, ret;

		src = dst = NULL;

		/* first argument is the first host */
		if (GETWORD(tmp, ptr, (const char **)&ptr)) {
			src = strdup(tmp);
			if (GETWORD(tmp, ptr, (const char **)&ptr)) {
				dst = strdup(tmp);
			}
		}
		if (src == NULL || dst == NULL) {
			ABORT("I need 2 hosts/cliques\n");
		}
		if (GETWORD(tmp, ptr, (const char **)&ptr)) {
			level = (int)strtol(tmp, NULL, 10);
		} else {
			level = 2;
		}

		 /* let's get all the cliques */
		objs = NULL;
		if (!GetObjects(NWSAPI_CLIQUES, &objs)) {
			ABORT("no clique found!\n");
		}

		/* if we have a :port we are dealing with hosts,
		 * otherwise with cliques */
		ret = 0;
		if (strchr(src, ':') == NULL) {
			/* we are dealing with cliques */
			ret = FindCliquesPath(src, dst, objs, &path, level);
		} else { 
			ret = FindHostsPath(src, dst, objs, &path, level);
		}
		FREE(src);
		FREE(dst);

		if (!ret) {
			printf("Didn't find anything at level %d: did you specify the port for the hosts\n", level);
		} else {
			ForEachObject(path, obj) {
				char host[MAX_HOST_NAME], 
				     target[MAX_HOST_NAME];
				const char *s, *clique;
				ForecastCollection b, l;

				/* we want to print host, target, clique and
				 * the measurement of bw and latency.
				 * First of all let's get clique and host
				 * lists*/
				src = NwsAttributeValue_r(FindNwsAttribute(obj, "hosts"));
				dst = NwsAttributeValue_r(FindNwsAttribute(obj, "cliques"));

				host[0] = '\0';
				s = src;
				clique = dst;
				while(GETTOK(tmp, s, ",", &s)) {
					/* it's the first time around ? */
					if (host[0] == '\0') {
						strcpy(host, tmp);
						continue;
					}
					strcpy(target, tmp);

					if (!GETTOK(tmp, clique, ",", &clique)){
						ABORT("not enough cliques!\n");
					}

					/* not let's get the
					 * measurements */
					b.measurement.timeStamp = 0;
					DamnDNS(host, target, "bandwidthTcp", tmp, &b);
					l.measurement.timeStamp = 0;
					DamnDNS(host, target, "latencyTcp", tmp, &l);
					/* all right let's print the data
					 * we found */
					printf("%s->%s in clique %s", host, target, tmp);
					if (b.measurement.timeStamp) {
						printf(" bw: %.2f", b.forecasts[NWSAPI_MAE_FORECAST].forecast);
					}
					if (l.measurement.timeStamp) {
						printf(" lat: %.2f", l.forecasts[NWSAPI_MAE_FORECAST].forecast);
					}
					printf("\n");

					/* destination gets to be source
					 * next time around */
					strcpy(host, target);
				}
				FREE(src);
				FREE(dst);
			}
			FreeObjectSet(&path);
		}
		FreeObjectSet(&objs);
	} else if (strcmp(command, "tree") == 0) {
		char **cliques;
		int *tree, i, j, *supers, s;

		c = NWSAPI_FindCliquesTree(&cliques, &tree);
		supers = MALLOC(sizeof(int) * c, 1);
		s = 0;
		supers[s] = -1;

		/* let's print the top-level */
		for (i = 0; i < c; i++) {
			for (j = 0; j < c; j++) {
				/* we are top-level if we are not
				 * 'child-of' of anybody else */
				if (tree[i + j*c] == 1 && tree[j + i*c] == 0) {
					break;
				}
			}
			if (j >= c) {
				printf("%s is top level\n", cliques[i]);
				supers[s] = i;
				supers[++s] = -1;
				PrintChild(tree, cliques, c, i, supers);
			}
		}
		while (--i >= 0) {
			FREE(cliques[i]);
		}
		FREE(cliques);
		FREE(tree);
	} else {
		/* unknown command */
		usage();
		exit(1);
	}
	free(command);
	free(commandLine);

	return(0);
}
