/* ntfs-conf - tool to enable/disable write support for NTFS.
 *
 * Copyright (C) 2007 Mertens Florent <flomertens@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
 
#include "mounter.h"
#include "prober.h"
#include "main.h"

#include <config.h>
#include <glade/glade.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <glib/gprintf.h>
#include <string.h>
#include <stdio.h>
#include <mntent.h>
#include <unistd.h>


const char      *hal_conf_dir = "/etc/hal/fdi/policy/";
const char      *write_policy = "/etc/hal/fdi/policy/20-ntfs-config-write-policy.fdi";
const char      *ro_policy = "/etc/hal/fdi/policy/20-ntfs-config-ro-policy.fdi";
const char      *fstab_sav = "/etc/fstab.pre-ntfs-config";

const gchar     fstab_header[] = "# /etc/fstab: static file system information.\n"
    "#\n"
    "#  -- This file has been automaticly generated by ntfs-config -- \n"
    "#\n"
    "# <file system> <mount point>   <type>  <options>       <dump>  <pass>\n"
    "\n";


typedef struct
{
    GtkWidget           *dialog;
    GtkToggleButton     *internal;
    GtkToggleButton     *external;
    GtkButton           *aboutbutton;
    GtkButton           *cancelbutton;
    GtkButton           *applybutton;
} App;

typedef struct
{
    gboolean        internal;
    gboolean        external;
} Settings;

static void 
show_about(gpointer data)
{
    gchar*  authors[] = {"Mertens Florent <flomertens@gmail.com>", NULL};

    gchar*  license = 
        "This program is free software; you can redistribute it and/or\n"
        "modify it under the terms of the GNU General Public License\n"
        "as published by the Free Software Foundation; either\n"
        "version 2 of the License, or (at your option) any later version.\n"
        "\n"
        "This program is distributed in the hope that it will be useful,\n"
        "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
        "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n"
        "See the GNU General Public License for more details.\n"
        "\n"
        "You should have received a copy of the GNU General Public License\n"
        "along with this library; if not, write to the Free Software\n"
        "Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA";

    gtk_show_about_dialog (NULL, 
        "authors", authors,
        "copyright", "Copyright © 2007 Mertens Florent",
        "translator-credits", _("translator-credits"),
        "name", "ntfs-config",
        "license", license,
        "logo-icon-name", "gnome-dev-harddisk",
        "comments", _("Enable/disable NTFS write support with one click.\n\
This tool use the ntfs-3g driver : http://www.ntfs-3g.org"),
        "website", "http://givre.cabspace.com/ntfs-config",
        "website-label", "ntfs-config homepage",
        "version", VERSION,
        NULL);
}

void
show_error (const GError *err)
{
    if ( !err )
        return;
    g_warning ("%s\n",err->message);

    GtkWidget *dialog = gtk_message_dialog_new (
        NULL,
        GTK_DIALOG_DESTROY_WITH_PARENT,
        GTK_MESSAGE_WARNING,
        GTK_BUTTONS_OK, err->message);
    
    gtk_window_set_title (GTK_WINDOW (dialog), "ntfs-config");

	gtk_dialog_run (GTK_DIALOG (dialog));
    gtk_widget_destroy (dialog);
}

static gboolean
check_ntfs_module ()
{
    char        *args[3]= {MODPROBE, "-l", "ntfs"};
    char        *sout = NULL;
    
    g_spawn_sync ("/", 
                    args, NULL, 0, NULL,
                    NULL, &sout, NULL, NULL, 
                    NULL);
    g_debug ("Searching for %s driver : %s", args[2], sout);
    if( !( strcmp(sout, "") ) )
    {
        g_debug ("Can't find an ntfs kernel module, fall back to ntfs-3g ro");
        return FALSE;
    }
        return TRUE;
}

static gchar*
get_node_from_uuid (gchar *uuid)
{
    gchar**     split;
    gchar       path[64];
    gchar*      node;
    GError**     err = NULL;
            
    split = g_new (char*, 2);
    split = g_strsplit (uuid, "=", 2);
    g_debug (" uuid is : %s", split[1]);
    g_snprintf ( path, 64, "%s%s", UUID_DEV, split[1]);
    if ( !( node = g_file_read_link (path, err) ) )
    {
        g_warning ("Can't find device with uuid = %s\n", split[1]);
        g_free (split);
        return " !! UNKNOW DEVICE !!";
    }
    g_free (split);
    split = g_new (char*, 3);
    split = g_strsplit (node, "/", 3);
    node = split[2];
    g_debug (" node path is : /dev/%s", node);
    g_free (split);
    return node;
}
            

static gboolean
enable_internal (gchar **message)
{
    GPtrArray*      devices;
    FILE*           f=0;
    FILE*           tmp=0;
    struct mntent   *entry;
    char*           buffer[64];
    char            tf_path[64];
    int             tfd = -1;
    int             i;

    if( !( f = fopen ( FSTAB, "r") ) ) 
    {
        g_warning ("Error : Can't open /etc/fstab in read mode.\n");
        return FALSE;
    }

    strcpy (tf_path, "/tmp/ntfs-config_XXXXXX");
		
    if( ( tfd = g_mkstemp (tf_path)) < 0 )
    {
        g_warning ("Error : Failed to generate a temp file.\n");
        return FALSE;
    }
    tmp = fdopen(tfd,"w+");
    fprintf (tmp,fstab_header);

    devices = g_ptr_array_new ();
    
    while( ( entry = getmntent ( f ) ) != NULL )
    {
        if ( g_str_has_prefix (entry->mnt_fsname, "UUID=") 
                && g_file_test (UUID_DEV, G_FILE_TEST_IS_DIR) )
        {
            gchar*  node;
            g_debug ("Found an UUID entry. Searching the node path...");
            node = get_node_from_uuid (entry->mnt_fsname);
            g_fprintf (tmp, "# Entry for /dev/%s :\n", node);
        }
        
        if ( !( strcmp(entry->mnt_type, "ntfs") ) ||
                    ( (strstr(entry->mnt_opts, "ro") != NULL) 
                && !( strcmp(entry->mnt_type, "ntfs-3g") ) ) )
        {
            char* mount_point;
            entry->mnt_type = "ntfs-3g";
            entry->mnt_opts = g_strdup_printf ("defaults,locale=%s", 
                        setlocale( LC_ALL, "" ));
            addmntent (tmp, entry);
            g_debug ("Enable write support for %s.", entry->mnt_dir); 
            mount_point = g_strdup (entry->mnt_dir);
            g_ptr_array_add (devices, mount_point);
        }
        else
            addmntent ( tmp, entry);
    }

    fclose (f);
    fflush (tmp);

    if( !( f = fopen ( FSTAB, "w") ) ) 
    {
        g_warning ("Error : Can't open /etc/fstab in write mode.\n");
        return FALSE;
    }

    fseek (tmp, 0, 0);
    while ( ( i = fread (buffer, 1, sizeof (buffer), tmp) ) > 0 )    
        fwrite (buffer, 1, i, f);

    fclose (tmp);
    fclose (f);
    unlink (tf_path);

    remount_all (devices);

    g_free (entry);
    g_ptr_array_free (devices, TRUE);

    return TRUE;
}

static gboolean
disable_internal (gchar **message)
{
    GPtrArray*      devices;
    FILE*           f;
    FILE*           tmp;
    struct mntent   *entry;
    char*           buffer[64];
    char            tf_path[64];
    int             tfd = -1;
    int             i;

    if( !( f = fopen ( FSTAB, "r") ) ) 
    {
        g_warning ("Error : Can't open /etc/fstab in read mode.\n");
        return FALSE;
    }
	
    strcpy (tf_path, "/tmp/ntfs-config_XXXXXX");
	
    if( ( tfd = g_mkstemp (tf_path)) < 0 )
    {
        g_warning ("Error : Failed to generate temp file.\n");		
        return FALSE;
    }
    tmp = fdopen (tfd,"w+");
    fprintf (tmp,fstab_header);

    devices = g_ptr_array_new ();

    while( ( entry = getmntent ( f ) ) != NULL )
    {
        if ( g_str_has_prefix (entry->mnt_fsname, "UUID=") 
                && g_file_test (UUID_DEV, G_FILE_TEST_IS_DIR) )
        {
            gchar*  node;
            g_debug ("Found an UUID entry. Searching the node path...");
            node = get_node_from_uuid (entry->mnt_fsname);
            g_fprintf (tmp, "# Entry for /dev/%s :\n", node);
        }
        
        if ( !( strcmp(entry->mnt_type, "ntfs-3g") ) )
        {
            char* mount_point;
            if ( check_ntfs_module() )
            {
                entry->mnt_type = "ntfs";
                entry->mnt_opts = "umask=222,utf8";
            }
            else
            {
                entry->mnt_type = "ntfs-3g";
                entry->mnt_opts = g_strdup_printf ("defaults,ro,locale=%s", 
                        setlocale( LC_ALL, "" ));
            }
            addmntent ( tmp, entry);
            g_debug ("Disable write support for %s.", entry->mnt_dir); 
            mount_point = g_strdup (entry->mnt_dir);
            g_ptr_array_add (devices, mount_point);
        }
        else
            addmntent ( tmp, entry);
    }

    fclose (f);
    fflush (tmp);

    if( !( f = fopen ( FSTAB, "w") ) ) 
    {
        g_warning ("Error : Can't open /etc/fstab in write mode.\n");
        return FALSE;
    }

    fseek (tmp, 0, 0);
    while ( ( i = fread (buffer, 1, sizeof (buffer), tmp) ) > 0 ) 
        fwrite (buffer, 1, i, f);

    fclose (tmp);
    fclose (f);	
    unlink (tf_path);

    remount_all (devices);

    g_ptr_array_free (devices, TRUE);

    return TRUE;
}

static gboolean
enable_external (gchar **message)
{
    if ( !( g_file_test (write_policy, G_FILE_TEST_EXISTS) ) )
    {
        if ( symlink (FDI_FILE_WRITE, write_policy) != 0 )
        {
            g_warning ("Error : Can't create write policy file.\n");
            return FALSE;
        }
        g_debug ("Enable write support for external device.");
    }
    
    if ( g_file_test (ro_policy, G_FILE_TEST_EXISTS) )
    {
        if ( unlink (ro_policy) != 0 )
        {
            g_warning ("Error : Can't remove ro policy file.\n");
            return FALSE;
        }
    }

    return TRUE;
}

static gboolean
disable_external (gchar **message)
{
    if ( g_file_test (write_policy, G_FILE_TEST_EXISTS) )
    {
        if ( unlink (write_policy) != 0 )
        {
            g_warning ("Error : Can't remove write policy file.\n");
            return FALSE;
        }
        g_debug("Disable write support for external device.");
    }
    
    if ( !( g_file_test (ro_policy, G_FILE_TEST_EXISTS) ) )
    {
        if ( symlink (FDI_FILE_RO, ro_policy) != 0 )
        {
            g_warning ("Error : Can't create write policy file.\n");
            return FALSE;
        }
    }

    return TRUE;
}

static void
get_widget_settings (App *app,
            Settings *settings)
{
    if ( !settings )
        return;

    settings->internal = gtk_toggle_button_get_active (app->internal);
    settings->external = gtk_toggle_button_get_active (app->external);
}

static void
on_apply (GtkWidget *widget,
            App       *app)
{
    Settings        settings;
    GError          *err = NULL;
	gchar           *message;
    
    get_widget_settings (app, &settings);

    /* On the first run, backup the current /etc/fstab */
    if ( !( g_file_test(fstab_sav, G_FILE_TEST_EXISTS) ) )
    {
        FILE	*from, *to;
        char 	buffer[64];
        int		i;

        if( !( from = fopen ( FSTAB, "r") ) ) 
        {
            g_warning ("Error : Can't open /etc/fstab in read mode.\n");
            return;
        }

        if( !( to = fopen ( fstab_sav, "w+") ) ) 
        {
            err = g_error_new (1, -1, _("Error : Can't backup /etc/fstab.\n"));
            show_error (err);
            return;
        }
        
        g_debug ("Creating backup file...");
        fseek (from, 0, 0);
        while ( ( i = fread (buffer, 1, sizeof(buffer), from) ) > 0 )    
        fwrite (buffer, 1, i, to);

        fclose (from);
        fclose (to);
    }

    if( settings.internal )
    {
        if( ! (enable_internal (&message) ) )
        {
            err = g_error_new (1, -1, _("Error : Failed to enable write support \
for internal device."));
            show_error (err);
        }
    }
    else
    {
        if( ! (disable_internal (&message) ) )
        {
            err = g_error_new (1, -1, _("Error : Failed to disable write support \
for internal device."));
            show_error (err);
        }
    }

    if( settings.external )
    {
        if( ! (enable_external (&message) ) )
        {
            err = g_error_new (1, -1, _("Error : Failed to enable write support \
for external device.\nThis probably means that your installation is not complete\
 please reinstall ntfs-config and retry. Thanks."));
            show_error (err);
        }
    }
    else
    {
        if( ! (disable_external (&message) ) )
        {
            err = g_error_new (1, -1, _("Error : Failed to disable write support \
for external device."));
            show_error (err);
        }
    }
	
    gtk_main_quit ();
}
    
static void
on_internal_toogle (GtkWidget *widget,
            App *app)
{
    gtk_toggle_button_set_inconsistent (app->internal, FALSE);
}

static gboolean
set_widgets (App *app,
        GError **err)
{
    gboolean		external_setting = FALSE;
    int				internal_setting = 0;
    FILE* 			f;
    struct mntent 	*entry;

    if( !( f = fopen (FSTAB, "r") ) ) 
    {
        *err = g_error_new (1, -1, _("Error : Can't open /etc/fstab\n \
This is not going to work."));
        return FALSE;
    }

    g_debug("Looking at /etc/fstab...");
    while( ( entry = getmntent (f) ) != NULL ) 
    {
        if ( !( strcmp(entry->mnt_type, "ntfs") ) ||
                    ( (strstr(entry->mnt_opts, "ro") != NULL) 
                && !( strcmp(entry->mnt_type, "ntfs-3g") ) ) )
        {
            g_debug("-> found %s NTFS read-only (%s)", 
                    entry->mnt_fsname, entry->mnt_type);
            if ( ( internal_setting == 2 ) || ( internal_setting == 3 ) )
                internal_setting = 3;
            else
                internal_setting = 1;
            continue;
        }
        if ( !( strcmp(entry->mnt_type, "ntfs-3g") ) )
        {
            g_debug("-> found %s NTFS read-write (%s)", 
                    entry->mnt_fsname, entry->mnt_type);
            if ( ( internal_setting == 1 ) || ( internal_setting == 3 ) )
                internal_setting = 3;
            else
                internal_setting = 2;
        }
    }

    if ( internal_setting == 0 )
        gtk_widget_set_sensitive (GTK_WIDGET (app->internal), FALSE);
    if ( internal_setting == 1 )
        gtk_toggle_button_set_active (app->internal, FALSE);
    if ( internal_setting == 2 )
        gtk_toggle_button_set_active (app->internal, TRUE);
    if ( internal_setting == 3 )
        gtk_toggle_button_set_inconsistent (app->internal, TRUE);

    external_setting = g_file_test (write_policy, G_FILE_TEST_EXISTS);

    if( !( g_file_test (FDI_FILE_WRITE, G_FILE_TEST_EXISTS) )
        || !( g_file_test (FDI_FILE_RO, G_FILE_TEST_EXISTS) ) )
    {
        *err = g_error_new(1, -1, _("Error : Could not find the fdi file.\n\
Configuration of external device is disable.\nYou'll probably need to \
reinstall ntfs-config\nto restore it."));
        show_error (*err);
        gtk_widget_set_sensitive (GTK_WIDGET (app->external), FALSE);
    }
    if( !( g_file_test (hal_conf_dir, G_FILE_TEST_IS_DIR) ) )
    {
        g_warning (_("Warning : Could not find %s.\n\
Assuming that this means the HAL is not properly installed\n\
and therefore configuration of external devices is disabled."), hal_conf_dir);
        gtk_widget_set_sensitive (GTK_WIDGET (app->external), FALSE);
    }

    gtk_toggle_button_set_active (app->external, external_setting);

    return TRUE;
}

static gboolean
init_app (App *app,
        GError **err)
{
    GladeXML     *xml;

    xml = glade_xml_new (GLADE_FILE, NULL, NULL);

    if ( !xml )
    {
        g_warning ("Could not open " GLADE_FILE);
        return FALSE;
    }

    app->dialog = glade_xml_get_widget (xml, "dialog1");
    app->internal = GTK_TOGGLE_BUTTON (
                    glade_xml_get_widget (xml, "internal"));
    app->external = GTK_TOGGLE_BUTTON (
                    glade_xml_get_widget (xml, "external"));
    app->aboutbutton = GTK_BUTTON (
                    glade_xml_get_widget (xml, "aboutbutton"));
    app->cancelbutton = GTK_BUTTON (
                    glade_xml_get_widget (xml, "cancelbutton"));
    app->applybutton = GTK_BUTTON (
                    glade_xml_get_widget (xml, "applybutton"));

    if ( !( set_widgets(app, err) ) ) 
        return FALSE;

    g_signal_connect (app->aboutbutton, "clicked",
                G_CALLBACK (show_about), app);
    g_signal_connect (app->cancelbutton, "clicked",
                G_CALLBACK (gtk_main_quit), NULL);
    g_signal_connect (app->internal, "toggled",
                G_CALLBACK (on_internal_toogle), app);
    g_signal_connect (app->dialog, "destroy",
                G_CALLBACK (gtk_main_quit), NULL);	
    g_signal_connect (app->applybutton, "clicked",
                G_CALLBACK (on_apply), app);
	
    return TRUE;
}

static void 
my_empty_log_handler (const gchar *log_domain,
            GLogLevelFlags log_level,
            const gchar *message,
            gpointer user_data)
{
}

static void
revert_changes (void)
{
    gchar   reply[128];
    g_print("Warning : This will revert all changes made by ntfs-config\n\
Do you want to continue ? (y/n) ");
    scanf ( "%s" , reply);
    if ( !( strcmp(reply, "y") ) )
    {
        g_print("\nReverting your /etc/fstab...");
        if ( ( g_file_test (fstab_sav, G_FILE_TEST_EXISTS) ) )
        {
            FILE	*from, *to;
            char 	buffer[64];
            int		i;

            from = fopen ( fstab_sav, "r");
            to = fopen ( FSTAB, "w+");
            fseek (from, 0, 0);
            while ( ( i = fread (buffer, 1, sizeof(buffer), from) ) > 0 )    
                fwrite (buffer, 1, i, to);

            fclose (from);
            fclose (to);
            g_print("Done.\n");
        }
        else
            g_print ("\nCan't find any backup file. Did you already run ntfs-config ?\n"); 
        
        g_print("Removing all NTFS external device policy...");
        if ( g_file_test (write_policy, G_FILE_TEST_EXISTS) )
        {
            if ( unlink (write_policy) != 0 )
                g_warning ("\nError : Can't remove write policy file.\n");
        }
        if ( g_file_test (ro_policy, G_FILE_TEST_EXISTS) )
        {
            if ( unlink (ro_policy) != 0 )
                g_warning ("\nError : Can't remove read policy file.\n");
        }
        g_print("Done.\n"); 
    }
    else
        g_print("No change made. Exiting...\n");
    
    return;
}

int
main (int argc, char **argv)
{
    App                 *app;
    GError              *err = NULL;
    static gboolean     opt_debug = FALSE;
    static gboolean     opt_version = FALSE;
    static gboolean     opt_revert = FALSE;
    GOptionContext      *context = NULL;

    static GOptionEntry entries[] =
        {
            { "debug", 'd', 0, G_OPTION_ARG_NONE, &opt_debug, \
                        "Display more debug information", NULL},
            { "version", 'v', 0, G_OPTION_ARG_NONE, &opt_version,
                        "Display information on the version", NULL},
            { "revert", 'r', 0, G_OPTION_ARG_NONE, &opt_revert, \
                        "Revert all changes made by ntfs-config", NULL}
        };

    bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
    bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
    textdomain (GETTEXT_PACKAGE);

    context = g_option_context_new ("");
    g_option_context_add_main_entries (context, entries, PACKAGE);
    g_option_context_add_group (context, gtk_get_option_group (FALSE));
    g_option_context_parse (context, &argc, &argv, &err);

    if ( opt_version )
	{
        g_print(_("%s.\nEnable/disable write support \
for NTFS with a simple click.\n"), PACKAGE_STRING);
		return 0;
	}

    if ( !( opt_debug ) ) 
    {
        g_log_set_handler (NULL, G_LOG_LEVEL_DEBUG, my_empty_log_handler, NULL);
    }
    
    if ( opt_revert )
    {
        /* We should be able to use --revert in a non graphical environnement */
        if ( getuid() )
            g_warning("Error : This programm need to be run as root.\n");
        else
            revert_changes();
        return 0;
    }
    
    gtk_init (&argc, &argv);

    gtk_window_set_default_icon_name ("gnome-dev-harddisk");
    
    if ( geteuid() )
	{
        err = g_error_new(1, -1, _("Error : This programm need to be run as root.\n"));
        show_error (err);
        return 0;
    }
    
    if ( !( g_file_test (NTFS_BINARY, G_FILE_TEST_EXISTS) ) )
    {
        err = g_error_new(1, -1, _("Error : It seams that the ntfs-3g driver \
necessary\nto get write support, is not install on your system.\n\
Check http://www.ntfs-3g.org to know how to install it.\n"));
        show_error (err);
        return 0;
    }

    g_debug ("-- Initialization --");

    if ( !( search_new_devices () ) )
    {
        err = g_error_new(1, -1, _("Error : An error occured when trying to \
initialize HAL. Can't search for new partition.\n"));
        show_error (err);
    }
    
    app = g_new0 (App, 1);

    if ( !( init_app (app, &err) ) )
    {
        show_error (err);
        return 0;
    }
    
    g_debug ("-- End of the initialization --");

    gtk_widget_show (app->dialog);

    gtk_window_set_icon_name (GTK_WINDOW (app->dialog), "gnome-dev-harddisk");

    gtk_main ();

    return 0;
}
