/*
 	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com).
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#include "Entity_PKI_ASN1.h"
#include <PKI_ERR.h>
#include <openssl/asn1t.h>

ASN1_SEQUENCE(STORED_ENTITY_CONF) = {
	ASN1_SIMPLE(STORED_ENTITY_CONF, conf, ENTITY_CONF),
	ASN1_SIMPLE(STORED_ENTITY_CONF, certificate, X509),
}ASN1_SEQUENCE_END(STORED_ENTITY_CONF)
ASN1_SEQUENCE(STORED_ENTITY_CONF_BETA4) = {
	ASN1_SIMPLE(STORED_ENTITY_CONF_BETA4, conf, ENTITY_CONF_BETA4),
	ASN1_SIMPLE(STORED_ENTITY_CONF_BETA4, certificate, X509),
}ASN1_SEQUENCE_END(STORED_ENTITY_CONF_BETA4)
ASN1_SEQUENCE(PKI_CONF) = {
	ASN1_SEQUENCE_OF(PKI_CONF, confs, STORED_ENTITY_CONF),
	ASN1_SIMPLE(PKI_CONF, crypt_confs, EXPORTED_PKI_CONF),
	ASN1_SEQUENCE_OF(PKI_CONF, Repositories, REP_ENTRY_INFO),
	ASN1_SIMPLE(PKI_CONF, cas, INTERNAL_PKI_CA),
	ASN1_SIMPLE(PKI_CONF, Publications, PUBS_INFO),
	ASN1_SIMPLE(PKI_CONF, conf, ENTITY_CONF),
	ASN1_SEQUENCE_OF(PKI_CONF, links, ENTITY_LINKS),
}ASN1_SEQUENCE_END(PKI_CONF)
ASN1_SEQUENCE(PKI_CONF_BETA4) = {
	ASN1_SEQUENCE_OF(PKI_CONF_BETA4, confs, STORED_ENTITY_CONF_BETA4),
	ASN1_SIMPLE(PKI_CONF_BETA4, crypt_confs, EXPORTED_PKI_CONF),
	ASN1_SEQUENCE_OF(PKI_CONF_BETA4, Repositories, REP_ENTRY_INFO),
	ASN1_SIMPLE(PKI_CONF_BETA4, cas, INTERNAL_PKI_CA),
	ASN1_SIMPLE(PKI_CONF_BETA4, Publications, PUBS_INFO),
	ASN1_SIMPLE(PKI_CONF_BETA4, conf, ENTITY_CONF_BETA4),
	ASN1_SEQUENCE_OF(PKI_CONF_BETA4, links, ENTITY_LINKS),
}ASN1_SEQUENCE_END(PKI_CONF_BETA4)
StoredEntityConf StoredEntityConf::EmptyInstance;
bool StoredEntityConf::set_certificate(const PKI_CERT & c_certificate)
{
	m_certificate = c_certificate;
	return true;
}

const PKI_CERT & StoredEntityConf::get_certificate() const
{
	return m_certificate;
}

PKI_CERT & StoredEntityConf::get_certificate()
{
	return m_certificate;
}

bool StoredEntityConf::set_conf(const EntityConf & c_conf)
{
	m_conf = c_conf;
	return true;
}

const EntityConf & StoredEntityConf::get_conf() const
{
	return m_conf;
}

EntityConf & StoredEntityConf::get_conf()
{
	return m_conf;
}

StoredEntityConf::StoredEntityConf():NewPKIObject()
{
	resetAll();
}

StoredEntityConf::StoredEntityConf(const StoredEntityConf & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

StoredEntityConf::~StoredEntityConf()
{
	Clear();
}

void StoredEntityConf::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void StoredEntityConf::freeAll()
{
}

void StoredEntityConf::resetAll()
{
	m_certificate.Clear();
	m_conf.Clear();
}

bool StoredEntityConf::load_Datas(const STORED_ENTITY_CONF * Datas)
{
	Clear();
	if(Datas->certificate)
	{
		if(!m_certificate.load_Datas(Datas->certificate))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->conf)
	{
		if(!m_conf.load_Datas(Datas->conf))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_isOk=true;
	return true;
}

bool StoredEntityConf::give_Datas(STORED_ENTITY_CONF ** Datas) const
{
	if(!(*Datas) && !(*Datas = (STORED_ENTITY_CONF*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->certificate && !((*Datas)->certificate = (X509*)ASN1_item_new(ASN1_ITEM_rptr(X509))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_certificate.give_Datas(&(*Datas)->certificate))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->certificate, ASN1_ITEM_rptr(X509));
		(*Datas)->certificate = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->conf && !((*Datas)->conf = (ENTITY_CONF*)ASN1_item_new(ASN1_ITEM_rptr(ENTITY_CONF))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_conf.give_Datas(&(*Datas)->conf))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->conf, ASN1_ITEM_rptr(ENTITY_CONF));
		(*Datas)->conf = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool StoredEntityConf::operator=(const StoredEntityConf & other)
{
	Clear();
	m_certificate = other.m_certificate;
	m_conf = other.m_conf;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * StoredEntityConf::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(STORED_ENTITY_CONF);
}
StoredEntityConfBeta4 StoredEntityConfBeta4::EmptyInstance;
bool StoredEntityConfBeta4::set_certificate(const PKI_CERT & c_certificate)
{
	m_certificate = c_certificate;
	return true;
}

const PKI_CERT & StoredEntityConfBeta4::get_certificate() const
{
	return m_certificate;
}

PKI_CERT & StoredEntityConfBeta4::get_certificate()
{
	return m_certificate;
}

bool StoredEntityConfBeta4::set_conf(const EntityConfBeta4 & c_conf)
{
	m_conf = c_conf;
	return true;
}

const EntityConfBeta4 & StoredEntityConfBeta4::get_conf() const
{
	return m_conf;
}

EntityConfBeta4 & StoredEntityConfBeta4::get_conf()
{
	return m_conf;
}

StoredEntityConfBeta4::StoredEntityConfBeta4():NewPKIObject()
{
	resetAll();
}

StoredEntityConfBeta4::StoredEntityConfBeta4(const StoredEntityConfBeta4 & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

StoredEntityConfBeta4::~StoredEntityConfBeta4()
{
	Clear();
}

void StoredEntityConfBeta4::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void StoredEntityConfBeta4::freeAll()
{
}

void StoredEntityConfBeta4::resetAll()
{
	m_certificate.Clear();
	m_conf.Clear();
}

bool StoredEntityConfBeta4::load_Datas(const STORED_ENTITY_CONF_BETA4 * Datas)
{
	Clear();
	if(Datas->certificate)
	{
		if(!m_certificate.load_Datas(Datas->certificate))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->conf)
	{
		if(!m_conf.load_Datas(Datas->conf))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_isOk=true;
	return true;
}

bool StoredEntityConfBeta4::give_Datas(STORED_ENTITY_CONF_BETA4 ** Datas) const
{
	if(!(*Datas) && !(*Datas = (STORED_ENTITY_CONF_BETA4*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->certificate && !((*Datas)->certificate = (X509*)ASN1_item_new(ASN1_ITEM_rptr(X509))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_certificate.give_Datas(&(*Datas)->certificate))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->certificate, ASN1_ITEM_rptr(X509));
		(*Datas)->certificate = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->conf && !((*Datas)->conf = (ENTITY_CONF_BETA4*)ASN1_item_new(ASN1_ITEM_rptr(ENTITY_CONF_BETA4))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_conf.give_Datas(&(*Datas)->conf))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->conf, ASN1_ITEM_rptr(ENTITY_CONF_BETA4));
		(*Datas)->conf = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool StoredEntityConfBeta4::operator=(const StoredEntityConfBeta4 & other)
{
	Clear();
	m_certificate = other.m_certificate;
	m_conf = other.m_conf;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * StoredEntityConfBeta4::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(STORED_ENTITY_CONF_BETA4);
}
PkiConf PkiConf::EmptyInstance;
bool PkiConf::set_publications(const PubsInfo & c_publications)
{
	m_publications = c_publications;
	return true;
}

const PubsInfo & PkiConf::get_publications() const
{
	return m_publications;
}

PubsInfo & PkiConf::get_publications()
{
	return m_publications;
}

bool PkiConf::set_repositories(const mVector< RepEntryInfo > & c_repositories)
{
	m_repositories = c_repositories;
	return true;
}

const mVector< RepEntryInfo > & PkiConf::get_repositories() const
{
	return m_repositories;
}

mVector< RepEntryInfo > & PkiConf::get_repositories()
{
	return m_repositories;
}

bool PkiConf::set_cas(const InternalPkiCa & c_cas)
{
	m_cas = c_cas;
	return true;
}

const InternalPkiCa & PkiConf::get_cas() const
{
	return m_cas;
}

InternalPkiCa & PkiConf::get_cas()
{
	return m_cas;
}

bool PkiConf::set_conf(const EntityConf & c_conf)
{
	m_conf = c_conf;
	return true;
}

const EntityConf & PkiConf::get_conf() const
{
	return m_conf;
}

EntityConf & PkiConf::get_conf()
{
	return m_conf;
}

bool PkiConf::set_confs(const mVector< StoredEntityConf > & c_confs)
{
	m_confs = c_confs;
	return true;
}

const mVector< StoredEntityConf > & PkiConf::get_confs() const
{
	return m_confs;
}

mVector< StoredEntityConf > & PkiConf::get_confs()
{
	return m_confs;
}

bool PkiConf::set_cryptConfs(const ExportedPkiConf & c_cryptConfs)
{
	m_cryptConfs = c_cryptConfs;
	return true;
}

const ExportedPkiConf & PkiConf::get_cryptConfs() const
{
	return m_cryptConfs;
}

ExportedPkiConf & PkiConf::get_cryptConfs()
{
	return m_cryptConfs;
}

bool PkiConf::set_links(const mVector< EntityLinks > & c_links)
{
	m_links = c_links;
	return true;
}

const mVector< EntityLinks > & PkiConf::get_links() const
{
	return m_links;
}

mVector< EntityLinks > & PkiConf::get_links()
{
	return m_links;
}

bool PkiConf::to_SignEncrypt(Asn1EncryptSign & cryptinfo, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey, const EVP_MD * sig_md, const EVP_CIPHER * crypt_cypher) const
{
	PKI_CONF * c_localvar = NULL;
	if(!give_Datas(&c_localvar))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!Private_toSignEncrypt(cryptinfo, get_ASN1_ITEM(), (ASN1_VALUE*)c_localvar, sig_pkey, crypt_pkey, sig_md, crypt_cypher))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
		ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
	}
	ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
	return true;
}

bool PkiConf::from_SignEncrypt(const Asn1EncryptSign & cryptinfo, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey)
{
	PKI_CONF * c_localvar = NULL;
	if(!Private_fromSignEncrypt(cryptinfo, get_ASN1_ITEM(), (ASN1_VALUE**)&c_localvar, sig_pkey, crypt_pkey))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!load_Datas(c_localvar))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
		return false;
	}
	ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
	return true;
}

PkiConf::PkiConf():NewPKIObject(), LocalEntityConf()
{
	resetAll();
}

PkiConf::PkiConf(const PkiConf & other):NewPKIObject(), LocalEntityConf()
{
	resetAll();
	*this = other;
}

PkiConf::~PkiConf()
{
	Clear();
}

void PkiConf::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void PkiConf::freeAll()
{
}

void PkiConf::resetAll()
{
	m_publications.Clear();
	m_repositories.clear();
	m_cas.Clear();
	m_conf.Clear();
	m_confs.clear();
	m_cryptConfs.Clear();
	m_links.clear();
}

bool PkiConf::load_Datas(const PKI_CONF * Datas)
{
	Clear();
	ENTITY_LINKS * currlinks;
	STORED_ENTITY_CONF * currconfs;
	int i;
	REP_ENTRY_INFO * currrepositories;
	if(Datas->Publications)
	{
		if(!m_publications.load_Datas(Datas->Publications))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->Repositories)
	{
		for(i=0; i<SKM_sk_num(REP_ENTRY_INFO, Datas->Repositories); i++)
		{
			currrepositories = SKM_sk_value(REP_ENTRY_INFO, Datas->Repositories, i);
			if(!currrepositories)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			m_repositories.insert(m_repositories.begin() + i);
			if(!m_repositories[i].load_Datas(currrepositories))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
		}
	}
	if(Datas->cas)
	{
		if(!m_cas.load_Datas(Datas->cas))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->conf)
	{
		if(!m_conf.load_Datas(Datas->conf))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->confs)
	{
		for(i=0; i<SKM_sk_num(STORED_ENTITY_CONF, Datas->confs); i++)
		{
			currconfs = SKM_sk_value(STORED_ENTITY_CONF, Datas->confs, i);
			if(!currconfs)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			m_confs.insert(m_confs.begin() + i);
			if(!m_confs[i].load_Datas(currconfs))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
		}
	}
	if(Datas->crypt_confs)
	{
		if(!m_cryptConfs.load_Datas(Datas->crypt_confs))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->links)
	{
		for(i=0; i<SKM_sk_num(ENTITY_LINKS, Datas->links); i++)
		{
			currlinks = SKM_sk_value(ENTITY_LINKS, Datas->links, i);
			if(!currlinks)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			m_links.insert(m_links.begin() + i);
			if(!m_links[i].load_Datas(currlinks))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
		}
	}
	m_isOk=true;
	return true;
}

bool PkiConf::give_Datas(PKI_CONF ** Datas) const
{
	if(!(*Datas) && !(*Datas = (PKI_CONF*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	ENTITY_LINKS * currlinks;
	STORED_ENTITY_CONF * currconfs;
	REP_ENTRY_INFO * currrepositories;
	size_t i;
	if(!(*Datas)->Publications && !((*Datas)->Publications = (PUBS_INFO*)ASN1_item_new(ASN1_ITEM_rptr(PUBS_INFO))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_publications.give_Datas(&(*Datas)->Publications))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->Publications, ASN1_ITEM_rptr(PUBS_INFO));
		(*Datas)->Publications = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->Repositories && !((*Datas)->Repositories = SKM_sk_new_null(REP_ENTRY_INFO)))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	for(i = 0 ; i < m_repositories.size() ; i++)
	{
		currrepositories = NULL;
		if(!m_repositories[i].give_Datas(&currrepositories))
		{
			ASN1_item_free((ASN1_VALUE*)currrepositories, ASN1_ITEM_rptr(REP_ENTRY_INFO));
			currrepositories = NULL;
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(SKM_sk_push(REP_ENTRY_INFO, (*Datas)->Repositories, currrepositories) < 0)
		{
			ASN1_item_free((ASN1_VALUE*)currrepositories, ASN1_ITEM_rptr(REP_ENTRY_INFO));
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
			return false;
		}
	}
	if(!(*Datas)->cas && !((*Datas)->cas = (INTERNAL_PKI_CA*)ASN1_item_new(ASN1_ITEM_rptr(INTERNAL_PKI_CA))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_cas.give_Datas(&(*Datas)->cas))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->cas, ASN1_ITEM_rptr(INTERNAL_PKI_CA));
		(*Datas)->cas = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->conf && !((*Datas)->conf = (ENTITY_CONF*)ASN1_item_new(ASN1_ITEM_rptr(ENTITY_CONF))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_conf.give_Datas(&(*Datas)->conf))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->conf, ASN1_ITEM_rptr(ENTITY_CONF));
		(*Datas)->conf = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->confs && !((*Datas)->confs = SKM_sk_new_null(STORED_ENTITY_CONF)))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	for(i = 0 ; i < m_confs.size() ; i++)
	{
		currconfs = NULL;
		if(!m_confs[i].give_Datas(&currconfs))
		{
			ASN1_item_free((ASN1_VALUE*)currconfs, ASN1_ITEM_rptr(STORED_ENTITY_CONF));
			currconfs = NULL;
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(SKM_sk_push(STORED_ENTITY_CONF, (*Datas)->confs, currconfs) < 0)
		{
			ASN1_item_free((ASN1_VALUE*)currconfs, ASN1_ITEM_rptr(STORED_ENTITY_CONF));
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
			return false;
		}
	}
	if(!(*Datas)->crypt_confs && !((*Datas)->crypt_confs = (EXPORTED_PKI_CONF*)ASN1_item_new(ASN1_ITEM_rptr(EXPORTED_PKI_CONF))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_cryptConfs.give_Datas(&(*Datas)->crypt_confs))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->crypt_confs, ASN1_ITEM_rptr(EXPORTED_PKI_CONF));
		(*Datas)->crypt_confs = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->links && !((*Datas)->links = SKM_sk_new_null(ENTITY_LINKS)))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	for(i = 0 ; i < m_links.size() ; i++)
	{
		currlinks = NULL;
		if(!m_links[i].give_Datas(&currlinks))
		{
			ASN1_item_free((ASN1_VALUE*)currlinks, ASN1_ITEM_rptr(ENTITY_LINKS));
			currlinks = NULL;
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(SKM_sk_push(ENTITY_LINKS, (*Datas)->links, currlinks) < 0)
		{
			ASN1_item_free((ASN1_VALUE*)currlinks, ASN1_ITEM_rptr(ENTITY_LINKS));
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
			return false;
		}
	}
	return true;
}

bool PkiConf::operator=(const PkiConf & other)
{
	Clear();
	m_publications = other.m_publications;
	m_repositories = other.m_repositories;
	m_cas = other.m_cas;
	m_conf = other.m_conf;
	m_confs = other.m_confs;
	m_cryptConfs = other.m_cryptConfs;
	m_links = other.m_links;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * PkiConf::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(PKI_CONF);
}
PkiConfBeta4 PkiConfBeta4::EmptyInstance;
bool PkiConfBeta4::set_publications(const PubsInfo & c_publications)
{
	m_publications = c_publications;
	return true;
}

const PubsInfo & PkiConfBeta4::get_publications() const
{
	return m_publications;
}

PubsInfo & PkiConfBeta4::get_publications()
{
	return m_publications;
}

bool PkiConfBeta4::set_repositories(const mVector< RepEntryInfo > & c_repositories)
{
	m_repositories = c_repositories;
	return true;
}

const mVector< RepEntryInfo > & PkiConfBeta4::get_repositories() const
{
	return m_repositories;
}

mVector< RepEntryInfo > & PkiConfBeta4::get_repositories()
{
	return m_repositories;
}

bool PkiConfBeta4::set_cas(const InternalPkiCa & c_cas)
{
	m_cas = c_cas;
	return true;
}

const InternalPkiCa & PkiConfBeta4::get_cas() const
{
	return m_cas;
}

InternalPkiCa & PkiConfBeta4::get_cas()
{
	return m_cas;
}

bool PkiConfBeta4::set_conf(const EntityConfBeta4 & c_conf)
{
	m_conf = c_conf;
	return true;
}

const EntityConfBeta4 & PkiConfBeta4::get_conf() const
{
	return m_conf;
}

EntityConfBeta4 & PkiConfBeta4::get_conf()
{
	return m_conf;
}

bool PkiConfBeta4::set_confs(const mVector< StoredEntityConfBeta4 > & c_confs)
{
	m_confs = c_confs;
	return true;
}

const mVector< StoredEntityConfBeta4 > & PkiConfBeta4::get_confs() const
{
	return m_confs;
}

mVector< StoredEntityConfBeta4 > & PkiConfBeta4::get_confs()
{
	return m_confs;
}

bool PkiConfBeta4::set_cryptConfs(const ExportedPkiConf & c_cryptConfs)
{
	m_cryptConfs = c_cryptConfs;
	return true;
}

const ExportedPkiConf & PkiConfBeta4::get_cryptConfs() const
{
	return m_cryptConfs;
}

ExportedPkiConf & PkiConfBeta4::get_cryptConfs()
{
	return m_cryptConfs;
}

bool PkiConfBeta4::set_links(const mVector< EntityLinks > & c_links)
{
	m_links = c_links;
	return true;
}

const mVector< EntityLinks > & PkiConfBeta4::get_links() const
{
	return m_links;
}

mVector< EntityLinks > & PkiConfBeta4::get_links()
{
	return m_links;
}

bool PkiConfBeta4::to_SignEncrypt(Asn1EncryptSign & cryptinfo, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey, const EVP_MD * sig_md, const EVP_CIPHER * crypt_cypher) const
{
	PKI_CONF_BETA4 * c_localvar = NULL;
	if(!give_Datas(&c_localvar))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!Private_toSignEncrypt(cryptinfo, get_ASN1_ITEM(), (ASN1_VALUE*)c_localvar, sig_pkey, crypt_pkey, sig_md, crypt_cypher))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
		ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
	}
	ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
	return true;
}

bool PkiConfBeta4::from_SignEncrypt(const Asn1EncryptSign & cryptinfo, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey)
{
	PKI_CONF_BETA4 * c_localvar = NULL;
	if(!Private_fromSignEncrypt(cryptinfo, get_ASN1_ITEM(), (ASN1_VALUE**)&c_localvar, sig_pkey, crypt_pkey))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!load_Datas(c_localvar))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
		return false;
	}
	ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
	return true;
}

PkiConfBeta4::PkiConfBeta4():NewPKIObject(), LocalEntityConfBeta4()
{
	resetAll();
}

PkiConfBeta4::PkiConfBeta4(const PkiConfBeta4 & other):NewPKIObject(), LocalEntityConfBeta4()
{
	resetAll();
	*this = other;
}

PkiConfBeta4::~PkiConfBeta4()
{
	Clear();
}

void PkiConfBeta4::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void PkiConfBeta4::freeAll()
{
}

void PkiConfBeta4::resetAll()
{
	m_publications.Clear();
	m_repositories.clear();
	m_cas.Clear();
	m_conf.Clear();
	m_confs.clear();
	m_cryptConfs.Clear();
	m_links.clear();
}

bool PkiConfBeta4::load_Datas(const PKI_CONF_BETA4 * Datas)
{
	Clear();
	ENTITY_LINKS * currlinks;
	STORED_ENTITY_CONF_BETA4 * currconfs;
	int i;
	REP_ENTRY_INFO * currrepositories;
	if(Datas->Publications)
	{
		if(!m_publications.load_Datas(Datas->Publications))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->Repositories)
	{
		for(i=0; i<SKM_sk_num(REP_ENTRY_INFO, Datas->Repositories); i++)
		{
			currrepositories = SKM_sk_value(REP_ENTRY_INFO, Datas->Repositories, i);
			if(!currrepositories)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			m_repositories.insert(m_repositories.begin() + i);
			if(!m_repositories[i].load_Datas(currrepositories))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
		}
	}
	if(Datas->cas)
	{
		if(!m_cas.load_Datas(Datas->cas))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->conf)
	{
		if(!m_conf.load_Datas(Datas->conf))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->confs)
	{
		for(i=0; i<SKM_sk_num(STORED_ENTITY_CONF_BETA4, Datas->confs); i++)
		{
			currconfs = SKM_sk_value(STORED_ENTITY_CONF_BETA4, Datas->confs, i);
			if(!currconfs)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			m_confs.insert(m_confs.begin() + i);
			if(!m_confs[i].load_Datas(currconfs))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
		}
	}
	if(Datas->crypt_confs)
	{
		if(!m_cryptConfs.load_Datas(Datas->crypt_confs))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->links)
	{
		for(i=0; i<SKM_sk_num(ENTITY_LINKS, Datas->links); i++)
		{
			currlinks = SKM_sk_value(ENTITY_LINKS, Datas->links, i);
			if(!currlinks)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			m_links.insert(m_links.begin() + i);
			if(!m_links[i].load_Datas(currlinks))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
		}
	}
	m_isOk=true;
	return true;
}

bool PkiConfBeta4::give_Datas(PKI_CONF_BETA4 ** Datas) const
{
	if(!(*Datas) && !(*Datas = (PKI_CONF_BETA4*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	ENTITY_LINKS * currlinks;
	STORED_ENTITY_CONF_BETA4 * currconfs;
	REP_ENTRY_INFO * currrepositories;
	size_t i;
	if(!(*Datas)->Publications && !((*Datas)->Publications = (PUBS_INFO*)ASN1_item_new(ASN1_ITEM_rptr(PUBS_INFO))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_publications.give_Datas(&(*Datas)->Publications))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->Publications, ASN1_ITEM_rptr(PUBS_INFO));
		(*Datas)->Publications = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->Repositories && !((*Datas)->Repositories = SKM_sk_new_null(REP_ENTRY_INFO)))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	for(i = 0 ; i < m_repositories.size() ; i++)
	{
		currrepositories = NULL;
		if(!m_repositories[i].give_Datas(&currrepositories))
		{
			ASN1_item_free((ASN1_VALUE*)currrepositories, ASN1_ITEM_rptr(REP_ENTRY_INFO));
			currrepositories = NULL;
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(SKM_sk_push(REP_ENTRY_INFO, (*Datas)->Repositories, currrepositories) < 0)
		{
			ASN1_item_free((ASN1_VALUE*)currrepositories, ASN1_ITEM_rptr(REP_ENTRY_INFO));
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
			return false;
		}
	}
	if(!(*Datas)->cas && !((*Datas)->cas = (INTERNAL_PKI_CA*)ASN1_item_new(ASN1_ITEM_rptr(INTERNAL_PKI_CA))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_cas.give_Datas(&(*Datas)->cas))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->cas, ASN1_ITEM_rptr(INTERNAL_PKI_CA));
		(*Datas)->cas = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->conf && !((*Datas)->conf = (ENTITY_CONF_BETA4*)ASN1_item_new(ASN1_ITEM_rptr(ENTITY_CONF_BETA4))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_conf.give_Datas(&(*Datas)->conf))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->conf, ASN1_ITEM_rptr(ENTITY_CONF_BETA4));
		(*Datas)->conf = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->confs && !((*Datas)->confs = SKM_sk_new_null(STORED_ENTITY_CONF_BETA4)))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	for(i = 0 ; i < m_confs.size() ; i++)
	{
		currconfs = NULL;
		if(!m_confs[i].give_Datas(&currconfs))
		{
			ASN1_item_free((ASN1_VALUE*)currconfs, ASN1_ITEM_rptr(STORED_ENTITY_CONF_BETA4));
			currconfs = NULL;
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(SKM_sk_push(STORED_ENTITY_CONF_BETA4, (*Datas)->confs, currconfs) < 0)
		{
			ASN1_item_free((ASN1_VALUE*)currconfs, ASN1_ITEM_rptr(STORED_ENTITY_CONF_BETA4));
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
			return false;
		}
	}
	if(!(*Datas)->crypt_confs && !((*Datas)->crypt_confs = (EXPORTED_PKI_CONF*)ASN1_item_new(ASN1_ITEM_rptr(EXPORTED_PKI_CONF))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_cryptConfs.give_Datas(&(*Datas)->crypt_confs))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->crypt_confs, ASN1_ITEM_rptr(EXPORTED_PKI_CONF));
		(*Datas)->crypt_confs = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->links && !((*Datas)->links = SKM_sk_new_null(ENTITY_LINKS)))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	for(i = 0 ; i < m_links.size() ; i++)
	{
		currlinks = NULL;
		if(!m_links[i].give_Datas(&currlinks))
		{
			ASN1_item_free((ASN1_VALUE*)currlinks, ASN1_ITEM_rptr(ENTITY_LINKS));
			currlinks = NULL;
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(SKM_sk_push(ENTITY_LINKS, (*Datas)->links, currlinks) < 0)
		{
			ASN1_item_free((ASN1_VALUE*)currlinks, ASN1_ITEM_rptr(ENTITY_LINKS));
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
			return false;
		}
	}
	return true;
}

bool PkiConfBeta4::operator=(const PkiConfBeta4 & other)
{
	Clear();
	m_publications = other.m_publications;
	m_repositories = other.m_repositories;
	m_cas = other.m_cas;
	m_conf = other.m_conf;
	m_confs = other.m_confs;
	m_cryptConfs = other.m_cryptConfs;
	m_links = other.m_links;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * PkiConfBeta4::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(PKI_CONF_BETA4);
}
