/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/



// Entity.cpp: implementation of the Entity class.
//
//////////////////////////////////////////////////////////////////////

#include "Entity.h"
#include "svintl.h"
#include "MailInfo.h"


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

Entity::Entity(ENTITY_CONSTRUCTOR_PARAMETERS, 
				LocalEntityConf * PersonnalConf, 
				NewPKIStore * EntityStore,
				unsigned long AsynchMsgsType):
				AsynchMsgs(EntityName, e, AsynchMsgsType)
{
	m_Logging = NULL;
	m_DbConn = NULL;
	entered_client = 0;
	fullInit = false;
	m_Type = Type;
	m_EntityStore = EntityStore;
	m_PersonnalConf = PersonnalConf;

	
	m_Conf = Conf;
	m_AuthModule = authModule;
	m_Engine = e;

	
	
	m_EntityName = EntityName;

	//Doing initial connection to server
	try
	{
		m_DbConn = new SQL_Connection("", m_Conf.get_SqlServer(), m_Conf.get_SqlPort(), m_Conf.get_SqlUsername(), m_Conf.get_SqlPassword());
	}
	catch(ExceptionNewPKI e)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		throw ExceptionNewPKI();
	}
	if(!m_DbConn)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		throw ExceptionNewPKI();
	}
}

Entity::~Entity()
{
	m_Jobs.StopAll();
	if(m_Logging) delete m_Logging;
	if(m_DbConn) delete m_DbConn;
}


bool Entity::EntityExists(bool &Exists)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString db_name;
	long NumRows;
	long i;

	//We select the list of databases
	if(!sql.Execute(SELECT_DATABASE) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	//We verify each of them
	for(i=0; i<NumRows; i++)
	{
		if(!sql.Value(i, SELECT_DATABASE_RES, db_name)) continue;
		
		if(db_name == m_EntityName)
		{
			Exists = true;
			return true;
		}
	}
	Exists = false;
	return true;
}

bool Entity::Common_Upgrade(const char * Version)
{
	//We select the database
	if(!m_DbConn->SelectDatabase(m_EntityName))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!AsynchMsgs::SetDbConn(m_DbConn))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!AsynchMsgs::DoUpgrade(Version))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool Entity::Common_Create(const GenPrivateKey & genkey, const char ** InitialCreates)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	long i;
	char * CommonCreates[] = {COMMON_CREATE_1, COMMON_CREATE_2, COMMON_CREATE_3, COMMON_CREATE_4, NULL};
	bool Exists;
	const char * keyPem;
	mString req;

	//We check if there is another DB with same name
	if(!EntityExists(Exists))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(Exists)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ENTITY_EXISTS);
		return false;
	}

	//We create the database and select it
	if(!sql.CreateDatabase(m_EntityName))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	//We execute each request
	for(i=0; CommonCreates[i]; i++)
	{
		if(!sql.Execute(CommonCreates[i]))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			Destroy();
			return false;
		}
	}

	if(InitialCreates)
	{
		for(i=0; InitialCreates[i]; i++)
		{
			if(!sql.Execute(InitialCreates[i]))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				Destroy();
				return false;
			}
		}
	}

	if(!AsynchMsgs::CreateTables(m_DbConn))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Destroy();
		return false;
	}

	if(!m_Jobs.CreateTables(m_DbConn))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Destroy();
		return false;
	}

	// The store
	if(m_EntityStore && !m_EntityStore->CreateTables(m_DbConn))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Destroy();
		return false;
	}

	//Entity key
	if(genkey)
	{
		switch(genkey.get_type())
		{
			case GEN_PRIVATE_KEY_TYPE_ENGINE:
				keyPem = genkey.get_keyid().c_str();
				if(!m_EntityKey.SetKey(keyPem, m_Engine))
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
					Destroy();
					return false;
				}
			case GEN_PRIVATE_KEY_TYPE_KEYLEN:
				if(!m_EntityKey.GenerateKey(genkey.get_keylen(), m_Engine))
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
					Destroy();
					return false;
				}
				keyPem = m_EntityKey.GetRsaKeyPem().c_str();
				break;
			default:
				NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
				Destroy();
				return false;
				break;
		}
		//Insert key in DB
		if(req.sprintf(ENTITY_SET_CERT, "", keyPem) <= 0)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			Destroy();
			return false;
		}
		if(!sql.Execute(req))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			Destroy();
			return false;
		}
	}

	return true;
}


bool Entity::Common_Init(const PKI_CERT & EntityCert, const EntitySignatureResp & init_datas)
{
	mString key;
	long NumRows;
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;

	// We get the entity key
	if(!sql.Execute(ENTITY_GET_CERT))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!NumRows)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}

	if(!sql.Value(0, "entity_key", key))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!(m_EntityCert = EntityCert))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!m_EntityKey.SetKey(key.c_str(), m_Engine))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!m_EntityCert.SetPrivateKey(m_EntityKey))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	//We now insert the Entity cert in the database
	if(req.sprintf(ENTITY_SET_CERT, m_EntityCert.GetCertPEM().c_str(), key.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(m_EntityStore)
	{
		m_EntityStore->SetEntityCert(m_EntityCert);
		m_EntityStore->set_Jobs(&m_Jobs);
		m_EntityStore->set_Logging(m_Logging);
	}
	AsynchMsgs::SetEntityCert(m_EntityCert);
	AsynchMsgs::set_Logging(m_Logging);
	

	if(m_PersonnalConf && init_datas)
	{
		if(!m_PersonnalConf->set_cas(init_datas.get_cas()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(!ExtractMyConf(init_datas.get_conf()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}

	return true;
}


mString & Entity::GetName()
{
	return m_EntityName;
}

bool Entity::Load_Conf(Asn1EncryptSign * encrypt)
{
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString cert;
	mString key;
	mString pem_conf;
	long NumRows;

	//We select the database
	if(!m_DbConn->SelectDatabase(m_EntityName))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	// We get the entity certificate
	if(!sql.Execute(ENTITY_GET_CERT))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!NumRows)
	{
		return true;
	}

	if(!sql.Value(0, "entity_cert", cert) || !sql.Value(0, "entity_key", key))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	//Did we get our entity certificate yet ?
	if(!cert.size())
	{
		NewpkiDebug(LOG_LEVEL_INFO, m_EntityName.c_str(), _sv("Entity is not initialized"));
		return true;
	}


	if(!m_EntityCert.SetCert(cert.c_str()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!m_EntityKey.SetKey(key.c_str(), m_Engine))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!m_EntityCert.SetPrivateKey(m_EntityKey))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(encrypt)
	{
		// We get the personnal conf
		if(!sql.Execute(ENTITY_GET_CONF))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(!sql.NumRows(&NumRows))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(!NumRows)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			return false;
		}
		if(!sql.Value(0, "conf", pem_conf))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}

		//Convert from PEM
		if(!encrypt->from_PEM(pem_conf))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}

	return true;
}

bool Entity::Common_Load()
{
	Asn1EncryptSign encrypt;
	mVector<PKI_CERT> EntityParentCerts;


	if(!Load_Conf(m_PersonnalConf?&encrypt:NULL))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(m_EntityStore)
	{
		if(!m_EntityStore->SetDbConn(m_DbConn))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}

	if(!AsynchMsgs::SetDbConn(m_DbConn))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!m_EntityCert)
	{
		return true;
	}

	if(m_Logging)
	{
		delete m_Logging;
	}

	try
	{
		m_Logging = new EntityLog(m_DbConn, &m_EntityCert);
	}
	catch(ExceptionNewPKI e)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!m_Logging)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_MALLOC);
		return false;
	}


	m_Logging->SetAuditCalllBack(OnAudit, this);

	if(m_EntityStore)
	{
		m_EntityStore->SetEntityCert(m_EntityCert);
		m_EntityStore->set_Jobs(&m_Jobs);
		m_EntityStore->set_Logging(m_Logging);
	}
	AsynchMsgs::SetEntityCert(m_EntityCert);
	AsynchMsgs::set_Logging(m_Logging);


	const char *EntityMail;
	int Pos;

	Pos = m_EntityCert.GetCertDN().SeekEntryName("emailAddress", -1);
	if(Pos == HASHTABLE_NOT_FOUND)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		ERR_set_error_data(_sv("No emailAddress"), ERR_TXT_STRING);
		return false;
	}

	EntityMail = m_EntityCert.GetCertDN().Get(Pos);
	if(!EntityMail)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		ERR_set_error_data(_sv("No emailAddress"), ERR_TXT_STRING);
		return false;
	}

	m_EntityMail = EntityMail;

	if(!m_Jobs.SetDbConn(m_DbConn))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	m_Jobs.SetEntityName(m_EntityName);
	m_Jobs.SetEntityCert(m_EntityCert);
	m_Jobs.SetLogger(m_Logging);
	m_Jobs.SetX509_ACL_Validator(&AclValidator);

	if(m_PersonnalConf)
	{
		//Decrypt the conf
		if(!m_PersonnalConf->from_SignEncrypt(encrypt, m_EntityKey.GetRsaKey(), m_EntityKey.GetRsaKey()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		EntityParentCerts.push_back(m_PersonnalConf->get_cas().get_rootca());
		EntityParentCerts.push_back(m_PersonnalConf->get_cas().get_entitiesca());

		if(!ParseNewConf())
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(!InitDatas())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	//Start the mailer
	if(!m_Jobs.SetEntityParentsCert(EntityParentCerts))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!m_Jobs.StartMailer())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	// Start the asynch msgs handling
	if(!AsynchMsgs::Run())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	// We have been initialized fully
	fullInit = true;

	return true;
}

void Entity::ClientEnter()
{
	ClientsReadLock.EnterCS();
		entered_client++;
	ClientsReadLock.LeaveCS();
}

void Entity::ClientLeave()
{
	ClientsReadLock.EnterCS();
		entered_client--;
	ClientsReadLock.LeaveCS();
}

void Entity::WaitForNoClients()
{
	ClientsReadLock.EnterCS();
		time_t start_t;
		time_t curr_t;
		time(&start_t);

		while(entered_client)
		{
			NewpkiThread::Sleep(10000);
			time(&curr_t);
			if( (curr_t-start_t) > 1200)
			{
				ClientsReadLock.LeaveCS();
				return;
			}
		}
	ClientsReadLock.LeaveCS();
}

bool Entity::Destroy()
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);

	//We select the list of databases
	if(!sql.DropDatabase(m_EntityName) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

void Entity::ERR_to_ADMIN_RESPONSE(AdminResponseBody & response)
{
	if(!response.set_type(ADMIN_RESP_TYPE_ERRORS))
	{
		return;
	}	
	ERR_to_ERROR_ENTRIES(response.get_errors());
}

bool Entity::ResourceLock(const mString & ResourceName)
{
	ResourcesListLock.EnterCS();
	if(ResourcesList.find(ResourceName) != ResourcesList.end())
	{
		ResourcesListLock.LeaveCS();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_RESOURCE_BUSY);
		return false;
	}
	ResourcesList[ResourceName] = NewpkiThread::CurrentThreadId();
	ResourcesListLock.LeaveCS();
	return true;
}

void Entity::ResourceUnlock(const mString & ResourceName)
{
	std::map<mString, NEWPKI_THREAD_ID>::iterator rsc;
	ResourcesListLock.EnterCS();
	
	// Search resource lock
	rsc = ResourcesList.find(ResourceName);

	// Did we find the resource and am I the owner ?
	if(rsc == ResourcesList.end() ||
		rsc->second != NewpkiThread::CurrentThreadId())
	{
		ResourcesListLock.LeaveCS();
		return;
	}
	ResourcesList.erase(rsc);
	ResourcesListLock.LeaveCS();
}

void Entity::ResourcesClearUser()
{
	std::map<mString, NEWPKI_THREAD_ID>::iterator i;
	NEWPKI_THREAD_ID id = NewpkiThread::CurrentThreadId();

	ResourcesListLock.EnterCS();
LblResourcesClearUser:
	for(i=ResourcesList.begin(); i != ResourcesList.end(); i++)
	{
		if(i->second == id)
		{
			ResourcesList.erase(i);
			goto LblResourcesClearUser;
		}
	}
	ResourcesListLock.LeaveCS();
}

bool Entity::ResourceOwnerIsMe(const mString & ResourceName)
{
	std::map<mString, NEWPKI_THREAD_ID>::iterator rsc;
	ResourcesListLock.EnterCS();
	
	// Search resource lock
	rsc = ResourcesList.find(ResourceName);

	// Did we find the resource and am I the owner ?
	if(rsc == ResourcesList.end() ||
		rsc->second != NewpkiThread::CurrentThreadId())
	{
		ResourcesListLock.LeaveCS();
		return false;
	}
	ResourcesListLock.LeaveCS();
	return true;
}

void Entity::GetEntityCertificate(PKI_CERT & cEntityCert)
{
	cEntityCert = m_EntityCert;
}




void Entity::OnAudit(void *Param, LOG_MESSAGE_STATUS Status, LOG_MESSAGE_TYPE Message, const mString & User, const mString & Object, time_t log_timegmt, const mString & Error)
{
	MailInfo mail;
	mString Body;
	char ret[50];
	struct tm * tmval;
	Entity * me_this = (Entity *)Param;



	tmval = localtime(&log_timegmt);
	sprintf(ret, "%.2d/%.2d/%.2d %.2d:%.2d:%.2d GMT", tmval->tm_mday, tmval->tm_mon + 1, tmval->tm_year+1900, tmval->tm_hour, tmval->tm_min, tmval->tm_sec);


	Body = _sv("Type: ");
	Body += ASN1_logs_get_TypeString(Message);
	Body += _sv("\nStatus: ");
	Body += ASN1_logs_get_StatusString(Status);
	Body += _sv("\nObject: ");
	Body += Object;
	Body += _sv("\nUser: ");
	Body += User;
	Body += _sv("\nDate: ");
	Body += ret;
	if(Status == LOG_STATUS_TYPE_FAILURE && Error.size())
	{
		Body += _sv("\nError: ");
		Body += Error;
	}
	mail.set_Body(Body);
	mail.set_Subject(_sv("Audit Event"));
	mail.set_SignMail(true);

	me_this->m_Jobs.SendMail(me_this->m_EntityCert.GetStringName(), mail, true);
}

void Entity::OnUserLogout()
{
	NEWPKI_THREAD_ID user_id;
	user_id = NewpkiThread::CurrentThreadId();
	// Clear any resources to user might 
	// have forgoten to unlock
	ResourcesClearUser();

	AuthUsersLock.EnterCS();
	LogoutUser(AuthenticatedUsers[user_id]);
	AuthenticatedUsers.erase(user_id);
	AuthUsersLock.LeaveCS();
}

bool Entity::UserLogin(COMMAND_PARAMETERS)
{
	//We add it to the list of logged-in users
	int TypeUser;
	UserHandle m_hUser;
	NEWPKI_THREAD_ID user_id;

	//User is not supposed to be known yet
	if(hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	m_hUser.SetUserName(body.get_login().get_username());
	m_hUser.SetPassword(body.get_login().get_password());

	if(!m_hUser.SetUserCert(UserCert))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!LoginUser(m_hUser, TypeUser))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	user_id = NewpkiThread::CurrentThreadId();
	AuthUsersLock.EnterCS();
	AuthenticatedUsers[user_id] = m_hUser;

	if(!response.set_type(ADMIN_RESP_TYPE_USER_TYPE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		AuthenticatedUsers.erase(user_id);
		AuthUsersLock.LeaveCS();
		return false;
	}

	if(!response.set_usertype(TypeUser))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		AuthenticatedUsers.erase(user_id);
		AuthUsersLock.LeaveCS();
		return false;
	}
	AuthUsersLock.LeaveCS();
	return true;
}

UserHandle & Entity::GetUserHandle()
{
	NEWPKI_THREAD_ID user_id;
	user_id = NewpkiThread::CurrentThreadId();
	AuthUsersLock.EnterCS();
	UserHandle & ret = AuthenticatedUsers[user_id];
	AuthUsersLock.LeaveCS();
	return ret;
}

bool Entity::ExtractMyConf(const EntityConfCrypted & c_conf)
{
	EntityConf tmpConf;

	// Decrypt the conf
	if(m_PersonnalConf)
	{
		if(!tmpConf.from_SignEncrypt(c_conf.get_crypted(), m_PersonnalConf->get_cas().get_pkicert().GetPublicKey(), m_EntityKey.GetRsaKey()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}

		if(m_PersonnalConf->get_conf())
		{
			//Do we need to update the conf in the DB ?
			if(tmpConf.get_version() <= m_PersonnalConf->get_conf().get_version())
			{
				return true;
			}
		}
		if(!m_PersonnalConf->set_conf(tmpConf))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		tmpConf.Clear();
		NewpkiDebug(LOG_LEVEL_INFO, m_EntityName.c_str(), _sv("New personnal conf version %ld"), m_PersonnalConf->get_conf().get_version());

		if(!InitDatas())
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}

		if(!ParseNewConf())
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}

		if(!WritePersonnalConf())
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	return true;
}

bool Entity::InitDatas()
{
	if(m_PersonnalConf)
	{
		/*
		 *	Set the repositories used for the asynch msgs
		 */
		if(!SetRepositories(m_PersonnalConf->get_conf().get_repositories()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}

		if(!m_Jobs.SetRepositories(m_PersonnalConf->get_conf().get_repositories()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
			
		if(!m_Jobs.SetAdminMails(m_PersonnalConf->get_conf().get_acls().get_adminserials()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}

		if(m_Logging)
		{
			//Update the mail conf, we ignore the error
			m_Jobs.SetMailerInfo(m_EntityMail, m_PersonnalConf->get_conf().get_mailConf());
			//Update the audits
			if(!m_Logging->SetAudit(m_PersonnalConf->get_conf().get_audits()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
		}

		if(!AclValidator.SetGroups(m_PersonnalConf->get_conf().get_groups()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}

		if(!AclValidator.SetACL(m_PersonnalConf->get_conf().get_acls()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}

		if(!AclValidator.SetCRL(m_PersonnalConf->get_conf().get_crls()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}

		if(!AclValidator.SetCA(m_PersonnalConf->get_cas()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	return true;
}

bool Entity::IsFullyInit()
{
	return fullInit;
}


bool Entity::WritePersonnalConf()
{
	mString pem_conf;
	Asn1EncryptSign encrypt;

	if(!m_PersonnalConf)
	{
		return true;
	}

	if(!PrepareConfToWrite())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!m_PersonnalConf->to_SignEncrypt(encrypt, m_EntityKey.GetRsaKey(), m_EntityKey.GetRsaKey(), EVP_sha1(), EVP_des_ede3_cbc()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!encrypt.to_PEM(pem_conf))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;

	if(req.sprintf(ENTITY_INSERT_CONF, pem_conf.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}

bool Entity::CheckLogsIntegrity(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!hUser.GetPassword().size() && !AclValidator.CanUserPerform(UserCert, ACL_TYPE_VIEW_LOGS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}	

	if(!m_Logging->CheckLogsIntegrity())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}


bool Entity::AdminSendMail(COMMAND_PARAMETERS)
{
	MailInfo mail;
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!hUser.GetPassword().size() && !AclValidator.CanUserPerform(UserCert, ACL_TYPE_SEND_ADMIN_MAIL))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!m_Jobs.MailerIsUp())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_MAIL_SERVER_NOT_SET);
		return false;
	}

	if(!mail.FromMAIL_DATAS(body.get_adminMail()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!m_Jobs.SendMail(UserCert.GetStringName(), mail, true))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool Entity::GetLogsCount(COMMAND_PARAMETERS)
{
	unsigned long Count;

	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	if(!hUser.GetPassword().size() && !AclValidator.CanUserPerform(UserCert, ACL_TYPE_VIEW_LOGS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!m_Logging->GetLogsCount(Count, body.get_enumLogs()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!response.set_type(ADMIN_RESP_TYPE_LOGS_COUNT))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!response.set_status(Count))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool Entity::EnumLogs(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	if(!hUser.GetPassword().size() && !AclValidator.CanUserPerform(UserCert, ACL_TYPE_VIEW_LOGS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_ENUM_LOGS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!m_Logging->GetLogs(response.get_logs(), body.get_enumLogs()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}

bool Entity::GetLogsType(COMMAND_PARAMETERS)
{
	size_t i;
	
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	if(!hUser.GetPassword().size() && !AclValidator.CanUserPerform(UserCert, ACL_TYPE_VIEW_LOGS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_LOGS_TYPE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	for(i=0; i<LogsType.size(); i++)
	{
		response.get_logsType().push_back(LogsType[i]);
	}

	return true;
}

bool Entity::GetMyACL(COMMAND_PARAMETERS)
{
	if(!hUser || 
		AclValidator.ValidateCert(UserCert) != INTERNAL_CA_TYPE_USER)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_MY_ACL))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	ConfAccessLock.LockRead();
	unsigned char full_acl[] = {(char)0xFF, (char)0xFF, (char)0xFF, (char)0xFF, (char)0xFF, (char)0xFF, (char)0xFF, (char)0xFF, (char)0xFF, (char)0xFF, (char)0xFF, (char)0xFF, (char)0};

	//PKI Administrator
	if(AclValidator.IsPkiAdministrator(UserCert))
	{
		if(ASN1_BIT_STRING_set(response.get_myAcl(), full_acl, strlen((char*)full_acl)) <= 0)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			ConfAccessLock.UnlockRead();
			return false;
		}
	}
	else
	{
		//Get the user's ACL
		if(!AclValidator.GetUserRights(response.get_myAcl(), UserCert.GetSerial()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			ConfAccessLock.UnlockRead();
			return false;
		}
	}
	ConfAccessLock.UnlockRead();

	return true;
}




bool Entity::SignResponse(AdminResponse & response)
{
	ADMIN_RESPONSE * resp;
	if(!response.get_signingCert())
	{
		if(!response.set_signingCert(m_EntityCert))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}

	resp = NULL;
	if(!response.give_Datas(&resp))
	{
		if(resp) ASN1_item_free((ASN1_VALUE*)resp, AdminResponse::get_ASN1_ITEM());
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	//We don't have the signer cert yet
	if(ASN1_item_sign(AdminResponseBody::get_ASN1_ITEM(), resp->sig_algo, NULL, resp->signature, (char*)resp->body, (EVP_PKEY*)m_EntityKey.GetRsaKey(), EVP_sha1()) <= 0)
	{
		ASN1_item_free((ASN1_VALUE*)resp, AdminResponse::get_ASN1_ITEM());
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!response.set_sigAlgo(resp->sig_algo))
	{
		ASN1_item_free((ASN1_VALUE*)resp, AdminResponse::get_ASN1_ITEM());
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!response.set_signature(resp->signature))
	{
		ASN1_item_free((ASN1_VALUE*)resp, AdminResponse::get_ASN1_ITEM());
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	ASN1_item_free((ASN1_VALUE*)resp, AdminResponse::get_ASN1_ITEM());
	return true;
}

int Entity::GetType()
{
	return m_Type;
}

bool Entity::OnNewConf(const EntityConfCrypted & newConf)
{
	ConfAccessLock.LockWrite();
	if(!ExtractMyConf(newConf))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockWrite();
		return false;
	}
	ConfAccessLock.UnlockWrite();
	return true;
}

bool Entity::Responder_TreatRequest(const NewpkiRequest & Request, const mString & SenderName, NewpkiResponse & Response)
{
	return true;
}

bool Entity::Responder_TreatRequestAsynch(const NewpkiRequest & Request, const Asn1OctetString & transactionId, const mString & SenderName)
{
	return true;
}

bool Entity::Responder_ValidateRequest(const NewpkiRequest & Request, const X509_PUBKEY * Requester, mString & SenderName)
{
	return true;
}

bool Entity::Requester_OnNewResponse(const Asn1OctetString & transactionID, const X509_PUBKEY * sender, const NewpkiResponse & Response)
{
	return true;
}

