/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// EeStore.cpp: implementation of the EeStore class.
//
//////////////////////////////////////////////////////////////////////

#include "EeStore.h"
#include "svintl.h"
#include <PKI_PASSWD.h>
#include <openssl/ocsp.h>

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////





EeStore::EeStore(const mString & EntityName, ENGINE * e):NewPKIStore(EntityName, e)
{
}

EeStore::~EeStore()
{
}

bool EeStore::CreateTables(const SQL_Connection * DbConn)
{
	SQL sql(DbConn, SQL_ACCESS_WRITE);
	long i;
	char * CommonCreates[] = {EESTORE_CREATE_1, EESTORE_CREATE_2, EESTORE_CREATE_3, NULL};


	//We execute each request
	for(i=0; CommonCreates[i]; i++)
	{
		if(!sql.Execute(CommonCreates[i]))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	return true;
}

bool EeStore::SetOptions(const mVector<DnSpecs> & DnSpecs,
						 const HashTable_String & Policies,
						 bool ldap_auth,
						 const mString & ldap_server,
						 unsigned long ldap_port,
						 const mString & ldap_username, 
						 const mString & ldap_password,
						 const mString & ldap_base,
						 const mString & ldap_attr_name,
						 const mString & ldap_filters,
						 unsigned long utf8)
{
	ConfAccessLock.EnterCS();

	m_ldap_auth = ldap_auth;

	if(m_ldap_auth)
	{
		m_ldap_filters = ldap_filters;
		m_ldap_attr_name = ldap_attr_name;	
		m_ldap_server = ldap_server;
		m_ldap_port = ldap_port;
		m_DnSpecs = DnSpecs;
		if( !(m_Policies = Policies) )
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			ConfAccessLock.LeaveCS();
			m_ldap_available = false;
			return false;
		}
		if(!m_LdapClient.Connect(ldap_server, ldap_port, ldap_username, ldap_password, ldap_base, ldap_attr_name, utf8))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			ConfAccessLock.LeaveCS();
			m_ldap_available = false;
			return false;
		}
		m_ldap_available = true;
	}
	else
	{
		m_LdapClient.Disconnect();
		m_ldap_available = false;
	}
	ConfAccessLock.LeaveCS();
	return true;
}



bool EeStore::InsertDnValidationRequest(const Asn1OctetString & transactionId, unsigned long ra_id, const X509_NAME * dn)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	mString strDn;
	mString strId;
	time_t startTime;
	MailInfo NoticeMail;
	const char * email;
	HashTable_Dn hDn;
	int pos;
	
	if(!transactionIDtoString(transactionId, strId))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	// Load the DN
	if(!hDn.From_X509_NAME(dn))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if ( (pos = hDn.SeekEntryName("emailAddress", HASHTABLE_NOT_FOUND)) == HASHTABLE_NOT_FOUND ||
		 !(email = hDn.Get(pos)) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_DN_FIELD_UNPRESENT);
		ERR_set_error_data("emailAddress", ERR_TXT_STRING);
		return false;
	}

	if(!X509_NAMEtoString(dn, strDn))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(req.sprintf(EESTORE_INSERT_DN_VAL, strId.c_str(), ra_id, strDn.c_str(), email, time(NULL)) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	// Every 5000 DN we request the table to be optimized
	if( (sql.GetLastID() % 5000) == 0 )
	{
		time(&startTime);
		NewpkiDebug(LOG_LEVEL_INFO, m_EntityName.c_str(), _sv("Optimizing DN validation table..."));
		if(!sql.OptimizeTable(EESTORE_DN_VAL_TABLE))
		{
			req = "";
			ERR_to_mstring(req);
			NewpkiDebug(LOG_LEVEL_WARNING, m_EntityName.c_str(), _sv("Failed to optimize DN validation table - Reason: %s"), req.c_str());
			ERR_clear_error();
		}
		else
		{
			NewpkiDebug(LOG_LEVEL_INFO, m_EntityName.c_str(), _sv("Optimized DN validation table in %ld secondes"), time(NULL) - startTime);
		}
	}

	if(req.sprintf(_sv("In order to generate your certificate, the PKI needs you to validate your personal information.\nPlease logon to \"%svalidate_dn.php?ID=%s\""), 
		m_siteName.c_str(), 
		strId.c_str()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		DeleteDnValidationRequest(transactionId);
		return false;
	}

	NoticeMail.set_MailTo(email);
	NoticeMail.set_Subject(_sv("Personal Information Validation"));
	NoticeMail.set_Body(req);
	NoticeMail.set_SignMail(false);
	if(!SendMail("EE Verification", NoticeMail, false))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		DeleteDnValidationRequest(transactionId);
		return false;
	}

	return true;
}

bool EeStore::DeleteDnValidationRequest(const Asn1OctetString & transactionId)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	mString strTID;

	if(!transactionIDtoString(transactionId, strTID))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(req.sprintf(EESTORE_DEL_DN_VAL, strTID.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

X509_NAME * EeStore::GetDnValidationRequest(const Asn1OctetString & transactionId)
{
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString req;
	mString strTID;
	X509_NAME * dn;
	int len;
	unsigned char * datas;
	unsigned char * p;
	long NumRows;

	if(!transactionIDtoString(transactionId, strTID))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return NULL;
	}

	if(req.sprintf(EESTORE_GET_DN_VAL, strTID.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return NULL;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return NULL;
	}
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!NumRows)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		ERR_add_error_data(1, _sv("Unknown Id"));
		return false;
	}

	if(!sql.Value(0, "dn", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return NULL;
	}

	if( !(datas = StringtoBINARY(req, len)) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return NULL;
	}
	p = datas;
	if( !(dn = d2i_X509_NAME(NULL, &p, len)) )
	{
		free(datas);
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return NULL;
	}
	free(datas);

	return dn;
}

bool EeStore::GetDnValidationRequestRaId(const Asn1OctetString & transactionId, unsigned long & ra_id)
{
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString req;
	mString strTID;
	long NumRows;

	if(!transactionIDtoString(transactionId, strTID))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(req.sprintf(EESTORE_GET_DN_VAL, strTID.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!NumRows)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		ERR_add_error_data(1, _sv("Unknown Id"));
		return false;
	}
	if(!sql.Value(0, "ra_id", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	ra_id = req.c_ulng();

	return true;
}

void EeStore::set_SiteName(const mString &siteName)
{
	if(siteName.size())
	{
		m_siteName = siteName;
		if(m_siteName[m_siteName.size() - 1] != '/')
			m_siteName += "/";
	}
	else
		m_siteName = "http://localhost/";

}

X509_NAME * EeStore::WebUserGetUserDn(unsigned long user_id)
{
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString req;
	long NumRows;
	X509_NAME * dn;


	if(req.sprintf(EESTORE_GET_USER, user_id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return NULL;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return NULL;
	}
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return NULL;
	}
	if(!NumRows)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_UNKNOWN_USER);
		return NULL;
	}

	if(!sql.Value(0, "dn", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return NULL;
	}
	dn = StringtoX509_NAME(req);
	if(!dn)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return NULL;
	}
	return dn;
}

bool EeStore::WebUserGetUserId(const mString & otp, unsigned long & user_id)
{
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString req;
	long NumRows;

	if(req.sprintf(EESTORE_GET_USER_OTP, otp.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!NumRows)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_UNKNOWN_USER);
		return false;
	}

	if(!sql.Value(0, "user_id", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	user_id = req.c_ulng();

	return true;
}

bool EeStore::WebUserActivate(const mString & otp)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;

	if(!otp.size())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}
	if(req.sprintf(EESTORE_ACTIVATE_USER, otp.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}

bool EeStore::WebUserDeactivate(const mString & otp)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;

	if(!otp.size())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}
	if(req.sprintf(EESTORE_DEACTIVATE_USER, otp.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}

bool EeStore::WebUserRaValidated(unsigned long user_id)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;

	if(req.sprintf(EESTORE_RA_VALIDATED_USER, user_id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool EeStore::WebUserDelete(unsigned long user_id)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;

	if(!user_id)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}
	if(req.sprintf(EESTORE_DELETE_USER, user_id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool EeStore::WebUserGetEmail(unsigned long user_id, mString & email)
{
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString req;
	long NumRows;

	if(req.sprintf(EESTORE_GET_USER, user_id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!NumRows)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_UNKNOWN_USER);
		return false;
	}

	if(!sql.Value(0, "email", email))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool EeStore::WebUserLdapLogin(const mString & email, const mString & password, LdapResult & Result)
{
	mVector<LdapResult> Results;
	mString Filters;
	ClientLDAP LdapClient;
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;

	ConfAccessLock.EnterCS();
	if(!m_ldap_available && !m_LdapClient.Reconnect())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_UNKNOWN_USER);
		ERR_add_error_data(1, _sv("LDAP server not available, please contact your system administrator."));
		ConfAccessLock.LeaveCS();
		return false;
	}

	// Search user
	if(Filters.sprintf("(&%s(%s=%s))", 
						m_ldap_filters.c_str(),
						m_ldap_attr_name.c_str(),
						email.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		ConfAccessLock.LeaveCS();
		return false;
	}

	if(!m_LdapClient.Search(Filters, Results))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.LeaveCS();
		return false;
	}
	ConfAccessLock.LeaveCS();

	if(Results.size() == 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_UNKNOWN_USER);
		return false;
	}
	if(Results.size() > 1)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_UNKNOWN_USER);
		ERR_add_error_data(1, _sv("More than one user has this email address, please contact your administrator"));
		return false;
	}

	ConfAccessLock.EnterCS();
	// Try to bind as the user
	if(!LdapClient.Connect(m_ldap_server, 
							m_ldap_port,
							Results[0].get_rdn(),
							password,
							mString::EmptyInstance,
							mString::EmptyInstance,
							0))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_UNKNOWN_USER);
		ConfAccessLock.LeaveCS();
		return false;
	}
	ConfAccessLock.LeaveCS();
	LdapClient.Disconnect();

	Result = Results[0];

	return true;
}

unsigned long EeStore::WebUserLogin(const mString & email, const mString & password, unsigned long & created_user_id)
{
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString req;
	long NumRows;
	PKI_PASSWD pki_password;
	LdapResult lResult;

	if(!email.size() || !password.size())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return 0;
	}

	if(m_ldap_auth && !WebUserLdapLogin(email, password, lResult))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return 0;
	}

	if(req.sprintf(EESTORE_LOGIN_USER, email.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return 0;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return 0;
	}
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return 0;
	}
	if(!NumRows)
	{
		if(!m_ldap_auth)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_UNKNOWN_USER);
			return 0;
		}

		// User doesn't exists, we must create it
		created_user_id = WebUserLdapCreate(email, lResult);
		if(!created_user_id)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return 0;
		}
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		ERR_add_error_data(1, _sv("Account hasn't been validated by RA"));
		return 0;
	}
	if(!m_ldap_auth)
	{
		if(!sql.Value(0, "password", req))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return 0;
		}
		if(!pki_password.VerifySHA1Password(password, req))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_UNKNOWN_USER);
			return 0;
		}
		if(!sql.Value(0, "activated", req))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return 0;
		}
		if(!req.c_int())
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
			ERR_add_error_data(1, _sv("Account isn't active yet"));
			return 0;
		}
	}
	if(!sql.Value(0, "ra_validated", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return 0;
	}
	if(!req.c_int())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		ERR_add_error_data(1, _sv("Account hasn't been validated by RA"));
		return 0;
	}
	if(!sql.Value(0, "user_id", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return 0;
	}
	return req.c_ulng();
}

bool EeStore::ProfileExists(const mString & dn_hash, bool & exists, unsigned long & UserId)
{
	mString req;
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	long NumRows;
	mString strpid;

	if(UserId)
	{
		if(strpid.sprintf(" AND user_id != '%ld'", UserId) <= 0)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			return false;
		}
	}

	if(req.sprintf(EESTORE_SEARCH_USER_DN, dn_hash.c_str(), strpid.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(NumRows)
	{
		if(!sql.Value(0, "user_id", req))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		UserId = req.c_ulng();
		exists = true;
	}
	else
		exists = false;

	return true;
}


unsigned long EeStore::WebUserInsert(const mString & email, const mString & password, const X509_NAME * dn, const mString & otp)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	PKI_PASSWD pwd;
	char * hpwd;
	mString strDn;
	mString DnString;
	mString strHash;
	unsigned long user_id;
	bool Exists;

	// Search if the profile is unique
	if(!X509_NAMEtoHash(dn, strHash))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	user_id = 0;
	if(!ProfileExists(strHash, Exists, user_id))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(Exists)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_UNIQUE_PROFILE);
		if(strHash.sprintf("%ld", user_id) > 0)
			ERR_add_error_data(1, strHash.c_str());
		return false;
	}
	hpwd = X509_NAME_oneline((X509_NAME*)dn, NULL, 0);
	if(!hpwd)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	DnString = sql.FormatString(hpwd);
	free(hpwd);


	
	if(password.size())
	{
		hpwd = pwd.CalcSHA1Password(password);
		if(!hpwd)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN);
			return 0;
		}
	}
	else
	{
		hpwd = "";
	}

	if(!X509_NAMEtoString(dn, strDn))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return 0;
	}

	if(req.sprintf(EESTORE_INSERT_USER, email.c_str(), hpwd, 
										strDn.c_str(),
										DnString.c_str(),
										strHash.c_str(), 
										otp.c_str(), time(NULL)) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return 0;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return 0;
	}
	user_id = sql.GetLastID();

	return user_id;
}


unsigned long EeStore::WebUserLdapCreate(const mString & email, const LdapResult & Result)
{
	HashTable_Dn Dn;
	HashTable_Dn IndexDn;
	size_t i;
	long j;
	unsigned long Min;
	unsigned long Max;
	size_t ValueLen;
	const char * Value;
	X509_NAME * dn;
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	unsigned long user_id;

	// We first generate a DN from the LDAP entry
	for(i=0; i<Result.get_objects().size(); i++)
	{
		Dn.Add(Result.get_objects()[i].get_name().c_str(), 
				Result.get_objects()[i].get_value().c_str());
	}

	for(i=0; i<m_DnSpecs.size(); i++)
	{
		j = Dn.SeekEntryName(m_DnSpecs[i].get_name().c_str(), HASHTABLE_NOT_FOUND);
		if(j != HASHTABLE_NOT_FOUND)
		{
			Value = Dn.Get(j);
			if(!Value)
			{
				Value = m_DnSpecs[i].get_default().c_str();
				ValueLen = m_DnSpecs[i].get_default().size();
			}
			else
			{
				ValueLen = strlen(Value);
			}
		}
		else
		{
			Value = m_DnSpecs[i].get_default().c_str();
			ValueLen = m_DnSpecs[i].get_default().size();
		}

		if(Value && ValueLen)
		{
			Min = m_DnSpecs[i].get_min();
			Max = m_DnSpecs[i].get_max();

			// Does it respect the length contraint ?
			if(!Min || ValueLen >= Min)
			{
				if(!Max || ValueLen <= Max)
				{
					IndexDn.Add(m_DnSpecs[i].get_name().c_str(), Value);
				}
			}
			if(j != HASHTABLE_NOT_FOUND)
				Dn.Delete(j);
		}
	}

	if(!IndexDn.ValidateAgainstPolicy(m_Policies))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return 0;
	}

	dn = X509_NAME_new();
	if(!dn)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!IndexDn.To_X509_NAME(dn))
	{
		X509_NAME_free(dn);
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return 0;
	}
	// Now insert user in the DB
	user_id = WebUserInsert(email, "", dn, "");
	if(!user_id)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		X509_NAME_free(dn);
		return 0;
	}
	X509_NAME_free(dn);

	// User is created, since we know it really exists
	// we can activate it right away
	if(req.sprintf(EESTORE_ACTIVATE_USER_ID, user_id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		WebUserDelete(user_id);
		return 0;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		WebUserDelete(user_id);
		return 0;
	}
	return user_id;
}

unsigned long EeStore::WebUserCreate(const mString & email, const mString & password, const X509_NAME * dn)
{
	unsigned char rnds[SHA_DIGEST_LENGTH];
	unsigned char rndv[250];
	mString otp;
	unsigned long user_id;
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;

	if(!email.size() || !password.size() || !dn)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return 0;
	}

	// Generate a one time password
	RAND_bytes(rndv, sizeof(rndv));	
	if(!SHA1(rndv, sizeof(rndv), rnds))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return 0;
	}
	if(!BINARYtoString(rnds, sizeof(rnds), otp))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return 0;
	}

	user_id = WebUserInsert(email, password, dn, otp);
	if(!user_id)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return 0;
	}

	if(!SendUserMailNotification(email, otp))
	{
		WebUserDelete(user_id);
		return 0;
	}
	return user_id;
}

bool EeStore::WebUserSetStatusByRaId(unsigned long ra_id, int status, const PKI_CRL & lastCrl)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	long NumRows;
	mString req;
	mString ca_name;
	unsigned long CertReqId;

	if(req.sprintf(EESTORE_GET_CERT_BY_RA_ID, ra_id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(NumRows)
	{
		if(!sql.Value(0, "ca_name", ca_name))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(!sql.Value(0, "id", req))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		CertReqId = req.c_ulng();

		// Set status
		if(status == NEWPKI_PROFILE_CERT_STATE_REVOKED)
		{
			if(req.sprintf(EESTORE_SET_CERT_RESP, 0, NEWPKI_PROFILE_CERT_STATE_REVOKED, "", "", 
								"", "", 0, CertReqId) <= 0)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
				return 0;
			}
		}
		else
		{
			if(req.sprintf(EESTORE_SET_CERT_STATUS_RA_ID, status, ra_id) <= 0)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
				return 0;
			}
		}
		if(!sql.Execute(req))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		UpdateFromCRL(ca_name, lastCrl);
	}	
	return true;
}

bool EeStore::WebUserImportCertResponse(unsigned long CertReqId, const NewpkiEeCertResponse & cert_response, const mVector< ErrorEntry > & errors, unsigned long & user_id)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	int Status;
	mString req;
	mString strError;
	mString x509;
	mString p7b;
	mString p12;
	unsigned long serial = 0;
	unsigned long ra_id = 0;
	NewpkiProfileDatasCert cert;

	if(!WebUserGetCertReq(CertReqId, cert, user_id))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(errors.size())
	{
		Status = NEWPKI_PROFILE_CERT_STATE_ERROR;
		ERROR_ENTRIES_to_string(errors, strError);
	}
	else
	{
		if(cert_response.get_status() == CERT_RESPONSE_OK)
		{
			if(cert.get_state() != NEWPKI_PROFILE_CERT_STATE_WAITING)
			{
				return true;
			}
			Status = NEWPKI_PROFILE_CERT_STATE_ACTIVE;

			x509 = cert_response.get_certificate().GetCertPEM();
			serial = cert_response.get_certificate().GetSerial();
			ra_id = cert_response.get_raId();
			p7b = cert_response.get_p7b().GetPemP7B();
			if(cert.get_type() == NEWPKI_PROFILE_CERT_TYPE_PKCS12)
			{
				p12 = cert_response.get_p12().GetPemPKCS12();
			}
		}
		else if(cert_response.get_status() == CERT_RESPONSE_ERROR)
		{
			Status = NEWPKI_PROFILE_CERT_STATE_ERROR;
			ERROR_ENTRIES_to_string(cert_response.get_errors(), strError);
		}
		else
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			return false;
		}
		if(cert_response.get_lastCrl())
		{
			if(!UpdateFromCRL(cert.get_caName(), cert_response.get_lastCrl()))
				ERR_clear_error();
		}
	}

	strError = sql.FormatString(strError);
	if(req.sprintf(EESTORE_SET_CERT_RESP, ra_id, Status, x509.c_str(), p7b.c_str(), 
						strError.c_str(), p12.c_str(), serial, CertReqId) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return 0;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return 0;
	}
	return true;
}

bool EeStore::WebUserImportRevResponse(unsigned long CertReqId, const NewpkiEeRevResponse & rev_response, const mVector< ErrorEntry > & errors, unsigned long & user_id)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	long NumRows;
	int Status;
	mString strError;
	NewpkiProfileDatasCert cert;

	// Get the certificate
	if(req.sprintf(EESTORE_GET_CERT, CertReqId) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!NumRows)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	if(!sql.Value(0,"ra_id", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	// Make sure we're talking about the same ID
	if(req.c_ulng() != rev_response.get_raId())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!Sql2Cert(&sql, cert, 0))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(errors.size())
	{
		Status = NEWPKI_PROFILE_CERT_STATE_ACTIVE;
		ERROR_ENTRIES_to_string(errors, strError);
	}
	else 
	{
		if(cert.get_state() != NEWPKI_PROFILE_CERT_STATE_WAITING_REV)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_CERT_STATE);
			return false;
		}
		if(rev_response.get_status() == REV_RESPONSE_ERROR)
		{
			Status = CaStatus2StoreStatus(rev_response.get_certStatus(), 
											NEWPKI_PROFILE_CERT_STATE_ACTIVE);
		}
		else if(rev_response.get_status() == CERT_RESPONSE_OK)
		{
			Status = NEWPKI_PROFILE_CERT_STATE_REVOKED;
		}
		else
		{
			Status = NEWPKI_PROFILE_CERT_STATE_ACTIVE;
		}
		if(rev_response.get_lastCrl())
		{
			if(!UpdateFromCRL(cert.get_caName(), rev_response.get_lastCrl()))
				ERR_clear_error();
		}
	}

	if(req.sprintf(EESTORE_SET_CERT_RESP, 0, Status, "", "", 
						"", "", 0, CertReqId) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return 0;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return 0;
	}
	return true;
}

bool EeStore::WebUserInsertCertReq(const NewpkiEeRequestCert & request, unsigned long & CertReqId)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	time_t startTime;

	if(req.sprintf(EESTORE_INSERT_CERT, request.get_profileId(), 
										request.get_caName().c_str(), 
										request.get_type(), 
										NEWPKI_PROFILE_CERT_STATE_WAITING) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	CertReqId = sql.GetLastID();
	if(CertReqId == (unsigned long)-1)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	// Every 5000 certificates we request the table to be optimized
	if( (CertReqId % 5000) == 0 )
	{
		time(&startTime);
		NewpkiDebug(LOG_LEVEL_INFO, m_EntityName.c_str(), _sv("Optimizing certificates table..."));
		if(!sql.OptimizeTable(EESTORE_CERTS_TABLE))
		{
			req = "";
			ERR_to_mstring(req);
			NewpkiDebug(LOG_LEVEL_WARNING, m_EntityName.c_str(), _sv("Failed to optimize certificates table - Reason: %s"), req.c_str());
			ERR_clear_error();
		}
		else
		{
			NewpkiDebug(LOG_LEVEL_INFO, m_EntityName.c_str(), _sv("Optimized certificates table in %ld secondes"), time(NULL) - startTime);
		}
	}
	return true;
}

bool EeStore::WebUserMarkCertRev(unsigned long user_id, unsigned long cert_id, unsigned long & ra_id)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	long NumRows;
	mString req;


	// First make sure user is the right owner
	if(req.sprintf(EESTORE_GET_CERT, cert_id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!NumRows)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	if(!sql.Value(0, "user_id", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(req.c_ulng() != user_id)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	if(!sql.Value(0, "ra_id", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	ra_id = req.c_ulng();
	if(!sql.Value(0, "status", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(req.c_int() != NEWPKI_PROFILE_CERT_STATE_ACTIVE &&
		req.c_int() != NEWPKI_PROFILE_CERT_STATE_SUSPENDED)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_CERT_STATE);
		return false;
	}
	

	// Mark cert as waiting for revocation
	if(req.sprintf(EESTORE_SET_CERT_STATUS, NEWPKI_PROFILE_CERT_STATE_WAITING_REV, cert_id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool EeStore::WebUserUnmarkCertRev(unsigned long cert_id)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;

	// Unmark cert as waiting for revocation
	if(req.sprintf(EESTORE_SET_CERT_STATUS, NEWPKI_PROFILE_CERT_STATE_ACTIVE, cert_id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool EeStore::WebUserDeleteCertReq(unsigned long CertReqId)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	
	if(req.sprintf(EESTORE_DELETE_CERT, CertReqId) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool EeStore::WebUserGetCertReq(unsigned long cert_id, NewpkiProfileDatasCert & Cert, unsigned long & user_id)
{
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString req;
	long NumRows;
	
	if(req.sprintf(EESTORE_GET_CERT, cert_id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!NumRows)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	if(!sql.Value(0,"user_id", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	user_id = req.c_ulng();
	if(!user_id)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}	
	if(!Sql2Cert(&sql, Cert, 0))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool EeStore::WebUserEnumCerts(unsigned long user_id, mVector<NewpkiProfileDatasCert> & Certs)
{
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString req;
	long NumRows;
	long i;
	
	if(req.sprintf(EESTORE_GET_CERTS, user_id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	for(i=0; i<NumRows; i++)
	{
		Certs.insert(Certs.begin() + i);
		ERR_clear_error();
		if(!Sql2Cert(&sql, Certs[i], i) || !Certs[i])
		{
			Certs[i].set_id(0);
			Certs[i].set_flags(0);
			Certs[i].set_state(NEWPKI_PROFILE_CERT_STATE_ERROR);
			Certs[i].set_type(NEWPKI_PROFILE_CERT_TYPE_PKCS12);
			Certs[i].set_caName(_sv("Unknown"));
			ERR_to_mstring(Certs[i].get_error());
		}
		Certs[i].set_isOK();
	}
	return true;
}

bool EeStore::WebUserChangePassword(const WebuserChangePasswd & chgPasswd, bool check_password)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	long NumRows;
	PKI_PASSWD pki_password;
	char * hpwd;

	if(check_password)
	{
		if(req.sprintf(EESTORE_GET_USER, chgPasswd.get_userId()) <= 0)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			return false;
		}
		if(!sql.Execute(req))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(!sql.NumRows(&NumRows))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(!NumRows)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_UNKNOWN_USER);
			return false;
		}
		if(!sql.Value(0, "password", req))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(!pki_password.VerifySHA1Password(chgPasswd.get_oldPassword(), req))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_UNKNOWN_USER);
			return false;
		}
	}
	hpwd = pki_password.CalcSHA1Password(chgPasswd.get_password());
	if(!hpwd)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(req.sprintf(EESTORE_SET_PASSWORD, hpwd, chgPasswd.get_userId()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

X509_NAME * EeStore::WebUserGetDn(unsigned long user_id)
{
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString req;
	long NumRows;
	X509_NAME * dn;


	if(req.sprintf(EESTORE_GET_USER, user_id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return NULL;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return NULL;
	}
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return NULL;
	}
	if(!NumRows)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_UNKNOWN_USER);
		return NULL;
	}

	if(!sql.Value(0, "dn", req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return NULL;
	}
	dn = StringtoX509_NAME(req);
	if(!dn)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return NULL;
	}
	return dn;
}


bool EeStore::Sql2Cert(SQL *sql, NewpkiProfileDatasCert & Cert, int index)
{
	mString value;

	Cert.Clear();

	if(!sql->Value(index,"id", value))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	Cert.set_id(value.c_ulng());

	if(!sql->Value(index,"ca_name", value))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	Cert.set_caName(value);

	if(!sql->Value(index,"status", value))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	Cert.set_state(value.c_ulng());


	if(!sql->Value(index,"type", value))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	Cert.set_type(value.c_ulng());

	// Are we in state of having the PKCS#12 ?
	if(Cert.get_state() == NEWPKI_PROFILE_CERT_STATE_ACTIVE && 
		Cert.get_type() == NEWPKI_PROFILE_CERT_TYPE_PKCS12)
	{
		// Do we have the PKCS#12 ?
		if(!sql->Value(index,"p12", value))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(value.size())
		{
			if(!Cert.get_p12().Load(value, NULL))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
		}
	}
	
	
	if(!sql->Value(index,"x509", value))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(value.size())
	{
		if(!Cert.get_cert().SetCert(value.c_str()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}

	if(!sql->Value(index,"p7b", value))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(value.size())
	{
		if(!Cert.get_p7b().Load(value.c_str()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}

	if(!sql->Value(index,"error", value))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(value.size())
		Cert.set_error(value);

	Cert.set_isOK();
	return true;
}

bool EeStore::UpdateFromCRL(const mString & ca_name, const PKI_CRL &Crl)
{
	mString req;
	size_t i;
	const REVOCATION_INFO * currRev;
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString supsList;
	mString tmpString;

	if(!Crl.RevokedCertsCount())
	{
		return true;
	}

	// Update the certificates that were unsuspended directly on the CA
	for(i=0; i<Crl.RevokedCertsCount(); i++)
	{
		currRev = Crl.GetRevokedCert(i);
		if(!currRev)
			continue;
		if(currRev->reason != OCSP_REVOKED_STATUS_CERTIFICATEHOLD)
			continue;
		if(supsList.size())
		{
			if(tmpString.sprintf(", %ld", currRev->serial) <= 0)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
				return false;
			}
		}
		else
		{
			if(tmpString.sprintf("%ld", currRev->serial) <= 0)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
				return false;
			}
		}

		supsList += tmpString;
	}
	if(supsList.size())
	{
		if(req.sprintf(EESTORE_UPDATE_FROM_CRL_UNSUSP, NEWPKI_PROFILE_CERT_STATE_ACTIVE, 
						ca_name.c_str(), NEWPKI_PROFILE_CERT_STATE_SUSPENDED, 
						supsList.c_str()) <= 0)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			return false;
		}
		if(!sql.Execute(req))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	else
	{
		// There is no suspended cert, we mark as active all suspended certs
		if(req.sprintf(EESTORE_UPDATE_NO_SUSP, NEWPKI_PROFILE_CERT_STATE_ACTIVE, 
						ca_name.c_str(), NEWPKI_PROFILE_CERT_STATE_SUSPENDED) <= 0)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			return false;
		}
		if(!sql.Execute(req))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}

	// Update the certificates that were revoked or suspended
	for(i=0; i<Crl.RevokedCertsCount(); i++)
	{
		currRev = Crl.GetRevokedCert(i);
		if(!currRev)
			continue;

		if(currRev->reason != OCSP_REVOKED_STATUS_CERTIFICATEHOLD)
		{
			if(req.sprintf(EESTORE_UPDATE_FROM_CRL_REV, NEWPKI_PROFILE_CERT_STATE_REVOKED, 
														ca_name.c_str(), 
														currRev->serial) <= 0)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
				return false;
			}
		}
		else
		{
			if(req.sprintf(EESTORE_UPDATE_FROM_CRL_SUSP, NEWPKI_PROFILE_CERT_STATE_SUSPENDED, 
														ca_name.c_str(), 
														currRev->serial) <= 0)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
				return false;
			}
		}
		if(!sql.Execute(req))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	return true;
}







bool EeStore::SendUserMailNotification(const mString & email, const mString & otp, bool deleteNotification)
{
	mString body;
	MailInfo NoticeMail;

	if(!deleteNotification)
	{
		if(body.sprintf(_sv("Please logon to \"%sactivate_account.php?ID=%s\" to activate your account"), 
			m_siteName.c_str(), 
			otp.c_str()) <= 0)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			return false;
		}
	}
	else
	{
		body = _sv("Since you failed to activate your account in time it has been deleted");
	}

	NoticeMail.set_MailTo(email);
	NoticeMail.set_Subject(!deleteNotification ? _sv("Account Activation") : _sv("Account Deletion"));
	NoticeMail.set_Body(body);
	NoticeMail.set_SignMail(false);
	if(!SendMail("EE Verification", NoticeMail, false))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	return true;
}

bool EeStore::SendUserMailNotifications(time_t t, int scount)
{
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	SQL sql2(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	long NumRows;
	long i;
	mString email;
	mString otp;
	unsigned long user_id;

	if(req.sprintf(EESTORE_GET_USERS_MAIL, t, scount) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!NumRows)
	{
		return true;
	}
	for(i=0; i<NumRows; i++)
	{
		if(!sql.Value(i, "email", email))
			continue;
		if(!sql.Value(i, "otp", otp))
			continue;
		if(!sql.Value(i, "user_id", req))
			continue;
		user_id = req.c_ulng();

		if(scount < 3)
		{
			if(!SendUserMailNotification(email, otp))
				continue;
		}
		else
		{
			if(!SendUserMailNotification(email, otp, true))
				continue;
		}
		if(scount < 3)
		{
			// Update the email count
			if(req.sprintf(EESTORE_UPDATE_USERS_MAIL, time(NULL), scount+1, user_id) > 0)
			{
				sql2.Execute(req);
			}
		}
		else
		{
			WebUserDelete(user_id);
		}
	}
	return true;
}
void EeStore::ResendNotificationMails()
{
	time_t currTime;

	time(&currTime);

	/*
	 *	Every 3 days we resend the notification email,
	 *  after 9 days we remove the account
	 */
	SendUserMailNotifications(currTime - 259200, 1);
	SendUserMailNotifications(currTime - 259200, 2);
	SendUserMailNotifications(currTime - 259200, 3);
}

bool EeStore::WebUserPublishCert(const NewpkiEeCertPublish &Request)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	mString p7b;
	mString p12;

	if(!Request.get_certificate())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}

	if(Request.get_p7b())
		p7b = Request.get_p7b().GetPemP7B();
	if(Request.get_p12())
		p12 = Request.get_p12().GetPemPKCS12();

	if(req.sprintf(EESTORE_INSERT_CERT_ALL, Request.get_raId(), Request.get_profileId(),
											Request.get_caName().c_str(), (int)Request.get_type(), 
											NEWPKI_PROFILE_CERT_STATE_ACTIVE,
											Request.get_certificate().GetSerial(), 
											p12.c_str(), 
											Request.get_certificate().GetCertPEM().c_str(),
											p7b.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(Request.get_lastCrl())
	{
		if(!UpdateFromCRL(Request.get_caName(), Request.get_lastCrl()))
			ERR_clear_error();
	}
	return true;
}

bool EeStore::EnumWebUsers(mVector<NewpkiEeUser> & Users, 
						long index, long num, long state, const mString & filter)
{
	long NumRows;
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString req;
	mString Where;
	mString tmpDatas;
	size_t v_index, i;

	switch(state)
	{
		case ENUM_EE_USERS_VALID:
			if(!Where.size())
				Where = "WHERE ";
			else
				Where += " AND ";
			Where += "ra_validated = '1' AND activated = '1'";
			break;
		case ENUM_EE_USERS_NOT_RA_VALIDATED:
			if(!Where.size())
				Where = "WHERE ";
			else
				Where += " AND ";
			Where += "ra_validated = '0'";
			break;
		case ENUM_EE_USERS_NOT_ACTIVE:
			if(!Where.size())
				Where = "WHERE ";
			else
				Where += " AND ";
			Where += "activated = '0'";
			break;
		default:
			break;
	}

	if(filter.size())
	{
		if(!Where.size())
			Where = "WHERE ";
		else
			Where += " AND ";
		Where += "LOWER(dn_string) LIKE '%%";
		Where += sql.FormatString(filter).c_str();
		Where += "%%'";
	}

	if(req.sprintf(EESTORE_GET_USERS, Where.c_str(), index, num) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	v_index = 0;
	for(i=0; (long)i<NumRows; i++)
	{
		Users.insert(Users.begin()+v_index);
		
		if(!SqlToNewpkiEeProfile(sql, i, Users[v_index]))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		//Get the certs associated with the user !
		if(!WebUserEnumCerts(Users[v_index].get_id(), Users[v_index].get_certs()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		Users[v_index].set_isOK();
		v_index++;
	}
	
	return true;
}

bool EeStore::SqlToNewpkiEeProfile(SQL & sql, long pos, NewpkiEeUser & Profile)
{
	mString tmpDatas;
	X509_NAME * dn;

	// The profile id
	if(!sql.Value(pos, "user_id", tmpDatas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	Profile.set_id(tmpDatas.c_ulng());

	// The email
	if(!sql.Value(pos, "email", tmpDatas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!Profile.set_email(tmpDatas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	// The DN
	if(!sql.Value(pos, "dn", tmpDatas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	dn = StringtoX509_NAME(tmpDatas);
	if(!dn)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!Profile.set_dn(dn))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		X509_NAME_free(dn);
		return false;
	}
	X509_NAME_free(dn);

	// The ra_validated flag
	if(!sql.Value(pos, "ra_validated", tmpDatas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	Profile.set_raValidated(tmpDatas.c_ulng());


	// The activated flag
	if(!sql.Value(pos, "activated", tmpDatas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	Profile.set_activated(tmpDatas.c_ulng());


	// The last_send
	if(!sql.Value(pos, "last_send", tmpDatas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	Profile.set_lastSend(tmpDatas.c_ulng());


	// The send_count
	if(!sql.Value(pos, "send_count", tmpDatas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	Profile.set_sendCount(tmpDatas.c_ulng());

	Profile.set_isOK();
	return true;
}

bool EeStore::DeletePKCS12(unsigned long id)
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;

	if(req.sprintf(EESTORE_DELETE_PKCS12, id) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}

