/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// PkiUserCreation.cpp: implementation of the PkiUserCreation class.
//
//////////////////////////////////////////////////////////////////////

#include "PkiUserCreation.h"
#include "Error.h"
#include "clintl.h"
#include <PKI_P7B.h>
#include <PKI_CERT.h>

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

PkiUserCreation::PkiUserCreation()
{
	m_pubkey = NULL;
	m_type = PKI_USER_KEY_TYPE_SOFTKEY;
}

PkiUserCreation::~PkiUserCreation()
{
	if(m_pubkey)
		X509_PUBKEY_free(m_pubkey);
}

void PkiUserCreation::set_Info(const mString &cn, const mString &email, int PkiAdmin)
{
	m_cn = cn;
	m_email = email;
	m_admin = PkiAdmin;
}

void PkiUserCreation::set_SoftwareKey(int keylen, const mString &password)
{
	m_keylen = keylen;
	m_password = password;
	m_type = PKI_USER_KEY_TYPE_SOFTKEY;
	m_internaltype = REQ_INTERNAL_TYPE_P12;
}

void PkiUserCreation::set_HardwareKey(int keylen, const mString &provider)
{
	m_keylen = keylen;
	m_provider = provider;
	m_type = PKI_USER_KEY_TYPE_PUBKEY;
	m_internaltype = REQ_INTERNAL_TYPE_SC;
}

void PkiUserCreation::set_PublicKey(const X509_PUBKEY * pubkey)
{
	if(m_pubkey && m_internaltype != REQ_INTERNAL_TYPE_P7)
	{
		X509_PUBKEY_free(m_pubkey);
		m_pubkey = NULL;
	}
	m_pubkey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (void*)pubkey);
	m_type = PKI_USER_KEY_TYPE_PUBKEY;
	m_internaltype = REQ_INTERNAL_TYPE_P7;
}

bool PkiUserCreation::generate_Request(CreatePkiUserRequest & request)
{
	X509_PUBKEY * pubkey;

	if(m_pubkey && m_internaltype != REQ_INTERNAL_TYPE_P7)
	{
		X509_PUBKEY_free(m_pubkey);
		m_pubkey = NULL;
	}

	if(!m_cn.size() || !m_email.size())
	{
		HandleError(NEWPKIerrGetStr(ERROR_BAD_PARAM), m_parent);
		return false;
	}
	if(!request.get_ukey().set_type(m_type))
	{
		HandleErrorResult(NULL, m_parent, 0);
		return false;
	}
	if(!request.set_cn(m_cn))
	{
		HandleError(NEWPKIerrGetStr(ERROR_MALLOC), m_parent);
		return false;
	}
	if(!request.set_email(m_email))
	{
		HandleError(NEWPKIerrGetStr(ERROR_MALLOC), m_parent);
		return false;
	}
	request.set_pkiadmin(m_admin);

	switch(m_internaltype)
	{
		case REQ_INTERNAL_TYPE_P12:
			if(!request.get_ukey().get_softkey().set_password(m_password))
			{
				HandleError(NEWPKIerrGetStr(ERROR_MALLOC), m_parent);
				return false;
			}
			request.get_ukey().get_softkey().set_keylen(m_keylen);
			break;
		case REQ_INTERNAL_TYPE_SC:
			#ifndef _WIN32
			HandleError(NEWPKIerrGetStr(ERROR_NOT_ALLOWED), m_parent);
			return false;
			#else
			pubkey = m_enroll.GeneratePrivateKey(m_provider.c_str(), m_keylen);
			if(!pubkey)
			{
				HandleErrorResult(NULL, m_parent, m_enroll.GetEnrollLastError());
				return false;
			}
			if(!request.get_ukey().set_pubkey(pubkey))
			{
				HandleErrorResult(NULL, m_parent, 0);
				X509_PUBKEY_free(pubkey);
				return false;
			}
			m_pubkey = pubkey;
			#endif
			break;

		case REQ_INTERNAL_TYPE_P7:
			if(!m_pubkey)
			{
				HandleError(NEWPKIerrGetStr(ERROR_MALLOC), m_parent);
				return false;
			}
			if(!request.get_ukey().set_pubkey(m_pubkey))
			{
				HandleErrorResult(NULL, m_parent, 0);
				return false;
			}
			break;

		default:
			HandleError(NEWPKIerrGetStr(ERROR_BAD_DATAS), m_parent);
			return false;
	}

	return true;
}

void PkiUserCreation::Cancel()
{
#ifdef _WIN32
	m_enroll.DestroyContainer();
#endif
}

bool PkiUserCreation::import_Response(const CreatePkiUserResponse & response)
{
	if(m_type != response.get_type())
	{
		HandleError(NEWPKIerrGetStr(ERROR_BAD_DATAS), m_parent);
		return false;
	}

	switch(m_internaltype)
	{
		case REQ_INTERNAL_TYPE_P12:
			return import_PKCS12(response.get_p12().GetPKCS12());
			break;
		case REQ_INTERNAL_TYPE_P7:
			return import_PKCS7(response.get_p7b().GetPKCS7());
			break;
		case REQ_INTERNAL_TYPE_SC:
			return import_SC(response.get_p7b());
			break;
		default:
			HandleError(NEWPKIerrGetStr(ERROR_BAD_DATAS), m_parent);
			return false;
	}
}

bool PkiUserCreation::import_PKCS12(const PKCS12 *p12)
{
	FILE * fd;

	wxFileDialog dialog(m_parent, _("Save PKCS#12"), "", m_cn.c_str(), _("PKCS#12 File (*.p12)|*.p12"), wxSAVE|wxOVERWRITE_PROMPT);
	if (dialog.ShowModal() != wxID_OK)
	{
		return false;
	}
	fd = fopen(dialog.GetPath(), "wb");
	if(!fd)
	{
		HandleError(strerror(errno), m_parent);
		return false;
	}

	if(i2d_PKCS12_fp(fd, (PKCS12*)p12) <= 0)
	{
		fclose(fd);
		HandleErrorResult(NULL, m_parent, 0);
		return false;
	}
	fclose(fd);

	return true;
}

bool PkiUserCreation::import_PKCS7(const PKCS7 *p7b)
{
	FILE * fd;

	wxFileDialog dialog(m_parent, _("Save PKCS#7"), "", m_cn.c_str(), _("PKCS#7 File (*.p7)|*.p7"), wxSAVE|wxOVERWRITE_PROMPT);
	if (dialog.ShowModal() != wxID_OK)
	{
		return false;
	}
	fd = fopen(dialog.GetPath(), "wb");
	if(!fd)
	{
		HandleError(strerror(errno), m_parent);
		return false;
	}

	if(i2d_PKCS7_fp(fd, (PKCS7*)p7b) <= 0)
	{
		fclose(fd);
		HandleErrorResult(NULL, m_parent, 0);
		return false;
	}
	fclose(fd);

	return true;
}

bool PkiUserCreation::import_SC(const PKI_P7B & p7b)
{
#ifndef _WIN32
	HandleError(NEWPKIerrGetStr(ERROR_NOT_ALLOWED), m_parent);
	return false;
#else
	int i;
	X509 * currCert;
	PKI_CERT Cert;

	if(!m_pubkey)
	{
		HandleError(NEWPKIerrGetStr(ERROR_BAD_PARAM), m_parent);
		return false;
	}

	for(i=0; i<p7b.GetNumCert(); i++)
	{
		currCert = p7b.GetCert(i);
		if(!currCert)
		{
			HandleError(NEWPKIerrGetStr(ERROR_UNKNOWN), m_parent);
			return false;
		}
		if(!Cert.SetCert(currCert))
		{
			HandleErrorResult(NULL, m_parent, 0);
			return false;
		}

		if(Cert == m_pubkey)
		{
			if(!m_enroll.ImportCert(Cert.GetCertPEM().c_str()))
			{
				HandleErrorResult(NULL, m_parent, m_enroll.GetEnrollLastError());
				return false;
			}
			return true;
		}
	}

	HandleError(NEWPKIerrGetStr(ERROR_BAD_DATAS), m_parent);
	return false;
#endif
}

void PkiUserCreation::LoadProviders(wxComboBox *combo, const char * Default)
{
#ifdef _WIN32
		LPTSTR      pszName;
		DWORD       dwType;
		DWORD       cbName;
		DWORD       dwIndex=0;
		int Index = 0;
		int i;

		dwIndex = 0;
		i = 0;
		while(CryptEnumProviders(dwIndex, NULL, 0, &dwType, NULL, &cbName))
		{

			if ((pszName = (LPTSTR)malloc(cbName)))
			{
				if (CryptEnumProviders(
									   dwIndex++,
									   NULL,
									   NULL,
									   &dwType,   
									   pszName,
									   &cbName))     
				{
					if(Default && strcmp(Default, pszName) == 0)
						Index = i;

					combo->Append(pszName, (void*)NULL);
				}
				free(pszName);
			}

			i++;
		}
	
		combo->SetSelection(Index);
#else

		combo->Enable(FALSE);
		combo->Append(_("MS Windows only"));
		combo->SetSelection(0);
#endif

}

void PkiUserCreation::set_Parent(wxWindow *parent)
{
	m_parent = parent;
}

