/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// DlgPublishDN.cpp: implementation of the DlgPublishDN class.
//
//////////////////////////////////////////////////////////////////////

#include "DlgPublishDN.h"
#include "dlgs_wdr.h"
#include "clintl.h"


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

BEGIN_EVENT_TABLE(DlgPublishDNListCtrl, wxListCtrl)
    EVT_LIST_ITEM_SELECTED(IDC_LIST_FIELDS, DlgPublishDNListCtrl::OnItemClick)
END_EVENT_TABLE()

BEGIN_EVENT_TABLE(DlgPublishDN, wxDialog)
    EVT_BUTTON(IDC_SEARCH_LDAP, DlgPublishDN::OnSearchLDAP)
    EVT_BUTTON(IDC_OK, DlgPublishDN::OnOk)
    EVT_BUTTON(IDC_CANCEL, DlgPublishDN::OnCancel)
	EVT_TEXT(IDC_FIELD_VALUE, DlgPublishDN::OnSave)
END_EVENT_TABLE()

void DlgPublishDNListCtrl::OnItemClick(wxListEvent& event)
{
	DlgPublishDN * wParent = (DlgPublishDN *)GetParent();
	wParent->OnItemClick(event);
}

DlgPublishDN::DlgPublishDN(wxWindow * wParent, const EntityConfBody & EntityConf, ASN1_BIT_STRING * my_acl, PkiClient * ClientPki, const X509_NAME * dn, unsigned long profile_id):
		wxDialog(wParent, (wxWindowID)-1, dn?_("Modify a profile's DN"):_("Create a profile"), wxDefaultPosition),
		m_EntityConf(EntityConf)
{
	size_t i;

	m_ClientPki = ClientPki;
	Msg = NULL;
	m_imageListSmall = NULL;
	m_IsOk = false;

	m_dn = dn;
	m_profile_id = profile_id;


	DlgPublishDN_SetWindow(this);

	Msg = new DlgMessage(this);
	if(!Msg)
	{
		HandleErrorResult(NULL, this, 0);
		Close(TRUE);
		return;
	}

	if(!dn)
	{
		Msg->wShow(_("Loading Groups..."));
		if(!m_ClientPki->GetGroups(m_groups))
		{
			Msg->wHide();
			HandleError(m_ClientPki->GetError(), this);
			Close(TRUE);
			return;
		}
		Msg->wHide();

		((wxComboBox *)FindWindow(IDC_GROUPS))->Append(_("None"), (void*)0);

		if(!ASN1_BIT_STRING_get_bit(my_acl, ACL_TYPE_RA_DELEGATE_PROFILE_OWNERSHIP))
		{
			((wxComboBox *)FindWindow(IDC_GROUPS))->Enable(FALSE);
		}
		else
		{
			for(i=0; i<m_groups.size(); i++)
			{
				((wxComboBox *)FindWindow(IDC_GROUPS))->Append(m_groups[i].get_name().c_str(), (void*)m_groups[i].get_serial());
			}
		}
		((wxComboBox *)FindWindow(IDC_GROUPS))->SetSelection(0);
	}
	else
	{
		((wxComboBox *)FindWindow(IDC_GROUPS))->Append(_("None"), (void*)0);
		((wxComboBox *)FindWindow(IDC_GROUPS))->SetSelection(0);
		((wxCheckBox *)FindWindow(IDC_EE_VALIDATION))->Enable(FALSE);
		((wxComboBox *)FindWindow(IDC_GROUPS))->Enable(FALSE);
		((wxTextCtrl *)FindWindow(IDC_UID))->Enable(FALSE);
		((wxStaticText *)FindWindow(IDC_LBL_UID))->Enable(FALSE);
		((wxStaticText *)FindWindow(IDC_LBL_OWNER))->Enable(FALSE);
	}
	
	m_listCtrl = ((DlgPublishDNListCtrl *)FindWindow(IDC_LIST_FIELDS));


	//Initialization of the listview
	m_imageListSmall = new wxImageList(16, 16, TRUE);
	wxIcon ico;
	ico.CopyFromBitmap(DlgPublishDN_GetBitmap(0));
	m_imageListSmall->Add( ico );

	m_listCtrl->SetImageList(m_imageListSmall, wxIMAGE_LIST_SMALL);
	m_listCtrl->InsertColumn(0, _("Name"), wxLIST_FORMAT_LEFT, 150);
	m_listCtrl->InsertColumn(1, _("Value"), wxLIST_FORMAT_LEFT, 140);
	m_listCtrl->InsertColumn(2, _("Comment"), wxLIST_FORMAT_LEFT, 140);

	InitList(dn);

	m_listCtrl->SetItemState(0, wxLIST_STATE_SELECTED, wxLIST_STATE_SELECTED);
	wxListEvent event;
	OnItemClick(event);
	

	CenterOnScreen();
	ShowModal();
}

DlgPublishDN::~DlgPublishDN()
{
	ClearList();
	if(Msg) delete Msg;
	if(m_imageListSmall) delete m_imageListSmall;
}

void DlgPublishDN::OnItemClick(wxListEvent& event)
{
	FIELD_INFO * CurrentField;
	long SelectedItem;

	SelectedItem = m_listCtrl->GetNextItem(-1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
	if(SelectedItem == -1) return;
	CurrentField = (FIELD_INFO *)m_listCtrl->GetItemData(SelectedItem);
	if(!CurrentField) return;

	((wxTextCtrl *)FindWindow(IDC_FIELD_VALUE))->SetValue(CurrentField->Value.c_str());
	((wxTextCtrl *)FindWindow(IDC_FIELD_VALUE))->SetFocus();
}

void DlgPublishDN::OnSave(wxCommandEvent &event)
{
	FIELD_INFO * CurrentField;
	long SelectedItem;

	SelectedItem = m_listCtrl->GetNextItem(-1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
	if(SelectedItem == -1) return;
	CurrentField = (FIELD_INFO *)m_listCtrl->GetItemData(SelectedItem);
	if(!CurrentField) return;

	CurrentField->Value = ((wxTextCtrl *)FindWindow(IDC_FIELD_VALUE))->GetValue().GetData();

	m_listCtrl->SetItem(SelectedItem, 1, CurrentField->Value.c_str());
}

void DlgPublishDN::OnOk(wxCommandEvent &event)
{
	NewpkiProfile newProfile;
	FIELD_INFO * CurrentField;
	int nItem;
	int ItemCount;
	wxListEvent levent;
	const char * ObjectName;
	int nid;
	int SelectedItem;
	unsigned long OwnerGroup;
	wxString Uid;


	if(!m_dn)
	{
		SelectedItem = ((wxComboBox *)FindWindow(IDC_GROUPS))->GetSelection();
		if(SelectedItem == -1) return;
		OwnerGroup = (unsigned long)((wxComboBox *)FindWindow(IDC_GROUPS))->GetClientData(SelectedItem);

		newProfile.set_ownerGroupSerial(OwnerGroup);

		if(((wxCheckBox *)FindWindow(IDC_EE_VALIDATION))->GetValue())
		{
			newProfile.set_eeValidation(1);
		}
		else
		{
			newProfile.set_eeValidation(0);
		}

		// The LDAP RDN for publication
		Uid = ((wxTextCtrl *)FindWindow(IDC_UID))->GetValue();
		if(Uid.Len())
			newProfile.set_ldapUid(Uid.GetData());

	}


	ItemCount = m_listCtrl->GetItemCount();
	for(nItem=0; nItem<ItemCount; nItem++)
	{
		CurrentField = (FIELD_INFO *)m_listCtrl->GetItemData(nItem);
		if(!CurrentField) continue;

		if(CurrentField->Value.size())
		{
			ObjectName = FormatObject(CurrentField->Specs.get_name().c_str());
			if(!IsValidObject(ObjectName))
				continue;
			nid = OBJ_txt2nid(ObjectName);
			if(nid == NID_undef)
				continue;
			
			if(CurrentField->Specs.get_min() && 
				CurrentField->Value.size() < CurrentField->Specs.get_min())
			{
				m_listCtrl->SetItemState(nItem, wxLIST_STATE_SELECTED, wxLIST_STATE_SELECTED);
				OnItemClick(levent);
				HandleError(_("The selected fields doesn't respect the length constraint"), this);
				return;
			}
			if(CurrentField->Specs.get_max() && 
				CurrentField->Value.size() > CurrentField->Specs.get_max())
			{
				m_listCtrl->SetItemState(nItem, wxLIST_STATE_SELECTED, wxLIST_STATE_SELECTED);
				OnItemClick(levent);
				HandleError(_("The selected fields doesn't respect the length constraint"), this);
				return;
			}
			
			if(!X509_NAME_add_entry_by_NID(newProfile.get_dn(), nid, MBSTRING_ASC, (unsigned char*)CurrentField->Value.c_str(), -1, -1, 0))
			{
				m_listCtrl->SetItemState(nItem, wxLIST_STATE_SELECTED, wxLIST_STATE_SELECTED);
				OnItemClick(levent);
				HandleError(NEWPKIerrGetStr(ERROR_UNKNOWN), this);
				return;
			}
		}
	}


	unsigned long ProfileId;

	if(!m_dn)
	{
		Msg->wShow(_("Importing Profile..."));
		if(!m_ClientPki->ImportProfile(newProfile, ProfileId))
		{
			Msg->wHide();
			HandleError(m_ClientPki->GetError(), this);
			return;
		}
	}
	else
	{
		Msg->wShow(_("Changing Profile's DN..."));
		if(!m_ClientPki->ChangeProfileDN(m_profile_id, newProfile.get_dn()))
		{
			Msg->wHide();
			HandleError(m_ClientPki->GetError(), this);
			return;
		}
	}
	Msg->wHide();

	m_IsOk = true;
	Close(TRUE);
}

bool DlgPublishDN::IsOk()
{
	return m_IsOk;
}

void DlgPublishDN::OnSearchLDAP(wxCommandEvent& event)
{
	HashTable_Dn Dn;
	int i;
	int j;
	const char * name;
	const char * value;
	FIELD_INFO * CurrentField;


	DlgSearchLDAP Dlg(this, m_ClientPki);
	if(!Dlg.IsOK())
		return;

	Dn = Dlg.GetDN();

	InitList(m_dn);

	// Do some matching
	for(i=0; i<m_listCtrl->GetItemCount(); i++)
	{
		CurrentField = (FIELD_INFO *)m_listCtrl->GetItemData(i);
		if(!CurrentField) continue;

		for(j=0; j<Dn.EntriesCount(); j++)
		{
			name = Dn.GetName(j);
			if(!name) continue;

			if(strcmp(FormatObject(CurrentField->Specs.get_name().c_str()), name) == 0)
			{
				value = Dn.Get(j);
				if(!value) continue;
				
				CurrentField->Value = value;
					continue;

				m_listCtrl->SetItem(i, 1, value);
				Dn.Delete(j);
				break;
			}
		}
	}
	
	// Are we allowed to have some fields that are not 
	// present in the DN specs ?
	if(!ASN1_BIT_STRING_get_bit((ASN1_BIT_STRING*)m_EntityConf.get_raConf().RA_CONF_PTR.get_flags(), RA_REMOVE_UNKNOWN_DN_FIELDS))
	{
		for(j=0; j<Dn.EntriesCount(); j++)
		{
			name = Dn.GetName(j);
			if(!name) continue;

			value = Dn.Get(j);
			if(!value) continue;


			CurrentField = new FIELD_INFO();
			if(!CurrentField) continue;


			CurrentField->Specs.set_name(name);
			CurrentField->Specs.set_min(0);
			CurrentField->Specs.set_max(0);
			CurrentField->Value = value;

			i = m_listCtrl->GetItemCount();
			m_listCtrl->InsertItem(i, name, 0);
			m_listCtrl->SetItemData(i, (long)CurrentField);
			m_listCtrl->SetItem(i, 1, value);
		}
	}

	((wxTextCtrl *)FindWindow(IDC_UID))->SetValue(Dlg.GetUID().c_str());

	m_listCtrl->SetItemState(0, wxLIST_STATE_SELECTED, wxLIST_STATE_SELECTED);
	wxListEvent levent;
	OnItemClick(levent);
}



void DlgPublishDN::InitList(const X509_NAME * dn)
{
	size_t i;
	long pos;
	FIELD_INFO * CurrentField;
	HashTable_Dn lDn;
	const char * Value;

	lDn.From_X509_NAME(dn);

	ClearList();

	for(i=0; i<m_EntityConf.get_raConf().RA_CONF_PTR.get_dnspecs().size(); i++)
	{
		CurrentField = new FIELD_INFO();
		if(!CurrentField) continue;

		CurrentField->Specs = m_EntityConf.get_raConf().RA_CONF_PTR.get_dnspecs()[i];

		pos = lDn.SeekEntryName(CurrentField->Specs.get_name().c_str(), HASHTABLE_NOT_FOUND);
		if(pos != HASHTABLE_NOT_FOUND && (Value = lDn.Get(pos)))
		{
			CurrentField->Value = Value;
		}
		else if(m_EntityConf.get_raConf().RA_CONF_PTR.get_dnspecs()[i].get_default().size())
		{
			CurrentField->Value = m_EntityConf.get_raConf().RA_CONF_PTR.get_dnspecs()[i].get_default();
		}
		
		m_listCtrl->InsertItem(i, CurrentField->Specs.get_name().c_str(), 0);
		m_listCtrl->SetItemData(i, (long)CurrentField);
		m_listCtrl->SetItem(i, 1, CurrentField->Value.c_str());
		m_listCtrl->SetItem(i, 2, CurrentField->Specs.get_comment().c_str());
	}
}

void DlgPublishDN::ClearList()
{
	FIELD_INFO * CurrentField;
	int nItem;
	int ItemCount;

	ItemCount = m_listCtrl->GetItemCount();

	for(nItem=0; nItem<ItemCount; nItem++)
	{
		CurrentField = (FIELD_INFO *)m_listCtrl->GetItemData(nItem);
		if(!CurrentField) continue;
		delete CurrentField;
	}
	m_listCtrl->DeleteAllItems();
}

void DlgPublishDN::OnCancel(wxCommandEvent &event)
{
	m_IsOk=false;
	Close(TRUE);
}

