/*  Neutrino:  neutrino-window-menus.c
 *
 *  Copyright (C) 2002 David A Knight <david@ritter.demon.co.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>

#include <glib/gi18n.h>

#include <gtk/gtk.h>

#include <gconf/gconf-client.h>

#include <string.h>

#include "neutrino-shell.h"
#include "neutrino-window.h"
#include "neutrino-window-private.h"

static void add_widget( GtkUIManager *merge, GtkWidget *widget,
			NeutrinoWindow *window )
{
	if( GTK_IS_MENU_SHELL( widget ) ) {
		gtk_box_pack_start( GTK_BOX( window->content_hbox ),
				    widget, FALSE, TRUE, 0 );
		gtk_box_reorder_child( GTK_BOX( window->content_hbox ),
					widget, 0 );
	} else if( GTK_IS_TOOLBAR( widget ) ) {
		guint toolbars;

		toolbars = GPOINTER_TO_UINT( g_object_get_data( G_OBJECT( merge ), "toolbars" ) );
		++ toolbars;

		gtk_toolbar_set_show_arrow( GTK_TOOLBAR( widget ),
						TRUE );
		gtk_toolbar_set_tooltips( GTK_TOOLBAR( widget ),
						TRUE );
		gtk_box_pack_start( GTK_BOX( window->content_hbox ),
				widget, FALSE, TRUE, 0 );
		gtk_box_reorder_child( GTK_BOX( window->content_hbox ),
					widget,
					toolbars );
		g_object_set_data( G_OBJECT( merge ), "toolbars",
				GUINT_TO_POINTER( toolbars ) );
	}
}

static void
file_menu_close_window_callback( GtkAction *action,
				 gpointer user_data )
{
	neutrino_window_close( NEUTRINO_WINDOW( user_data ) );
}

static void
file_menu_exit_callback( GtkAction *action, gpointer user_data )
{
	NeutrinoShell *app;
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( user_data );
	app = window->shell;

	gdk_threads_leave();
	neutrino_shell_close_all_windows( app );
	gdk_threads_enter();
}


static void
edit_menu_eax_callback( GtkAction *action, gpointer user_data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( user_data );

	neutrino_window_edit_eax( window );
}

static void
edit_menu_preferences_callback( GtkAction *action, gpointer user_data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( user_data );

	neutrino_edit_preferences( window );
}

static void
edit_menu_selectall_callback( GtkAction *action, gpointer user_data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( user_data );

	neutrino_window_select_all( window );
}

static void
view_menu_sidebar_callback( GtkAction *action, gpointer user_data )
{	
	GConfClient *client;
	gboolean state;

	state = gtk_toggle_action_get_active( GTK_TOGGLE_ACTION( action ) );

	client = gconf_client_get_default();

	gconf_client_set_bool( client, "/apps/neutrino/ui/show_sidebar",
			       state, NULL );
	g_object_unref( client );
}

static void
view_menu_browser_callback( GtkAction *action, gpointer user_data )
{	
	GConfClient *client;
	gboolean state;

	state = gtk_toggle_action_get_active( GTK_TOGGLE_ACTION( action ) );

	client = gconf_client_get_default();

	gconf_client_set_bool( client, "/apps/neutrino/ui/show_browser",
			       state, NULL );
	g_object_unref( client );
}

static void
view_menu_playback_callback( GtkAction *action, gpointer user_data )
{	
	GConfClient *client;
	gboolean state;

	state = gtk_toggle_action_get_active( GTK_TOGGLE_ACTION( action ) );

	client = gconf_client_get_default();

	gconf_client_set_bool( client, "/apps/neutrino/ui/show_playback",
			       state, NULL );
	g_object_unref( client );
}

static void
view_menu_toolbar_callback( GtkAction *action, gpointer user_data )
{	
	GConfClient *client;
	gboolean state;

	state = gtk_toggle_action_get_active( GTK_TOGGLE_ACTION( action ) );

	client = gconf_client_get_default();

	gconf_client_set_bool( client, "/apps/neutrino/ui/show_toolbar",
			       state, NULL );
	g_object_unref( client );
}

static void
help_menu_about_neutrino_callback( GtkAction *action,
				   gpointer user_data )
{
	static GtkWidget *about = NULL;

	const gchar *COPYRIGHT = "(C) 2002-2005 David A Knight, parts (C) 2002 Linus Walleij";
	const gchar *AUTHORS[] = {
		"David A Knight <david@ritter.demon.co.uk>",
		"Linus Walleij <triad@df.lth.se> - GNOMAD author",
		NULL
	};
	const gchar *DESCRIPTION = "\
Neutrino is a GNOME shell for managing your Creative Nomad/D.A.P Jukebox\n The jukebox code was derived from that of GNOMAD, a GTK+ shell with similar goals.\n\nNeutrino is licensed under the GNU General Public License version 2 or above";
        GdkPixbuf *logo = NULL;
	const gchar *DOCUMENTERS[] = {
		NULL
	};
	const gchar *TRANSLATERS = _( "translater credits" );

	if( ! about ) {
		logo = NULL;
		about = gtk_about_dialog_new();

		gtk_about_dialog_set_name( GTK_ABOUT_DIALOG( about ),
				"Neutrino" );
		gtk_about_dialog_set_version( GTK_ABOUT_DIALOG( about ),
				VERSION );
		gtk_about_dialog_set_copyright( GTK_ABOUT_DIALOG( about ),
				COPYRIGHT );
		gtk_about_dialog_set_comments( GTK_ABOUT_DIALOG( about ),
				DESCRIPTION );
		gtk_about_dialog_set_authors( GTK_ABOUT_DIALOG( about ),
				AUTHORS );
		gtk_about_dialog_set_documenters( GTK_ABOUT_DIALOG( about ),
				DOCUMENTERS );
		if( strcmp( TRANSLATERS, "translater credits" ) ) {
			gtk_about_dialog_set_translator_credits( GTK_ABOUT_DIALOG( about ),
					TRANSLATERS );
		}
		
		gtk_about_dialog_set_website( GTK_ABOUT_DIALOG( about ),
				"http://neutrino.sourceforge.net" );

		g_signal_connect_swapped( G_OBJECT( about ), "delete_event",
					  G_CALLBACK( g_nullify_pointer ),
					  &about );
        	g_signal_connect_swapped( G_OBJECT( about ), "destroy",
                	                  G_CALLBACK( g_nullify_pointer ),
					  &about );
        }

	gtk_window_set_transient_for( GTK_WINDOW( about ),
				      GTK_WINDOW( user_data ) );
	gtk_widget_show( about );
	gdk_window_raise( GDK_WINDOW( about->window ) );
}

static void commands_upload_callback( GtkAction *action,
				    gpointer user_data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( user_data );

	neutrino_window_upload( window );
}

static void commands_download_callback( GtkAction *action,
					gpointer user_data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( user_data );

	neutrino_window_download( window );
}

static void commands_cancel_callback( GtkAction *action,
				      gpointer user_data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( user_data );

	neutrino_window_cancel( window );
}


static void commands_delete_callback( GtkAction *action,
				      gpointer user_data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( user_data );

	neutrino_window_delete( window );
}

static void commands_scan_callback( GtkAction *action,
				    gpointer user_data )
{
	NeutrinoWindow *window;
	NeutrinoShell *shell;

	window = NEUTRINO_WINDOW( user_data );
	shell = window->shell;

	gdk_threads_leave();
	neutrino_shell_scan_jukeboxes( shell );
	gdk_threads_enter();
}

static void commands_edit_meta_callback( GtkAction *action,
					 gpointer user_data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( user_data );

	neutrino_window_edit_meta( window );
}

static void commands_add_to_playlist_callback( GtkAction *action,
					       gpointer user_data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( user_data );

	neutrino_window_add_to_playlist( window );
}

static void commands_new_playlist_callback( GtkAction *action,
					    gpointer user_data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( user_data );

	neutrino_window_new_playlist( window );
}

static void commands_delete_playlist_callback( GtkAction *action,
					       gpointer user_data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( user_data );

	neutrino_window_delete_playlist( window );
}

static void commands_rename_playlist_callback( GtkAction *action,
					       gpointer user_data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( user_data );

	neutrino_window_rename_playlist( window );
}

static void commands_jukebox_info_callback( GtkAction *action,
					    gpointer user_data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( user_data );

	neutrino_window_jukebox_info( window );
}

static void commands_playback_callback( GtkAction *action,
					gpointer user_data )
{
	neutrino_window_playback( NEUTRINO_WINDOW( user_data ) );
}

static void commands_previous_callback( GtkAction *action,
					gpointer user_data )
{
	neutrino_window_play_prev( NEUTRINO_WINDOW( user_data ) );
}

static void commands_stop_callback( GtkAction *action,
				    gpointer user_data )
{
	neutrino_window_play_close( NEUTRINO_WINDOW( user_data ) );
}

static void commands_next_callback( GtkAction *action,
				    gpointer user_data )
{
	neutrino_window_play_next( NEUTRINO_WINDOW( user_data ) );
}

static void commands_shuffle_callback( GtkAction *action,
				       gpointer user_data )
{
	gboolean state;

	state = gtk_toggle_action_get_active( GTK_TOGGLE_ACTION( action ) );
	g_object_set_data( G_OBJECT( user_data ), "shuffle",
			GINT_TO_POINTER( state ) );
}

static void commands_repeat_callback( GtkAction *action,
				       gpointer user_data )
{
	gboolean state;

	state = gtk_toggle_action_get_active( GTK_TOGGLE_ACTION( action ) );
	g_object_set_data( G_OBJECT( user_data ), "repeat",
			GINT_TO_POINTER( state ) );

}

static void commands_queue_callback( GtkAction *action,
				     gpointer user_data )
{
	neutrino_window_queue( NEUTRINO_WINDOW( user_data ) );
}


static GtkActionEntry neutrino_window_action_entries[] = {

  {"File", NULL, N_("_File"), NULL, NULL, NULL},
  {"Edit", NULL, N_("_Edit"), NULL, NULL, NULL},
  {"View", NULL, N_("_View"), NULL, NULL, NULL},
  {"Music", NULL, N_("_Music"), NULL, NULL, NULL},
  {"Playlists", NULL, N_("_Playlists"), NULL, NULL, NULL},
  {"Help", NULL, N_("_Help"), NULL, NULL, NULL},
  {"Customize Interface", NULL, N_("_Customize Interface" ), NULL, NULL, NULL },
  {"tracks", NULL, N_("tracks"), NULL, NULL, NULL },


  {"Upload", GTK_STOCK_GO_FORWARD, N_("Transfer _To"), "",
   N_("Transfer file(s) to jukebox"), 
   G_CALLBACK (commands_upload_callback) },

  {"Download", GTK_STOCK_GO_BACK, N_("Transfer _From"), "",
   N_("Transfer file(s) from jukebox"), 
   G_CALLBACK (commands_download_callback) },

  {"Cancel", GTK_STOCK_STOP, N_("_Cancel Transfer"), "",
   N_("Cancel transfer of files"), 
   G_CALLBACK (commands_cancel_callback) },

  {"Quit", GTK_STOCK_QUIT, N_( "_Quit" ), "<control>q",
   N_( "Quit Neutrino" ), G_CALLBACK (file_menu_exit_callback) },

  {"Preferences", GTK_STOCK_PREFERENCES, N_( "Preferences" ), "",
   N_( "Edit Preferences" ),
   G_CALLBACK (edit_menu_preferences_callback) },

  {"Delete", GTK_STOCK_DELETE, N_("_Delete"), "",
   N_("Delete track or playlist from jukebox"), 
   G_CALLBACK (commands_delete_callback) },

  {"Scan", GTK_STOCK_REFRESH, N_("Scan for Jukeboxes"), "",
   N_("(Re)Scan for attached Jukeboxes"), 
   G_CALLBACK (commands_scan_callback) },

  {"EditMeta", GTK_STOCK_PROPERTIES, N_("_Edit Track Properties"), "",
   N_
   ("Edit the information for the track(s), e.g. artist, title, album etc."),
   G_CALLBACK (commands_edit_meta_callback) },

  {"NewPlaylist", GTK_STOCK_NEW, N_("_New Playlist..."), "",
   N_("Create a new playlist"), 
   G_CALLBACK (commands_new_playlist_callback) },

  {"AddToPlaylist", GTK_STOCK_ADD, N_("_Add to Playlist"), "",
   N_("Add the selected tracks / albums to a playlist"), 
   G_CALLBACK (commands_add_to_playlist_callback) },

  {"DeletePlaylist", GTK_STOCK_DELETE, N_("_Delete Playlist"), "",
   N_("Delete the selected playlist"), 
   G_CALLBACK (commands_delete_playlist_callback) },

  {"RenamePlaylist", NULL, N_("_Rename Playlist..."), "",
   N_("Rename the selected playlist"), 
   G_CALLBACK (commands_rename_playlist_callback) },

  {"JukeboxInfo", GTK_STOCK_DIALOG_INFO, N_("Jukebox Info"), "",
   N_("Show Jukebox Information"), 
   G_CALLBACK (commands_jukebox_info_callback) },

  {"EAX", GTK_STOCK_PROPERTIES, N_("E_AX"), "",
   N_("Edit EAX Settings"), G_CALLBACK (edit_menu_eax_callback) },

  {"playback", GTK_STOCK_MEDIA_PLAY, N_("Play"), "",
   N_("Playback selected tracks / playlists"), 
   G_CALLBACK (commands_playback_callback) },

  {"stop", GTK_STOCK_MEDIA_STOP, N_("Stop"), "",
   N_("Stop Playback"), G_CALLBACK (commands_stop_callback) },

  {"prev", GTK_STOCK_MEDIA_PREVIOUS, N_("Previous"), "",
   N_("Play previous track"), G_CALLBACK (commands_previous_callback)},

  {"next", GTK_STOCK_MEDIA_NEXT, N_("Next"), "",
   N_("Play next track"), G_CALLBACK (commands_next_callback) },

  {"Queue", NULL, N_("Queue"), "",
   N_("Queue track(s) for playback"), 
   G_CALLBACK (commands_queue_callback) },

  {"Selectall", NULL, N_("Select All"), "<Control>A",
   N_("Select All tracks in the track list"), 
   G_CALLBACK (edit_menu_selectall_callback) },

  {"About", GTK_STOCK_ABOUT, N_("About Neutrino"), "",
  N_( "Display credits for the creators of Neutrino" ),
  G_CALLBACK (help_menu_about_neutrino_callback) }
};

static GtkToggleActionEntry neutrino_window_toggle_action_entries[] = {
  {"Shuffle", "Neutrino_Shuffle", N_("Shuffle"), "",
   N_("Shuffle Mode"), 
   G_CALLBACK (commands_shuffle_callback) },

  {"Repeat", "Neutrino_Repeat", N_("Repeat"), "",
   N_("Repeat in non Shuffle Mode"), 
   G_CALLBACK (commands_repeat_callback) },

  {"ViewSidebar", NULL, N_("View Sidebar"), "",
   N_("Show/Hide the Sidebar"), 
   G_CALLBACK (view_menu_sidebar_callback) },

  {"ViewBrowser", NULL, N_("View Browser"), "",
   N_("Show/Hide the Browser"), 
   G_CALLBACK (view_menu_browser_callback) },

  {"ViewPlayback", NULL, N_("View Playback"), "",
   N_("Show/Hide the Playback display"), 
   G_CALLBACK (view_menu_playback_callback) },

  {"ViewToolbar", NULL, N_("View Toolbar"), "",
   N_("Show/Hide the Toolbar"), 
   G_CALLBACK (view_menu_toolbar_callback) }
};

static guint
neutrino_window_action_n_entries = G_N_ELEMENTS( neutrino_window_action_entries );
static guint
neutrino_window_toggle_action_n_entries = G_N_ELEMENTS( neutrino_window_toggle_action_entries );

void
neutrino_window_initialise_menus( NeutrinoWindow *window )
{
	GError *error;

	window->details->group = gtk_action_group_new( "NeutrinoActions" );
	gtk_action_group_set_translation_domain( window->details->group,
			GETTEXT_PACKAGE );

	gtk_action_group_add_actions( window->details->group,
				neutrino_window_action_entries,
				neutrino_window_action_n_entries,
				window );
	gtk_action_group_add_toggle_actions( window->details->group,
				neutrino_window_toggle_action_entries,
				neutrino_window_toggle_action_n_entries,
				window );

	window->details->merge = gtk_ui_manager_new();
	gtk_ui_manager_insert_action_group( window->details->merge,
					    window->details->group,
					    0 );
	g_signal_connect( window->details->merge,
			  "add_widget",
			  G_CALLBACK( add_widget ),
			  window );
	error = NULL;
	gtk_ui_manager_add_ui_from_file( window->details->merge,
					 UIDATADIR"/neutrino-ui.xml",
					 &error );
	gtk_window_add_accel_group( GTK_WINDOW( window ),
			gtk_ui_manager_get_accel_group( window->details->merge ) );

	if( error ) {
		g_print( "ERROR: %s\n", error->message );
		g_error_free( error );
	}
}


