#ifndef NOMAD_JUKEBOX_H
#define NOMAD_JUKEBOX_H

#include <glib-object.h>

#include <libnjb.h>

#include "nomad-jukebox-track.h"

#define NOMAD_TYPE_JUKEBOX  (nomad_jukebox_get_type())
#define NOMAD_JUKEBOX(obj)  (G_TYPE_CHECK_INSTANCE_CAST( ( obj ), \
                              NOMAD_TYPE_JUKEBOX, \
                              NomadJukebox ) )
#define NOMAD_IS_JUKEBOX(obj)  (G_TYPE_CHECK_INSTANCE_TYPE( ( obj ), \
                                 NOMAD_TYPE_JUKEBOX ) )

typedef struct {
	GObjectClass parent_spot;

	/* track signals */
	void (*scanning_tracks)( guint jobid );
	void (*track_add)( guint jobid,
			guint total_tracks, const NomadTrack *track );
	void (*scanned_tracks)( guint jobid, guint total );

	void (*transfer)( guint jobid, const gchar *uri,
			  gboolean upload, guint64 sent, guint64 total );
	void (*upload_complete)( guint jobid );
	void (*download_complete)( guint jobid );
	void (*downloaded_track)( guint jobid,
			const gchar *uri, 
			const NomadTrack *track );

	void (*track_remove)( guint jobid, const NomadTrack *track );

	void (*track_changed)( guint jobid, const NomadTrack *orig,
			       const NomadTrack *track );

	/* playlist signals */
	void (*scanning_playlists)( guint jobid );
	void (*playlist_add)( guint jobid, guint total_tracks,
			      const gchar *name );
	void (*scanned_playlists)( guint jobid, guint total );
	void (*playlist_track_erased)( guint jobid,
			const gchar *playlist,
			const gchar *artist, const gchar *album,
			const gchar *title );
	void (*playlist_tracks_erased)( guint jobid,
			const gchar *playlist );
	void (*playlist_remove)( guint jobid, const gchar *name );
	void (*playlist_renamed)( guint jobid, const gchar *orignal, 
				  const gchar *new_name );

	/* playback signals */
	void (*play_begin)( guint jobid, 
			const NomadTrack *track );
	void (*play_progress)( guint jobid,
			guint progress, gboolean changed,
			const NomadTrack *track );
	void (*play_finished)( guint jobid );

	/* return value signals */
	void (*time)( guint jobid, const gchar *time_string );
	void (*usage)( guint jobid,
			guint64 total, guint64 free, guint64 used );

	void (*error)( guint jobid, const gchar *message );

} NomadJukeboxClass;

typedef struct NomadJukeboxPrivate NomadJukeboxPrivate;

typedef struct  {
	GObject parent_object;
	NomadJukeboxPrivate *priv;
} NomadJukebox;


GType  nomad_jukebox_get_type( void );
NomadJukebox *nomad_jukebox_new( njb_t *njb );

void nomad_jukebox_lock( NomadJukebox *jukebox );
void nomad_jukebox_unlock( NomadJukebox *jukebox );

gboolean nomad_jukebox_acquire( NomadJukebox *jukebox );
void nomad_jukebox_release( NomadJukebox *jukebox );

/* instant return functions */
const gchar *nomad_jukebox_get_idstring( NomadJukebox *jukebox );
const gchar *nomad_jukebox_get_firmware( NomadJukebox *jukebox );
const gchar *nomad_jukebox_get_prodname( NomadJukebox *jukebox );
guint nomad_jukebox_get_num_tracks( NomadJukebox *jukebox );
guint nomad_jukebox_get_num_playlists( NomadJukebox *jukebox );
guint nomad_jukebox_get_num_data_files( NomadJukebox *jukebox );
gboolean nomad_jukebox_get_power( NomadJukebox *jukebox );
const gchar *nomad_jukebox_get_ownerstring( NomadJukebox *jukebox );

void nomad_jukebox_abort( NomadJukebox *jukebox );

/* current eax settings, instant return */
gint16 nomad_jukebox_get_volume( NomadJukebox *jukebox, 
		gint16 *min, gint16 *max );

gboolean nomad_jukebox_get_muting( NomadJukebox *jukebox );
gboolean nomad_jukebox_get_eq_active( NomadJukebox *jukebox );

gint16 nomad_jukebox_get_bass( NomadJukebox *jukebox, 
		gint16 *min, gint16 *max );
gint16 nomad_jukebox_get_midrange( NomadJukebox *jukebox,
		gint16 *min, gint16 *max );

gint16 nomad_jukebox_get_treble( NomadJukebox *jukebox,
		gint16 *min, gint16 *max );

gint16 nomad_jukebox_get_effamt( NomadJukebox *jukebox,
		gint16 *min, gint16 *max );

gint16 nomad_jukebox_get_midfreq( NomadJukebox *jukebox,
		const gchar ***names );
gint16 nomad_jukebox_get_effect( NomadJukebox *jukebox,
		const gchar ***names );
gint16 nomad_jukebox_get_headphone( NomadJukebox *jukebox,
		const gchar ***names );
gint16 nomad_jukebox_get_rearmode( NomadJukebox *jukebox,
		const gchar ***names );


/* the following functions return their job id, results
 * returned via signal, NOTE: this signal will be in a separate
 * thread */
guint nomad_jukebox_get_time( NomadJukebox *jukebox );

guint nomad_jukebox_build_tracklist( NomadJukebox *jukebox );
guint nomad_jukebox_build_playlist( NomadJukebox *jukebox );

guint nomad_jukebox_delete_tracks( NomadJukebox *jukebox, GList *list );

guint nomad_jukebox_delete_tracks_from_playlist( NomadJukebox *jukebox,
						   GList *list,
						   const gchar *name );

guint nomad_jukebox_delete_files( NomadJukebox *jukebox, GList *list );

guint nomad_jukebox_set_metadata( NomadJukebox *jukebox,
				    const NomadTrack *track );

guint nomad_jukebox_create_playlist( NomadJukebox *jukebox,
                                       const gchar *name );
guint nomad_jukebox_delete_playlist( NomadJukebox *jukebox, 
				       const gchar *name );
guint nomad_jukebox_rename_playlist( NomadJukebox *jukebox, 
				       const gchar *orig,
                                       const gchar *name );
guint nomad_jukebox_add_tracks_to_playlist( NomadJukebox *jukebox,
                                              const gchar *name,
                                              GList *songidlist );

guint nomad_jukebox_set_ownerstring( NomadJukebox *jukebox, 
				       const gchar *owner );

guint nomad_jukebox_getusage( NomadJukebox *jukebox );

guint nomad_jukebox_set_volume( NomadJukebox *jukebox, guint8 volume );
guint nomad_jukebox_set_muting( NomadJukebox *jukebox, 
				  gboolean mutingstatus );
guint nomad_jukebox_set_eq_active( NomadJukebox *jukebox, gboolean eqactive );
guint nomad_jukebox_set_bass( NomadJukebox *jukebox, gint8 bass );
guint nomad_jukebox_set_midrange( NomadJukebox *jukebox, gint8 midrange );
guint nomad_jukebox_set_treble( NomadJukebox *jukebox, gint8 treble );
guint nomad_jukebox_set_effamt( NomadJukebox *jukebox, gint8 effamt );
guint nomad_jukebox_set_midfreq( NomadJukebox *jukebox, gint8 freq );
guint nomad_jukebox_set_effect( NomadJukebox *jukebox, gint8 effect );
guint nomad_jukebox_set_headphone( NomadJukebox *jukebox, gint8 hpmode );
guint nomad_jukebox_set_rearmode( NomadJukebox *jukebox, gint8 rearmode );

guint nomad_jukebox_upload( NomadJukebox *jukebox, GList *list );
guint nomad_jukebox_download( NomadJukebox *jukebox, 
				const gchar *dest,
				const gchar *format,
				GList *list );

guint nomad_jukebox_play_track( NomadJukebox *jukebox, guint trackid );
guint nomad_jukebox_play_elapsed( NomadJukebox *jukebox );
guint nomad_jukebox_play_stop( NomadJukebox *jukebox );

/* end of signal return functions */

GHashTable *nomad_jukebox_request_tracks( NomadJukebox *jukebox );
GHashTable *nomad_jukebox_request_playlists( NomadJukebox *jukebox );

NomadTrack *nomad_jukebox_request_track( NomadJukebox *jukebox,
					       guint id );

#endif
