/*  Nomad:  nomad-jukebox.c
 *
 *  Copyright (C) 2002 David A Knight <david@ritter.demon.co.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <stdlib.h>

#include <string.h>

#include <glib/gqueue.h>
#include <glib/gthread.h>

#include <semaphore.h>

#include "nomad-jukebox.h"

#include "nomad-jukebox-job.h"

#include "nomad-util.h"

#include "id3/nomad-id3.h"

#include "nomadmarshal.h"

struct NomadTrack {
	guint id;
	gchar *artist;
	gchar *title;
	gchar *album;
	gchar *genre;
	gchar *length;
	gchar *year;
	gchar *size;
	gchar *codec;
	gchar *track;
	gchar *name;

	gboolean play_only;

	GMutex *lock;
};

enum {
	TIME_SIGNAL,
	USAGE_SIGNAL,
	SCANNING_TRACKS_SIGNAL,
	SCANNED_TRACKS_SIGNAL,
	SCANNING_PLAYLISTS_SIGNAL,
	PLAYLIST_ADD_SIGNAL,
	SCANNED_PLAYLISTS_SIGNAL,
	TRANSFER_SIGNAL,
	UPLOAD_COMPLETE_SIGNAL,
	DOWNLOAD_COMPLETE_SIGNAL,
	DOWNLOADED_TRACK_SIGNAL,
	PLAYLIST_TRACK_ERASED_SIGNAL,
	PLAYLIST_TRACKS_ERASED_SIGNAL,
	PLAYLIST_REMOVE_SIGNAL,
	PLAYLIST_RENAMED_SIGNAL,
	PLAY_BEGIN_SIGNAL,
	PLAY_PROGRESS_SIGNAL,
	PLAY_FINISHED_SIGNAL,

	TRACK_ADD_SIGNAL,
	TRACK_REMOVE_SIGNAL,
	TRACK_CHANGED_SIGNAL,

	ERROR_SIGNAL,

	LAST_SIGNAL
};
static guint nomad_jukebox_signals[LAST_SIGNAL] = { 0 };

/* eax support based off dumpeax output */
typedef enum {
	NJB_SOUND_SET_VOLUME,
	NJB_SOUND_SET_MUTING,
	NJB_SOUND_SET_EQSTATUS,
	NJB_SOUND_SET_BASS,
	NJB_SOUND_SET_MIDRANGE,
	NJB_SOUND_SET_TREBLE,
	NJB_SOUND_SET_EAXAMT,
	NJB_SOUND_SET_MIDFREQ,
	NJB_SOUND_SET_EAX,
	NJB_SOUND_SET_HEADPHONE,
	NJB_SOUND_SET_REAR,

	NJB_SOUND_NUM_EFFECTS
} NJBEffect;

typedef struct {
	const guint16 id;
	const njb_eax_control_t type;
	const gint16 min;
	const gint16 max;
	gint16 current;
	gchar **names;
} NJBEffects;

static const NJBEffects njb1_effects[ NJB_SOUND_NUM_EFFECTS ] = {
	{ 0x0001, NJB_EAX_SLIDER_CONTROL, 	0, 100, 0, NULL },
	{ 0x0004, NJB_EAX_FIXED_OPTION_CONTROL, 0, 1, 0, NULL },
	{ 0x000D, NJB_EAX_FIXED_OPTION_CONTROL, 0, 1, 0, NULL },
	{ 0x0002, NJB_EAX_SLIDER_CONTROL,	-12, 12, 0, NULL },
	{ 0x0005, NJB_EAX_SLIDER_CONTROL,	-12, 12, 0, NULL },
	{ 0x0003, NJB_EAX_SLIDER_CONTROL,	-12, 12, 0, NULL },
	{ 0x0008, NJB_EAX_SLIDER_CONTROL,	0, 100, 0, NULL },
	{ 0x0006, NJB_EAX_FIXED_OPTION_CONTROL,	0, 9, 0, NULL },
	{ 0x0007, NJB_EAX_FIXED_OPTION_CONTROL,	0, 10, 0, NULL },
	{ 0x0009, NJB_EAX_FIXED_OPTION_CONTROL,	0, 3, 0, NULL },
	{ 0x000A, NJB_EAX_FIXED_OPTION_CONTROL,	0, 2, 0, NULL }
};

static const NJBEffects zen_touch_effects[ NJB_SOUND_NUM_EFFECTS ] = {
	{ 0x0203, NJB_EAX_SLIDER_CONTROL, 	0, 100, 0, NULL },
	{ 0 },
	{ 0 },
	{ 0 },
	{ 0 },
	{ 0 },
	{ 0 },
	{ 0 },
	{ 0x0205, NJB_EAX_FIXED_OPTION_CONTROL,	0, 9, 0, NULL },
	{ 0 },
	{ 0 }
};

#define MAX_MODEL_NUM NJB_DEVICE_POCKETDJ + 1

static const NJBEffects *njb_effects[ MAX_MODEL_NUM ] = {
	njb1_effects,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	zen_touch_effects,
	NULL,
	NULL,
	NULL
};


struct NomadJukeboxPrivate {
	gboolean connected;

	njb_t *njb;
	
	gchar *firmware;

	gchar *owner;

	NJBEffects eax[ NJB_SOUND_NUM_EFFECTS ];
	
	GHashTable *tracks;
	GHashTable *playlists;
	GHashTable *playlistids;

	gchar *transfer_file;
	gpointer handle;

	GThread *transfer_thread;
	GQueue  *job_queue;
	GMutex  *job_queue_mutex;
	
	sem_t transfer_semaphore;

	/* used for playback / transfer */
	NomadTrack *track;

	gint abort;
	guint id3len;

	guint jobid;
	
	GMutex *lock;
};


static njb_songid_t *nomad_jukebox_songid_from_track( NomadTrack *track );

static void nomad_jukebox_refresh_id( NomadJukebox *jukebox );

static gint nomad_jukebox_xfer_cb( u_int64_t sent, u_int64_t total,
				      const gchar *buf, unsigned int len,
				      void *data );
static gpointer nomad_jukebox_transfer_thread( gpointer data );

static gchar *nomad_jukebox_filename_format( const gchar *dest, 
						const gchar *format,
						NomadTrack *track );

static void nomad_jukebox_class_init( NomadJukeboxClass *klass );
static void nomad_jukebox_instance_init( NomadJukebox *jukebox );
static void nomad_jukebox_finalize( GObject *object );
static void nomad_jukebox_set_prop( GObject *object, guint prop_id, 
				       const GValue *value, GParamSpec *spec );
static void nomad_jukebox_get_prop( GObject *object, guint prop_id, 
				       GValue *value, GParamSpec *spec );



static void nomad_jukebox_get_time_real( NomadJukebox *jukebox, 
		guint jobid );

static void nomad_jukebox_build_tracklist_real( NomadJukebox *jukebox, 
		guint jobid );
static void nomad_jukebox_build_playlist_real( NomadJukebox *jukebox, 
		guint jobid );

static void nomad_jukebox_set_ownerstring_real( NomadJukebox *jukebox, 
		guint jobid, const gchar *owner );

static void nomad_jukebox_getusage_real( NomadJukebox *jukebox, 
		guint jobid );

static void nomad_jukebox_upload_real( NomadJukebox *jukebox, 
		guint jobid, GList *list );
static void nomad_jukebox_download_real( NomadJukebox *jukebox, 
		guint jobid, const gchar *dest, const gchar *format,
		GList *list );
static void nomad_jukebox_eax_set( NomadJukebox *jukebox, guint jobid,
		gint type, gint8 value );

static void nomad_jukebox_delete_tracks_real( NomadJukebox *jukebox, 
		guint jobid, GList *list );
static void
nomad_jukebox_delete_tracks_from_playlist_real( NomadJukebox *jukebox,
		guint jobid, GList *list, const gchar *name );

static void nomad_jukebox_set_metadata_real( NomadJukebox *jukebox, 
		guint jobid, NomadTrack *track );

static void nomad_jukebox_create_playlist_real( NomadJukebox *jukebox, 
		guint jobid, const gchar *name );
static void nomad_jukebox_delete_playlist_real( NomadJukebox *jukebox,
		guint jobid, const gchar *name );
static void nomad_jukebox_rename_playlist_real( NomadJukebox *jukebox,
		guint jobid, const gchar *orig, const gchar *name );
static void
nomad_jukebox_add_tracks_to_playlist_real( NomadJukebox *jukebox, 
		guint jobid, const gchar *name, GList *songidlist );

static void nomad_jukebox_play_track_real( NomadJukebox *jukebox, 
		guint jobid, guint trackid );
static void nomad_jukebox_play_elapsed_real( NomadJukebox *jukebox, 
		guint jobid );
static void nomad_jukebox_play_stop_real( NomadJukebox *jukebox, 
		guint jobid );


static njb_playlist_t *nomad_jukebox_fetch_playlist( NomadJukebox *jukebox,
						    const gchar *name );

static guint nomad_jukebox_add_job( NomadJukebox *jukebox,
				      NomadJukeboxJobType type,
				      gpointer data,
				      gpointer data2,
				      gpointer data3,
				      guint priority );

static void nomad_jukebox_eax_init( NomadJukebox *jukebox );

NomadTrack* nomad_track_new( const gchar *artist, const gchar *title,
				   const gchar *album, const gchar *genre,
				   const gchar *length, const gchar *year,
				   const gchar *size, const gchar *codec,
				   const gchar *track, const gchar *name )
{
	NomadTrack *ntrack;

	ntrack = g_new0( NomadTrack, 1 );

	ntrack->lock = g_mutex_new();

	if( artist ) {
		ntrack->artist = g_strdup( artist );
	} else {
		ntrack->artist = g_strdup( "<Unknown>" );
	}

	if( title ) {
		ntrack->title = g_strdup( title );
	} else {
		ntrack->title = g_strdup( "<Unknown>" );
	}
	if( album ) {
		ntrack->album = g_strdup( album );
	} else {
		ntrack->album = g_strdup( "<Unknown>" );
	}
	if( genre ) {
		ntrack->genre = g_strdup( genre );
	} else {
		ntrack->genre = g_strdup( "<Unknown>" );
	}
	if( length ) {
		ntrack->length = g_strdup( length );
	} else {
		ntrack->length = g_strdup( "0:00" );
	}
	if( year ) {
		ntrack->year = g_strdup( year );
	} else {
		ntrack->year = g_strdup( "<Unknown>" );
	}
	if( size ) {
		ntrack->size = g_strdup( size );
	} else {
		ntrack->size = g_strdup( "0" );
	}
	if( codec ) {
		ntrack->codec = g_strdup( codec );
	} else {
		ntrack->codec = g_strdup( "1" );
	}
	if( track ) {
		ntrack->track = g_strdup( track );
	} else {
		ntrack->track = g_strdup( "<Unknown>" );
	}
	if( name ) {
		ntrack->name = g_strdup( name );
	} else {
		ntrack->name = g_strdup( "<Unknown>" );
	}

	return ntrack;
}

NomadTrack* nomad_track_new_with_id( guint id,
					   const gchar *artist,
					   const gchar *title,
					   const gchar *album,
					   const gchar *genre,
					   const gchar *length,
					   const gchar *year,
					   const gchar *size,
					   const gchar *codec,
					   const gchar *track, 
					   const gchar *name )
{
	NomadTrack *ntrack;

	ntrack = nomad_track_new( artist, title, album, genre, length,
				     year, size, codec, track, name );
	ntrack->id = id;

	return ntrack;
}

NomadTrack* nomad_track_copy( const NomadTrack *track )
{
	NomadTrack *ntrack;

	g_return_val_if_fail( track != NULL, NULL );

	g_mutex_lock( track->lock );

	ntrack = nomad_track_new_with_id( track->id,
					     track->artist,
					     track->title,
					     track->album,
					     track->genre,
					     track->length,
					     track->year,
					     track->size,
					     track->codec,
					     track->track,
					     track->name );

	g_mutex_unlock( track->lock );

	ntrack->play_only = track->play_only;

	return ntrack;
}

NomadTrack* nomad_track_copy_and_protect( const NomadTrack *track, 
		gboolean protect )
{
	NomadTrack *ntrack;

	ntrack = nomad_track_copy( track );
	if( ntrack ) {
		ntrack->play_only = protect;
	}

	return ntrack;
}

void nomad_track_assign( NomadTrack *track, const NomadTrack *from )
{
	g_mutex_lock( track->lock );
	g_mutex_lock( from->lock );

	g_free( track->artist );
	track->artist = g_strdup( from->artist );
	
	g_free( track->title );
	track->title = g_strdup( from->title );
	
	g_free( track->album );
	track->album = g_strdup( from->album );
		
	g_free( track->genre );
	track->genre = g_strdup( from->genre );
	
	g_free( track->length );
	track->length = g_strdup( from->length );

	g_free( track->year );
	track->year = g_strdup( from->year );

	g_free( track->size );
	track->size = g_strdup( from->size );

	g_free( track->codec );
	track->codec = g_strdup( from->codec );

	g_free( track->track );
	track->track = g_strdup( from->track );

	g_free( track->name );
	track->name = g_strdup( from->name );

	g_mutex_unlock( from->lock );
	g_mutex_unlock( track->lock );
}

void nomad_track_free( NomadTrack *track )
{
	g_return_if_fail( track != NULL );

	g_free( track->artist );
	g_free( track->title );
	g_free( track->album );
	g_free( track->genre );
	g_free( track->length );
	g_free( track->year );
	g_free( track->size );
	g_free( track->codec );
	g_free( track->track );
	g_free( track->name );

	g_mutex_free( track->lock );

	g_free( track );
}

guint nomad_track_get_id( const NomadTrack *track )
{
	guint ret;

	g_return_val_if_fail( track != NULL, 0 );

	g_mutex_lock( track->lock );

	ret = track->id;

	g_mutex_unlock( track->lock );

	return ret;
}

gchar *nomad_track_get_artist( const NomadTrack *track )
{
	gchar *ret;

	g_return_val_if_fail( track != NULL, NULL );

	g_mutex_lock( track->lock );

	ret = g_strdup( track->artist );

	g_mutex_unlock( track->lock );

	return ret;
}

gchar *nomad_track_get_title( const NomadTrack *track )
{
	gchar *ret;

	g_return_val_if_fail( track != NULL, NULL );

	g_mutex_lock( track->lock );

	ret = g_strdup( track->title );

	g_mutex_unlock( track->lock );

	return ret;
}

gchar *nomad_track_get_album( const NomadTrack *track )
{
	gchar *ret;

	g_return_val_if_fail( track != NULL, NULL );

	g_mutex_lock( track->lock );

	ret = g_strdup( track->album );

	g_mutex_unlock( track->lock );

	return ret;
}

gchar *nomad_track_get_genre( const NomadTrack *track )
{
	gchar *ret;

	g_return_val_if_fail( track != NULL, NULL );

	g_mutex_lock( track->lock );

	ret = g_strdup( track->genre );

	g_mutex_unlock( track->lock );

	return ret;
}

gchar *nomad_track_get_length( const NomadTrack *track )
{
	gchar *ret;

	g_return_val_if_fail( track != NULL, NULL );

	g_mutex_lock( track->lock );

	ret = g_strdup( track->length );

	g_mutex_unlock( track->lock );

	return ret;
}

gchar *nomad_track_get_year( const NomadTrack *track )
{
	gchar *ret;

	g_return_val_if_fail( track != NULL, NULL );

	g_mutex_lock( track->lock );

	ret = g_strdup( track->year );

	g_mutex_unlock( track->lock );

	return ret;
}

gchar *nomad_track_get_size( const NomadTrack *track )
{
	gchar *ret;

	g_return_val_if_fail( track != NULL, NULL );

	g_mutex_lock( track->lock );

	ret = g_strdup( track->size );

	g_mutex_unlock( track->lock );

	return ret;
}

gchar *nomad_track_get_codec( const NomadTrack *track )
{
	gchar *ret;

	g_return_val_if_fail( track != NULL, NULL );

	g_mutex_lock( track->lock );

	ret = g_strdup( track->codec );

	g_mutex_unlock( track->lock );

	return ret;
}

gchar *nomad_track_get_track( const NomadTrack *track )
{
	gchar *ret;

	g_return_val_if_fail( track != NULL, NULL );

	g_mutex_lock( track->lock );

	ret = g_strdup( track->track );

	g_mutex_unlock( track->lock );

	return ret;
}

gchar *nomad_track_get_name( const NomadTrack *track )
{
	gchar *ret;

	g_return_val_if_fail( track != NULL, NULL );

	g_mutex_lock( track->lock );

	ret = g_strdup( track->name );

	g_mutex_unlock( track->lock );

	return ret;
}

gboolean nomad_track_get_play_only( const NomadTrack *track )
{
	g_return_val_if_fail( track != NULL, FALSE );

	return track->play_only;
}

void nomad_track_get( const NomadTrack *track,
			 gchar **artist, gchar **title,
			 gchar **album, gchar **genre, gchar **length,
			 gchar **year, gchar **size, gchar **codec,
			 gchar **ttrack, gchar **name, guint *id )
{
	g_return_if_fail( track != NULL );

	g_mutex_lock( track->lock );

	if( id ) {
		*id = track->id;
	}

	if( artist ) {
		*artist = g_strdup( track->artist );
	}
	if( title ) {
		*title = g_strdup( track->title );
	}
	if( album ) {
		*album = g_strdup( track->album );
	}
	if( genre ) {
		*genre = g_strdup( track->genre );
	}
	if( length ) {
		*length = g_strdup( track->length );
	}
	if( year ) {
		*year = g_strdup( track->year );
	}
	if( size ) {
		*size = g_strdup( track->size );
	}
	if( codec ) {
		*codec = g_strdup( track->codec );
	}
	if( ttrack ) {
		*ttrack = g_strdup( track->track );
	}
	if( name ) {
		*name = g_strdup( track->name );
	}

	g_mutex_unlock( track->lock );
}

static njb_songid_t *nomad_jukebox_songid_from_track( NomadTrack *track )
{
	njb_songid_t *songid;
	njb_songid_frame_t *frame;
	guint16 length;
	guint16 tracknum;
	guint32 size;
	guint16 year;
	gchar *dummy;
	
	length = (guint16)mmss_to_seconds( track->length );
	tracknum = (guint16)strtoul( track->track, &dummy, 10 );
	size = strtoul( track->size, &dummy, 10 );
	year = (guint16)strtoul( track->year, &dummy, 10 );

	/* hack libnjb doesn't allow 0 length */
	if( length == 0 ) {
		length = 1;
	}
	
	songid = NJB_Songid_New();

	frame = NJB_Songid_Frame_New_Codec( track->codec );
	NJB_Songid_Addframe( songid, frame );
	frame = NJB_Songid_Frame_New_Title( track->title );
	NJB_Songid_Addframe( songid, frame );
	frame = NJB_Songid_Frame_New_Album( track->album );
	NJB_Songid_Addframe( songid, frame );
	frame = NJB_Songid_Frame_New_Genre( track->genre );
	NJB_Songid_Addframe( songid, frame );
	frame = NJB_Songid_Frame_New_Artist( track->artist );
	NJB_Songid_Addframe( songid, frame );
	frame = NJB_Songid_Frame_New_Year( year );
	NJB_Songid_Addframe( songid, frame );
	frame = NJB_Songid_Frame_New_Filename( track->name );
	NJB_Songid_Addframe( songid, frame );
	if( track->play_only ) {
		frame = NJB_Songid_Frame_New_Protected( 1 );
		NJB_Songid_Addframe( songid, frame );
	}
	frame = NJB_Songid_Frame_New_Length( length );
	NJB_Songid_Addframe( songid, frame );
	frame = NJB_Songid_Frame_New_Tracknum( tracknum );
	NJB_Songid_Addframe( songid, frame );
	frame = NJB_Songid_Frame_New_Filesize( size );
	NJB_Songid_Addframe( songid, frame );

	return songid;
}

NomadJukebox *nomad_jukebox_new( njb_t *njb )
{
	NomadJukebox *jukebox;

	jukebox = NOMAD_JUKEBOX( g_object_new( nomad_jukebox_get_type(),
						  NULL ) );

	jukebox->priv->njb = njb;

	return jukebox;
}

void nomad_jukebox_lock( NomadJukebox *jukebox )
{
	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_mutex_lock( jukebox->priv->lock );
}

void nomad_jukebox_unlock( NomadJukebox *jukebox )
{
	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_mutex_unlock( jukebox->priv->lock );
}

gboolean nomad_jukebox_acquire( NomadJukebox *jukebox )
{
	NomadJukeboxPrivate *priv;
	
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), FALSE );
	g_return_val_if_fail( jukebox->priv->njb != NULL, FALSE );

	priv = jukebox->priv;
	
	priv->connected = ( NJB_Open( priv->njb ) != -1 && 
			NJB_Capture( priv->njb ) != -1 );

	g_free( priv->owner );
	priv->owner = NULL;
	
	g_free( priv->firmware );
	priv->firmware = NULL;

	if( priv->connected ) {
		gchar *owner;
		guint8 major;
		guint8 minor;
		guint8 release;

		owner = NJB_Get_Owner_String( priv->njb );
		if( ! owner ) {
			owner = g_strdup( "" );
		}
		priv->owner = owner;

		nomad_jukebox_refresh_id( jukebox );

		NJB_Get_Firmware_Revision( jukebox->priv->njb,
				&major, &minor, &release );
		
		priv->firmware = g_strdup_printf( "%u.%u release %u",
				major, minor, release );

		nomad_jukebox_eax_init( jukebox );
	}

	return jukebox->priv->connected;
}

void nomad_jukebox_release( NomadJukebox *jukebox )
{
	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->priv->njb != NULL );
	g_return_if_fail( jukebox->priv->connected );
	
	if( NJB_Release( jukebox->priv->njb ) != -1 ) {
		NJB_Close( jukebox->priv->njb );
	} else {
		NJB_Error_Dump( jukebox->priv->njb, stdout );
	}

	g_free( jukebox->priv->owner );
	jukebox->priv->owner = NULL;
	
	g_free( jukebox->priv->firmware );
	jukebox->priv->firmware = NULL;
}

const gchar *nomad_jukebox_get_idstring( NomadJukebox *jukebox )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), NULL );
	g_return_val_if_fail( jukebox->priv->njb != NULL, NULL );
	g_return_val_if_fail( jukebox->priv->connected, NULL );

	return NJB_Get_Device_Name( jukebox->priv->njb, 0 );
}

const gchar *nomad_jukebox_get_firmware( NomadJukebox *jukebox )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), NULL );
	g_return_val_if_fail( jukebox->priv->njb != NULL, NULL );
	g_return_val_if_fail( jukebox->priv->connected, NULL );

	return jukebox->priv->firmware;
}

const gchar *nomad_jukebox_get_prodname( NomadJukebox *jukebox )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), NULL );
	g_return_val_if_fail( jukebox->priv->njb != NULL, NULL );
	g_return_val_if_fail( jukebox->priv->connected, NULL );

	return NJB_Get_Device_Name( jukebox->priv->njb, 1 );
}

guint nomad_jukebox_get_num_tracks( NomadJukebox *jukebox )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );

	return g_hash_table_size( jukebox->priv->tracks );
}

guint nomad_jukebox_get_num_playlists( NomadJukebox *jukebox )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );

	return g_hash_table_size( jukebox->priv->playlists );
}

guint nomad_jukebox_get_num_data_files( NomadJukebox *jukebox )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );

	return 0;
}



gboolean nomad_jukebox_get_power( NomadJukebox *jukebox )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), FALSE );
	g_return_val_if_fail( jukebox->priv->njb != NULL, FALSE );
	g_return_val_if_fail( jukebox->priv->connected, FALSE );

	return NJB_Get_Auxpower( jukebox->priv->njb );
}

guint nomad_jukebox_get_time( NomadJukebox *jukebox )
{
	return nomad_jukebox_add_job( jukebox, GET_TIME,
				  NULL, NULL, NULL, 0 );
}

static void nomad_jukebox_get_time_real( NomadJukebox *jukebox, 
		guint jobid )
{
	njb_time_t *time;

	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->priv->njb != NULL );
	g_return_if_fail( jukebox->priv->connected );
	
	time = NJB_Get_Time( jukebox->priv->njb );
	if( time ) {
		gchar *tstring;
		tstring = g_strdup_printf("%u-%.2u-%.2u %.2u:%.2u:%.2u",
					  time->year, time->month, time->day,
					  time->hours, time->minutes,
					  time->seconds);
		
		g_mutex_lock( jukebox->priv->job_queue_mutex );
		g_signal_emit( G_OBJECT( jukebox ),
			       nomad_jukebox_signals[ TIME_SIGNAL ], 0,
			       jobid, tstring );
		g_mutex_unlock( jukebox->priv->job_queue_mutex );
		
		g_free( tstring );

	}
}

guint nomad_jukebox_build_tracklist( NomadJukebox *jukebox )
{
	return nomad_jukebox_add_job( jukebox, BUILD_TRACKLIST,
				  NULL, NULL, NULL, 0 );
}

static void nomad_jukebox_build_tracklist_real( NomadJukebox *jukebox, 
		guint jobid )
{
	njb_songid_t *songtag;
	gint total_tracks;

	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->priv->njb != NULL );
	g_return_if_fail( jukebox->priv->connected );

	/* emit signal to signify that tracks are about to be scanned */
	g_mutex_lock( jukebox->priv->job_queue_mutex );
	g_signal_emit( G_OBJECT( jukebox ),
		       nomad_jukebox_signals[ SCANNING_TRACKS_SIGNAL ],
		       0, jobid );
	g_mutex_unlock( jukebox->priv->job_queue_mutex );

	NJB_Reset_Get_Track_Tag( jukebox->priv->njb );
	total_tracks = 0;
	while( ( songtag = NJB_Get_Track_Tag( jukebox->priv->njb ) ) ) {
		njb_songid_frame_t *frame;
		NomadTrack *track;

		NJB_Songid_Reset_Getframe( songtag );

		track = g_new0( NomadTrack, 1 );
		track->lock = g_mutex_new();

		while( ( frame = NJB_Songid_Getframe( songtag ) ) ) {
			const gchar *label;
			gchar *data;

			label = frame->label;
			
			switch( frame->type ) {
			case NJB_TYPE_UINT32:
				data = g_strdup_printf( "%lu",
						(unsigned long)frame->data.u_int32_val );
				break;
			case NJB_TYPE_UINT16:
				if( ! strcmp( label, FR_LENGTH ) ) {
					data = seconds_to_mmss( frame->data.u_int16_val );
				} else {
					data = g_strdup_printf( "%d",
								(unsigned short)frame->data.u_int16_val );
				}
				break;
			case NJB_TYPE_STRING:
				data = NULL;
				if( *frame->data.strval != '\0' ) {
					data = g_strdup( frame->data.strval );
				}
				if( ! data ) {
					data = g_strdup( "<Unknown>" );
				}
				break;
			default:
				g_warning( "Unknown frame type: %i\n",
					   frame->type );
				data = g_strdup( "" );
				break;
			}

			if( ! strcmp( label, FR_ARTIST ) ) {
				track->artist = g_strstrip( data );
			} else if( ! strcmp( label, FR_TITLE ) ) {
				track->title = g_strstrip( data );
			} else if( ! strcmp( label, FR_ALBUM ) ) {
				track->album = g_strstrip( data );
			} else if( ! strcmp( label, FR_GENRE ) ) {
				track->genre = g_strstrip( data );
			} else if( ! strcmp( label, FR_LENGTH ) ) {
				track->length = g_strstrip( data );
			} else if( ! strcmp( label, FR_YEAR ) ) {
				track->year = g_strstrip( data );
			} else if( ! strcmp( label, FR_SIZE ) ) {
				track->size = g_strstrip( data );
			} else if( ! strcmp( label, FR_CODEC ) ) {
				track->codec = g_strstrip( data );
			} else if( ! strcmp( label, FR_TRACK ) ) {
				track->track = g_strstrip( data );
			} else if( ! strcmp( label, FR_FNAME ) ) {
				track->name = g_strstrip( data );
			} else if( ! strcmp( label, FR_PROTECTED ) ) {
				/* ignore this we don't care about it,
				   we could possibly set a column in the model
				   to indicate if a track is play only, and
				   then use it to give some indication in the
				   view */
				if( *data == '1' ) {
					track->play_only = TRUE;
				}
				g_free( data );
			} else if( data ) {
				/* unsupported frame tag */
				g_warning( "Unsupported label: %s -> %s\n",
					   label, data );
				g_free( data );
			}

		}
		if( ! track->artist ) {
			track->artist = g_strdup( "<Unknown>" );
		}
		if( ! track->title ) {
			track->title = g_strdup( "<Unknown>" );
		}
		if( ! track->album ) {
			track->album = g_strdup( "<Unknown>" );
		}
		if( ! track->genre ) {
			track->genre = g_strdup( "<Unknown>" );
		}
		if( ! track->length ) {
			track->length = g_strdup( "0:00" );
		}
		if( ! track->year ) {
			track->year = g_strdup( "<Unknown>" );
		}
		if( ! track->size ) {
			track->size = g_strdup( "0" );
		}
		if( ! track->codec ) {
			track->codec = g_strdup( "1" );
		}
		if( ! track->track ) {
			/* GNOMAD sets this to songtag->trid,
			   this isn't right IMO as it is just an id number
			   and not suitable for a track, it can give values
			   of hundreds of thousands, and there are no CDs with
			   that many tracks on */
			track->track = g_strdup( "Unknown" );
		}
		track->id = songtag->trid;

		/* insert NomadTrack into tracks hashtable */
		g_hash_table_insert( jukebox->priv->tracks,
				     GUINT_TO_POINTER( songtag->trid ),
				     track );

		/* emit signal to signify that a track has been scanned */
		total_tracks ++;

		g_mutex_lock( jukebox->priv->job_queue_mutex );
		g_signal_emit( G_OBJECT( jukebox ),
			       nomad_jukebox_signals[ TRACK_ADD_SIGNAL ],
			       0, jobid, total_tracks, track );
		g_mutex_unlock( jukebox->priv->job_queue_mutex );

		NJB_Songid_Destroy( songtag );
	}
	/* emit signal to signify that tracks have been scanned */
	g_mutex_lock( jukebox->priv->job_queue_mutex );
	g_signal_emit( G_OBJECT( jukebox ),
		       nomad_jukebox_signals[ SCANNED_TRACKS_SIGNAL ], 
		       0, jobid, total_tracks );
	g_mutex_unlock( jukebox->priv->job_queue_mutex );
}

guint nomad_jukebox_build_playlist( NomadJukebox *jukebox )
{
	return nomad_jukebox_add_job( jukebox, BUILD_PLAYLIST,
				  NULL, NULL, NULL, 0 );
}

static void nomad_jukebox_build_playlist_real( NomadJukebox *jukebox, 
		guint jobid )
{
	njb_playlist_t *playlist;
        gint total_lists;
	
        g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
        g_return_if_fail( jukebox->priv->njb != NULL );
        g_return_if_fail( jukebox->priv->connected );
        g_return_if_fail( jukebox->priv->tracks != NULL );
	
	g_mutex_lock( jukebox->priv->job_queue_mutex );
        g_signal_emit( G_OBJECT( jukebox ),
                       nomad_jukebox_signals[ SCANNING_PLAYLISTS_SIGNAL ],
                       0, jobid );
	g_mutex_unlock( jukebox->priv->job_queue_mutex );

	NJB_Reset_Get_Playlist( jukebox->priv->njb );
        total_lists = 0;
        while( ( playlist = NJB_Get_Playlist( jukebox->priv->njb ) ) ) {
		GSList *tracks;
		gchar *name;
                njb_playlist_track_t *track;  
                guint total_tracks;
	                                    
                NJB_Playlist_Reset_Gettrack( playlist );
		for( tracks = NULL, total_tracks = 0;
		     ( track = NJB_Playlist_Gettrack( playlist ) ); ) {
                        NomadTrack *gdap_track;
			
                        gdap_track = g_hash_table_lookup( jukebox->priv->tracks,
                                                          GUINT_TO_POINTER( track->trackid ) );
			if( gdap_track ) {
				tracks = g_slist_prepend( tracks, gdap_track );
				total_tracks ++;
                        } else {
                                g_warning( "Unknown track in playlist: %lu\n",
                                           (unsigned long)track->trackid );
                        }
                }
		if( tracks ) {
			tracks = g_slist_reverse( tracks );
		}
		/* we add the playlist id to the front */
		tracks = g_slist_prepend( tracks, 
					  GUINT_TO_POINTER( playlist->plid ) );

		/* add to playlists - key by name, data is a list,
		   first item in the list is the playlist id,
		   subsequent items are NomadTrack pointers */
		name = g_strdup( playlist->name );
		g_hash_table_insert( jukebox->priv->playlists,
				     name,
				     tracks );
		g_hash_table_insert( jukebox->priv->playlistids,
				     GUINT_TO_POINTER( playlist->plid ), 
				     name );
                                        
                /* emit signal to signify that a playlist has
                   been scanned */
		g_mutex_lock( jukebox->priv->job_queue_mutex );
                g_signal_emit( G_OBJECT( jukebox ),
                               nomad_jukebox_signals[ PLAYLIST_ADD_SIGNAL],
                               0, jobid, total_tracks, playlist->name );
		g_mutex_unlock( jukebox->priv->job_queue_mutex );

		total_lists ++;

		NJB_Playlist_Destroy( playlist );
        }
        /* emit signal to signify that tracks have been scanned */
	g_mutex_lock( jukebox->priv->job_queue_mutex );
        g_signal_emit( G_OBJECT( jukebox ),
                       nomad_jukebox_signals[ SCANNED_PLAYLISTS_SIGNAL ], 
		       0, jobid, total_lists );
	g_mutex_unlock( jukebox->priv->job_queue_mutex );

}

guint nomad_jukebox_delete_tracks( NomadJukebox *jukebox, GList *list )
{
	/* erase tracks from playlists, we do this first,
	   as if something goes wrong it means the playlists
	   will be consistant, I.E. they won't contain non existant
	   tracks 
	   
		This job id is never returned, is this correct?
	   
	   */
	nomad_jukebox_delete_tracks_from_playlist( jukebox, list, NULL );

	return nomad_jukebox_add_job( jukebox, DELETE_TRACKS,
				  g_list_copy( list ), NULL, NULL, 0 );
}

guint nomad_jukebox_delete_tracks_from_playlist( NomadJukebox *jukebox,
						   GList *list,
						   const gchar *name )
{
	gchar *nm;

	if( name ) {
		nm = g_strdup( name );
	} else {
		nm = NULL;
	}

	return nomad_jukebox_add_job( jukebox, 
			DELETE_TRACKS_FROM_PLAYLIST,
			g_list_copy( list ), nm, NULL, 0 );
}

static void
nomad_jukebox_delete_tracks_from_playlist_real( NomadJukebox *jukebox, 
		guint jobid, GList *list, const gchar *name )
{
        GList *tmplist;
        njb_playlist_t *playlist;


        g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
        g_return_if_fail( jukebox->priv->njb != NULL );
        g_return_if_fail( jukebox->priv->connected );
        g_return_if_fail( list != NULL );
	
	NJB_Reset_Get_Playlist( jukebox->priv->njb );
        while( ( playlist = NJB_Get_Playlist( jukebox->priv->njb ) ) ) {
                njb_playlist_track_t *track;
                gint pos = NJB_PL_START;
                gboolean changed = FALSE;
		GSList *tracks;
		GSList *copy;
		gchar *current;
		const gchar *pname;

                if( name && strcmp( name, playlist->name ) ) {
                        continue;
                }

		if( ! name ) {
			current = g_strdup( playlist->name );
		} else {
			current = g_strdup( name );
		}

		tracks = g_hash_table_lookup( jukebox->priv->playlists, 
				current );
		g_assert( tracks );
		copy = g_slist_copy( tracks->next );
		
		while( ( track = NJB_Playlist_Gettrack( playlist ) ) ) {
                        for( tmplist = list; tmplist;
                             tmplist = tmplist->next ) {
                                if( GPOINTER_TO_UINT(tmplist->data) ==
                                    track->trackid ) {
                                        NomadTrack *dtrack;
					
                                        dtrack = (NomadTrack*)g_hash_table_lookup( jukebox->priv->tracks, tmplist->data );
					
                                        g_assert( dtrack );
	
					g_mutex_lock( jukebox->priv->job_queue_mutex );
                                        g_signal_emit( G_OBJECT( jukebox ),
                                                       nomad_jukebox_signals[ PLAYLIST_TRACK_ERASED_SIGNAL ],
                                                       0,
						       jobid,
                                                       current,
                                                       dtrack->artist,
                                                       dtrack->album,
                                                       dtrack->title );
					g_mutex_unlock( jukebox->priv->job_queue_mutex );
                                        NJB_Playlist_Deltrack( playlist,
                                                           pos );
                                        changed = TRUE;

					copy = g_slist_remove( copy,
							       dtrack );
                                }
                        }
                        pos ++;
                }
		
		g_free( current );
		
		if( changed ) {
                        if( NJB_Update_Playlist( jukebox->priv->njb, 
						 playlist ) != -1 ) {
				copy = g_slist_prepend( copy, GUINT_TO_POINTER( playlist->plid ) );
				pname = g_hash_table_lookup( jukebox->priv->playlistids,
							     tracks->data );
				g_assert( pname );
				g_hash_table_remove( jukebox->priv->playlistids,
						     tracks->data );
				g_hash_table_insert( jukebox->priv->playlistids,
						     copy->data,
						     (gpointer)pname );
				g_hash_table_replace( jukebox->priv->playlists,
						      (gpointer)pname,
						      copy );
				g_slist_free( tracks );

				g_mutex_lock( jukebox->priv->job_queue_mutex );
				g_signal_emit( G_OBJECT( jukebox ),
					       nomad_jukebox_signals[ PLAYLIST_TRACKS_ERASED_SIGNAL ],
					       0, jobid, pname );
				g_mutex_unlock( jukebox->priv->job_queue_mutex );
			} else {
				gchar *errmsg;

				errmsg = g_strdup_printf( "Failed to delete tracks from %s", pname );

				g_mutex_lock( jukebox->priv->job_queue_mutex );
				g_signal_emit( G_OBJECT( jukebox ),
					       nomad_jukebox_signals[ ERROR_SIGNAL ], 0,
					       jobid, errmsg );
				g_mutex_unlock( jukebox->priv->job_queue_mutex );

				g_free( errmsg );
				g_slist_free( copy );
                        }
                } else {
			g_slist_free( copy );
		}
                NJB_Playlist_Destroy( playlist );
		
                if( changed && name ) {
                        break;
                }
        }
}

static void nomad_jukebox_delete_tracks_real( NomadJukebox *jukebox,
		guint jobid, GList *list )
{
	GList *tmplist;

	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->priv->njb != NULL );
	g_return_if_fail( jukebox->priv->connected );
	g_return_if_fail( list != NULL );

	/* erase the tracks */
	for( tmplist = list; tmplist; tmplist = tmplist->next ) {
		u_int32_t id;
		NomadTrack *track;

		id = GPOINTER_TO_UINT( tmplist->data );

		track = nomad_jukebox_request_track( jukebox, id );
		g_assert( track );

		if( NJB_Delete_Track( jukebox->priv->njb, id ) != -1 ) {
			g_mutex_lock( jukebox->priv->job_queue_mutex );
			g_signal_emit( G_OBJECT( jukebox ),
				       nomad_jukebox_signals[ TRACK_REMOVE_SIGNAL ],
				       0, jobid, track );
			g_mutex_unlock( jukebox->priv->job_queue_mutex );
			g_hash_table_remove( jukebox->priv->tracks,
					     tmplist->data );
		} else {
			gchar *errmsg;
			gchar *trackname;

			trackname = nomad_track_get_title( track );
			errmsg = g_strdup_printf( "Failed to delete track %s",
						  trackname );
			g_mutex_lock( jukebox->priv->job_queue_mutex );
			g_signal_emit( G_OBJECT( jukebox ),
				       nomad_jukebox_signals[ ERROR_SIGNAL ], 0,
				       jobid, errmsg );
			g_mutex_unlock( jukebox->priv->job_queue_mutex );
			
			g_free( errmsg );
			g_free( trackname );
		}
		nomad_track_free( track );
	}
}

guint nomad_jukebox_delete_files( NomadJukebox *jukebox, GList *list )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );
	g_return_val_if_fail( jukebox->priv->njb != NULL, 0 );
	g_return_val_if_fail( jukebox->priv->connected, 0 );
	g_return_val_if_fail( list != NULL, 0 );

	g_warning( "Deleting files not yet supported\n" );

	return 0;
}

guint nomad_jukebox_set_metadata( NomadJukebox *jukebox,
				    const NomadTrack *track )
{
	g_return_val_if_fail( track != NULL, 0 );

	return nomad_jukebox_add_job( jukebox, SET_METADATA,
				  nomad_track_copy( track ),
				  NULL, NULL, 0 );
}

static void nomad_jukebox_set_metadata_real( NomadJukebox *jukebox,
		guint jobid, NomadTrack *track )
{
	NomadTrack *old;
	njb_songid_t *songid;
	
	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->priv->njb != NULL );
	g_return_if_fail( jukebox->priv->connected );
	g_return_if_fail( track != NULL );

	old = g_hash_table_lookup( jukebox->priv->tracks,
				   GUINT_TO_POINTER( track->id ) );
	g_assert( old );

	songid = nomad_jukebox_songid_from_track( track );
	
	if( NJB_Replace_Track_Tag( jukebox->priv->njb, track->id, 
				songid ) != -1 ) {
		/* set new track data and emit signal */
		g_mutex_lock( jukebox->priv->job_queue_mutex );
		g_signal_emit( G_OBJECT( jukebox ),
			       nomad_jukebox_signals[ TRACK_CHANGED_SIGNAL ],
			       0, jobid, old, track );
		g_mutex_unlock( jukebox->priv->job_queue_mutex );
		nomad_track_assign( old, track );
	} else {
		gchar *errmsg;
		gchar *trackname;

		trackname = nomad_track_get_title( track );
		errmsg = g_strdup_printf( "Failed to set track information for %s",
					  track->title );
		
		g_mutex_lock( jukebox->priv->job_queue_mutex );
		g_signal_emit( G_OBJECT( jukebox ),
			       nomad_jukebox_signals[ ERROR_SIGNAL ], 0,
			       jobid, errmsg );
		g_mutex_unlock( jukebox->priv->job_queue_mutex );
		
		g_free( errmsg );
		g_free( trackname );
	}
	NJB_Songid_Destroy( songid );
	nomad_track_free( track );
}

guint nomad_jukebox_create_playlist( NomadJukebox *jukebox,
                                       const gchar *name )
{
	g_return_val_if_fail( name != NULL, 0 );

	return nomad_jukebox_add_job( jukebox, CREATE_PLAYLIST,
				  g_strdup( name ), NULL, NULL, 0 );
}

static void nomad_jukebox_create_playlist_real( NomadJukebox *jukebox, 
		guint jobid, const gchar *name )
{
        njb_playlist_t *playlist;
        guint plid;
	gboolean failed;

        g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
        g_return_if_fail( jukebox->priv->njb != NULL );
        g_return_if_fail( jukebox->priv->connected );
        g_return_if_fail( name != NULL );
	
        plid = 0;
        playlist = NJB_Playlist_New();
	failed = FALSE;
	
        if( playlist ) {
		GSList *list;

		list = NULL;

                if( NJB_Playlist_Set_Name( playlist, name ) != -1 ) {
			plid = NJB_Update_Playlist(jukebox->priv->njb, playlist );
			if( plid != -1 ) {
				gchar *pname;
				
				list = g_slist_prepend( NULL, 
							GUINT_TO_POINTER( plid ) );
				pname = g_strdup( playlist->name );
				g_hash_table_insert( jukebox->priv->playlists,
						     pname, list );
				g_hash_table_insert( jukebox->priv->playlistids,
						     GUINT_TO_POINTER( plid ),
						     pname );
				g_mutex_lock( jukebox->priv->job_queue_mutex );
				g_signal_emit( G_OBJECT( jukebox ),
                                               nomad_jukebox_signals[PLAYLIST_ADD_SIGNAL ], 0,
                                               jobid, 0, 
					       playlist->name );
				g_mutex_unlock( jukebox->priv->job_queue_mutex );
			} else {
                                /* failed to update playlist */
				failed = TRUE;
                        }
                } else {
                        /* failed to set the playlist name! */
			failed = TRUE;
                }
		if( failed ) {
			gchar *errmsg;
		
			errmsg = g_strdup_printf( "Failed to create playlist %s",
						  name );
		
			g_mutex_lock( jukebox->priv->job_queue_mutex );
			g_signal_emit( G_OBJECT( jukebox ),
				       nomad_jukebox_signals[ ERROR_SIGNAL ], 0,
				       jobid, errmsg );
			g_mutex_unlock( jukebox->priv->job_queue_mutex );
			
			g_free( errmsg );
		}
		NJB_Playlist_Destroy( playlist );
	}
}


guint nomad_jukebox_delete_playlist( NomadJukebox *jukebox, 
				       const gchar *name )
{
	g_return_val_if_fail( name != NULL, 0 );

	return nomad_jukebox_add_job( jukebox, DELETE_PLAYLIST,
				  g_strdup( name ), NULL, NULL, 0 );
}

static void nomad_jukebox_delete_playlist_real( NomadJukebox *jukebox,
		guint jobid, const gchar *name )
{
	GSList *tracks;
	guint plid;

        g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
        g_return_if_fail( jukebox->priv->njb != NULL );
        g_return_if_fail( jukebox->priv->connected );
	g_return_if_fail( name != NULL );
	
	tracks = g_hash_table_lookup( jukebox->priv->playlists, name );
	g_assert( tracks );
	plid = GPOINTER_TO_UINT( tracks->data );

        if( NJB_Delete_Playlist( jukebox->priv->njb, plid ) != -1 ) {
		gchar *pname;
		
		pname = g_hash_table_lookup( jukebox->priv->playlistids,
					     tracks->data );
		g_assert( pname );

		g_mutex_lock( jukebox->priv->job_queue_mutex );
		g_signal_emit(G_OBJECT( jukebox ),
                              nomad_jukebox_signals[PLAYLIST_REMOVE_SIGNAL],
                              0, jobid, pname );
		g_mutex_unlock( jukebox->priv->job_queue_mutex );
	
		g_hash_table_remove( jukebox->priv->playlistids,
				     tracks->data );
		g_hash_table_remove( jukebox->priv->playlists, pname );
		g_slist_free( tracks );

		/* pname is freed by its removal from
		 * the playlists hash table */
        } else {
		gchar *errmsg;
		
		errmsg = g_strdup_printf( "Failed to delete playlist %s",
					  name );
		g_mutex_lock( jukebox->priv->job_queue_mutex );
		g_signal_emit( G_OBJECT( jukebox ),
			       nomad_jukebox_signals[ ERROR_SIGNAL ], 0,
			       jobid, errmsg );
		g_mutex_unlock( jukebox->priv->job_queue_mutex );
			
		g_free( errmsg );
        }
}
                  
guint nomad_jukebox_rename_playlist( NomadJukebox *jukebox,
				       const gchar *orig,
                                       const gchar *name )
{
	g_return_val_if_fail( orig != NULL, 0 );
        g_return_val_if_fail( name != NULL, 0 );
	
	return nomad_jukebox_add_job( jukebox, RENAME_PLAYLIST,
				  g_strdup( orig ), 
				  g_strdup( name ), NULL, 0 );
}

static void nomad_jukebox_rename_playlist_real( NomadJukebox *jukebox, 
		guint jobid, const gchar *orig, const gchar *name )
{
        njb_playlist_t *playlist;
	GSList *list;	

        g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
        g_return_if_fail( jukebox->priv->njb != NULL );
        g_return_if_fail( jukebox->priv->connected );
        g_return_if_fail( name != NULL );
	
	list = g_hash_table_lookup( jukebox->priv->playlists, (gpointer)orig );
	g_assert( list );

	playlist = nomad_jukebox_fetch_playlist( jukebox, orig );
	g_assert( playlist );

	if( NJB_Playlist_Set_Name( playlist, name ) != -1 ) {
		if( NJB_Update_Playlist( jukebox->priv->njb,
					 playlist ) == -1 ) {
			gchar *errmsg;
		
			errmsg = g_strdup_printf( "Failed to rename playlist %s to %s",
						  orig, name );
		
			g_mutex_lock( jukebox->priv->job_queue_mutex );
			g_signal_emit( G_OBJECT( jukebox ),
				       nomad_jukebox_signals[ ERROR_SIGNAL ], 0,
				       jobid, errmsg );
			g_mutex_unlock( jukebox->priv->job_queue_mutex );
			
			g_free( errmsg );
		} else {
			gchar *pname;
			gchar *newname;

			pname = g_hash_table_lookup( jukebox->priv->playlistids,
						     list->data );
			g_assert( pname );

			g_hash_table_remove( jukebox->priv->playlistids,
					     list->data );

			list->data = GUINT_TO_POINTER( playlist->plid );

			newname = g_strdup( name );
			g_hash_table_insert( jukebox->priv->playlistids,
					     list->data,
					     newname );

			g_hash_table_remove( jukebox->priv->playlists,
					     (gpointer)orig );
			
			g_hash_table_insert( jukebox->priv->playlists,
					     newname,
					     list );
			
			g_mutex_lock( jukebox->priv->job_queue_mutex );
			g_signal_emit( G_OBJECT( jukebox ),
				       nomad_jukebox_signals[PLAYLIST_RENAMED_SIGNAL ],
				       0, jobid, orig, newname );
			g_mutex_unlock( jukebox->priv->job_queue_mutex );

			/* pname is freed by its removal from
			 * the playlists hash table */
		}
	}
}

guint nomad_jukebox_add_tracks_to_playlist( NomadJukebox *jukebox,
                                              const gchar *name,
                                              GList *songidlist )
{
	g_return_val_if_fail( name != NULL, 0 );
	g_return_val_if_fail( songidlist != NULL, 0 );
	
	return nomad_jukebox_add_job( jukebox, ADD_TO_PLAYLIST,
				  g_strdup( name ),
				  g_list_copy( songidlist ), NULL, 0 );
}

static void
nomad_jukebox_add_tracks_to_playlist_real( NomadJukebox *jukebox, 
		guint jobid, const gchar *name, GList *songidlist )
{
        njb_playlist_track_t *pl_track;
        njb_playlist_t *playlist;
	guint playlistid;

	GList *list;
	GSList *tracks;
        guint plid;

        g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
        g_return_if_fail( jukebox->priv->njb != NULL );
        g_return_if_fail( jukebox->priv->connected );
        g_return_if_fail( name != NULL );
        g_return_if_fail( songidlist != NULL );
	
	tracks = g_hash_table_lookup( jukebox->priv->playlists, name );
	g_assert( tracks );
	playlistid = GPOINTER_TO_UINT( tracks->data );

	playlist = nomad_jukebox_fetch_playlist( jukebox, name );

	tracks = g_slist_reverse( tracks );

	for( list = songidlist; list; list = list->next ) {
		pl_track = NJB_Playlist_Track_New( GPOINTER_TO_UINT(list->data) );
		NJB_Playlist_Addtrack( playlist, pl_track, NJB_PL_END );
	}
	plid = NJB_Update_Playlist( jukebox->priv->njb, playlist );
	if( plid != -1 ) {
		NomadTrack *track;
		
		for( list = songidlist; list; list = list->next ) {
			track = (NomadTrack*)g_hash_table_lookup( jukebox->priv->tracks, list->data );
			g_assert( track );
			tracks = g_slist_prepend( tracks, track );
		}

		tracks = g_slist_reverse( tracks );
		tracks->data = GUINT_TO_POINTER( plid );

		name = g_hash_table_lookup( jukebox->priv->playlistids, 
					    GUINT_TO_POINTER( playlistid ) );
		g_assert( name );
		g_hash_table_remove( jukebox->priv->playlistids,
				     GUINT_TO_POINTER( playlistid ) );
		g_hash_table_insert( jukebox->priv->playlistids, 
				     GUINT_TO_POINTER( plid ),
				     (gpointer)name );
		g_hash_table_replace( jukebox->priv->playlists, (gpointer)name,
				      tracks );
	} else {
		gchar *errmsg;
		
		errmsg = g_strdup_printf( "Failed to add tracks to playlist %s",
					  name );
		
		g_mutex_lock( jukebox->priv->job_queue_mutex );
		g_signal_emit( G_OBJECT( jukebox ),
			       nomad_jukebox_signals[ ERROR_SIGNAL ], 0,
			       jobid, errmsg );
		g_mutex_unlock( jukebox->priv->job_queue_mutex );
		
		g_free( errmsg );
	}

	NJB_Playlist_Destroy( playlist );
}

const gchar *nomad_jukebox_get_ownerstring( NomadJukebox *jukebox )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), NULL );
	g_return_val_if_fail( jukebox->priv->njb != NULL, NULL );
	g_return_val_if_fail( jukebox->priv->connected, NULL );

	return jukebox->priv->owner;
}

guint nomad_jukebox_set_ownerstring( NomadJukebox *jukebox,
				       const gchar *owner )
{
	g_return_val_if_fail( owner != NULL, 0 );

	return nomad_jukebox_add_job( jukebox, SET_OWNER_STRING,
				  g_strdup( owner ), NULL, NULL, 0 );
}

static void nomad_jukebox_set_ownerstring_real( NomadJukebox *jukebox, 
		guint jobid, const gchar *owner )
{
	gint len;

	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->priv->njb != NULL );
	g_return_if_fail( jukebox->priv->connected );
	g_return_if_fail( owner != NULL );

	g_free( jukebox->priv->owner );
	jukebox->priv->owner = g_strdup( owner );

	len = strlen( jukebox->priv->owner );
	if( len >= OWNER_STRING_LENGTH ) {
		gchar *temp;

		temp = g_strndup( jukebox->priv->owner,
				  OWNER_STRING_LENGTH - 1 );
		g_free( jukebox->priv->owner );
		jukebox->priv->owner = temp;
	}

	if( NJB_Set_Owner_String( jukebox->priv->njb, jukebox->priv->owner ) == -1 ) {
		gchar *errmsg;
		
		errmsg = g_strdup_printf( "Failed to set owner string" );
		
		g_mutex_lock( jukebox->priv->job_queue_mutex );
		g_signal_emit( G_OBJECT( jukebox ),
			       nomad_jukebox_signals[ ERROR_SIGNAL ], 0,
			       jobid, errmsg );
		g_mutex_unlock( jukebox->priv->job_queue_mutex );
		
		g_free( errmsg );
	}
}

guint nomad_jukebox_getusage( NomadJukebox *jukebox )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );

	return nomad_jukebox_add_job( jukebox, GET_USAGE,
				  NULL, NULL, NULL, 0 );
}

static void nomad_jukebox_getusage_real( NomadJukebox *jukebox, 
		guint jobid )
{
	guint64 total;
	guint64 free;
	guint64 used;
	guint songs;
	guint playlists;
	guint datafiles;

	u_int64_t totalbytes;
	u_int64_t freebytes;
	
	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->priv->njb != NULL );
	g_return_if_fail( jukebox->priv->connected );
	
	if(NJB_Get_Disk_Usage( jukebox->priv->njb, &totalbytes, &freebytes ) != -1) {
		
		total = totalbytes;
		free = freebytes;
		used = totalbytes - freebytes;

		songs = -1;
		playlists = -1;
		datafiles = -1;
		
		g_mutex_lock( jukebox->priv->job_queue_mutex );
		g_signal_emit( G_OBJECT( jukebox ),
			       nomad_jukebox_signals[ USAGE_SIGNAL ], 0,
			       jobid, total, free, used );
		g_mutex_unlock( jukebox->priv->job_queue_mutex );
	} else {
		gchar *errmsg;
		
		errmsg = g_strdup_printf( "Failed to get usage information" );
		g_mutex_lock( jukebox->priv->job_queue_mutex );
		g_signal_emit( G_OBJECT( jukebox ),
			       nomad_jukebox_signals[ ERROR_SIGNAL ], 0,
			       jobid, errmsg );
		g_mutex_unlock( jukebox->priv->job_queue_mutex );
		
		g_free( errmsg );
	}
}

guint nomad_jukebox_set_volume( NomadJukebox *jukebox, guint8 volume )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );

	return nomad_jukebox_add_job( jukebox, SET_EAX,
			GINT_TO_POINTER( NJB_SOUND_SET_VOLUME ),
			GUINT_TO_POINTER( (guint)volume  ),
			NULL, 0 );
}

guint nomad_jukebox_set_muting( NomadJukebox *jukebox, 
				  gboolean mutingstatus )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );

	return nomad_jukebox_add_job( jukebox, SET_EAX,
			GINT_TO_POINTER( NJB_SOUND_SET_MUTING ),
			GINT_TO_POINTER( (guint)mutingstatus  ),
			NULL, 0 );
}

guint nomad_jukebox_set_eq_active( NomadJukebox *jukebox, 
				     gboolean eqactive )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );
	
	return nomad_jukebox_add_job( jukebox, SET_EAX,
			GINT_TO_POINTER( NJB_SOUND_SET_EQSTATUS ),
			GINT_TO_POINTER( (gint)eqactive  ),
			NULL, 0 );
}

guint nomad_jukebox_set_bass( NomadJukebox *jukebox, gint8 bass )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );

	return nomad_jukebox_add_job( jukebox, SET_EAX,
			GINT_TO_POINTER( NJB_SOUND_SET_BASS ),
			GINT_TO_POINTER( (gint)bass ),
			NULL, 0 );
}

guint nomad_jukebox_set_midrange( NomadJukebox *jukebox, gint8 midrange )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );

	return nomad_jukebox_add_job( jukebox, SET_EAX,
			GINT_TO_POINTER( NJB_SOUND_SET_MIDRANGE ),
			GINT_TO_POINTER( (gint)midrange ),
			NULL, 0 );
}

guint nomad_jukebox_set_treble( NomadJukebox *jukebox, gint8 treble )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );

	return nomad_jukebox_add_job( jukebox, SET_EAX,
			GINT_TO_POINTER( NJB_SOUND_SET_TREBLE ),
			GINT_TO_POINTER( (gint)treble ),
			NULL, 0 );
}

guint nomad_jukebox_set_effamt( NomadJukebox *jukebox, gint8 effamt )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );

	return nomad_jukebox_add_job( jukebox, SET_EAX,
			GINT_TO_POINTER( NJB_SOUND_SET_EAXAMT ),
			GINT_TO_POINTER( (gint)effamt ),
			NULL, 0 );
}

guint nomad_jukebox_set_midfreq( NomadJukebox *jukebox, gint8 freq )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );

	return nomad_jukebox_add_job( jukebox, SET_EAX,
			GINT_TO_POINTER( NJB_SOUND_SET_MIDFREQ ),
			GINT_TO_POINTER( (gint)freq ),
			NULL, 0 );
}

guint nomad_jukebox_set_effect( NomadJukebox *jukebox, gint8 effect )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );

	return nomad_jukebox_add_job( jukebox, SET_EAX,
			GINT_TO_POINTER( NJB_SOUND_SET_EAX ),
			GINT_TO_POINTER( (gint)effect ),
			NULL, 0 );
}

guint nomad_jukebox_set_headphone( NomadJukebox *jukebox, gint8 hpmode )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );

	return nomad_jukebox_add_job( jukebox, SET_EAX,
			GINT_TO_POINTER( NJB_SOUND_SET_HEADPHONE ),
			GINT_TO_POINTER( (gint)hpmode ),
			NULL, 0 );
}

guint nomad_jukebox_set_rearmode( NomadJukebox *jukebox, gint8 rearmode )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );

	return nomad_jukebox_add_job( jukebox, SET_EAX,
			GINT_TO_POINTER( NJB_SOUND_SET_REAR ),
			GINT_TO_POINTER( (gint)rearmode  ),
			NULL, 0 );
}

static void nomad_jukebox_eax_set( NomadJukebox *jukebox, guint jobid,
		gint type, gint8 value )
{
	NomadJukeboxPrivate *priv;
	const NJBEffects *effect;
	NJBEffects *leffect;
	guint16 id;
	guint16 setting;
	gint16 slider;
	gchar *errmsg;
	
	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->priv->njb != NULL );
	g_return_if_fail( jukebox->priv->connected );

	g_return_if_fail( type >= 0 && type < NJB_SOUND_NUM_EFFECTS );
	g_return_if_fail( jukebox->priv->njb->device_type < MAX_MODEL_NUM );
	
	priv = jukebox->priv;
	
	effect = njb_effects[ priv->njb->device_type ];

	errmsg = NULL;
	
	if( effect ) {
		g_return_if_fail( (gint16)value >= effect[ type ].min );
		g_return_if_fail( (gint16)value <= effect[ type ].max );
		
		id = effect[ type ].id;
		setting = 0;
		slider = 0;
		switch( effect[ type ].type ) {
			case NJB_EAX_SLIDER_CONTROL:
				slider = value;
				break;
			case NJB_EAX_FIXED_OPTION_CONTROL:
				setting = value;
				break;
			default:
				/* shouldn't occur */
				g_warning( "Unknown control type\n" );
				break;
		}
	
		NJB_Adjust_EAX( priv->njb, id, setting, slider );

		/* update our store of eax settings */
		leffect = priv->eax;
		leffect[ type ].current = value;
	} else {
		errmsg = "EAX feature support not available for your jukebox model";
	}

	if( errmsg ) {
		g_mutex_lock( priv->job_queue_mutex );
		g_signal_emit( G_OBJECT( jukebox ),
				nomad_jukebox_signals[ ERROR_SIGNAL ], 
				0, jobid, errmsg );
		g_mutex_unlock( priv->job_queue_mutex );
	}
}

guint nomad_jukebox_upload( NomadJukebox *jukebox, GList *list )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );

	return nomad_jukebox_add_job( jukebox, UPLOAD,
			g_list_copy( list ), NULL, NULL, 0 );
}

static void nomad_jukebox_upload_real( NomadJukebox *jukebox, 
		guint jobid, GList *list )
{
	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->priv->njb != NULL );
	g_return_if_fail( jukebox->priv->connected );
	
	jukebox->priv->abort = 0;
	while( ! jukebox->priv->abort && list ) {
		NomadTrack *track;
		njb_songid_t *songid;

		g_free( jukebox->priv->transfer_file );
		jukebox->priv->transfer_file = NULL;
		
		track = (NomadTrack*)list->data;

		if( track ) {
			songid = nomad_jukebox_songid_from_track( track );
			jukebox->priv->transfer_file = g_strdup( track->name );
			
			if( NJB_Send_Track( jukebox->priv->njb, 
					    jukebox->priv->transfer_file,
					    songid,
					    nomad_jukebox_xfer_cb,
					    jukebox, &track->id ) != -1 ) {
				guint total;
				/* insert a copy of the
				   NomadTrack into tracks hashtable */
				track = nomad_track_copy( track );
				g_hash_table_insert( jukebox->priv->tracks,
						     GUINT_TO_POINTER( track->id ),
						     track );
				total = g_hash_table_size( jukebox->priv->tracks );
				/* add to album / artist / genre stores */
				g_mutex_lock( jukebox->priv->job_queue_mutex );
				g_signal_emit( G_OBJECT( jukebox ),
					       nomad_jukebox_signals[ TRACK_ADD_SIGNAL ],
					       0, jobid, total, track );
				g_mutex_unlock( jukebox->priv->job_queue_mutex );
			} else {
				gchar *errmsg;
				
				errmsg = g_strdup_printf( "Failed to transfer %s to jukebox",
							  jukebox->priv->transfer_file );
				g_print( "ERR: %s\n", errmsg );
				g_print( "MSG: %s\n",
						NJB_Error_Geterror( jukebox->priv->njb ) );
				g_mutex_lock( jukebox->priv->job_queue_mutex );
				g_signal_emit( G_OBJECT( jukebox ),
					       nomad_jukebox_signals[ ERROR_SIGNAL ], 0,
					       jobid, errmsg );
				g_mutex_unlock( jukebox->priv->job_queue_mutex );
			
				g_free( errmsg );
			}
		}
		list = list->next;
	}
	/* signal uploading complete */
	g_mutex_lock( jukebox->priv->job_queue_mutex );
	g_signal_emit( G_OBJECT( jukebox ),
		       nomad_jukebox_signals[ UPLOAD_COMPLETE_SIGNAL ],
		       0, jobid );
	g_mutex_unlock( jukebox->priv->job_queue_mutex );
}


guint nomad_jukebox_download( NomadJukebox *jukebox, 
				const gchar *dest,
				const gchar *format,
				GList *list )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );
	g_return_val_if_fail( dest != NULL, 0 );
	g_return_val_if_fail( format != NULL, 0 );
	g_return_val_if_fail( list != NULL, 0 );
	
	return nomad_jukebox_add_job( jukebox, DOWNLOAD,
			g_strdup( dest ), g_strdup( format ),
			g_list_copy( list ), 0 );
}

static void nomad_jukebox_download_real( NomadJukebox *jukebox, 
		guint jobid, const gchar *dest, const gchar *format,
		GList *list )
{
	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->priv->njb != NULL );
	g_return_if_fail( jukebox->priv->connected );
	
	jukebox->priv->abort = 0;
	while( ! jukebox->priv->abort && list ) {
		NomadTrack *daptrack;	
		gchar *dummy;
		u_int32_t size;
		gchar *path;
		
		daptrack = nomad_jukebox_request_track( jukebox, 
				GPOINTER_TO_UINT( list->data ) );
		g_assert( daptrack );

		size = strtoul( daptrack->size, &dummy, 10 );
	
		g_free( jukebox->priv->transfer_file );
		jukebox->priv->transfer_file = NULL;
		
		path = nomad_jukebox_filename_format( dest, format,
							daptrack );
		jukebox->priv->transfer_file = path;
		jukebox->priv->track = daptrack;
		
		jukebox->priv->handle = nomad_create( jukebox->priv->transfer_file );
		if( jukebox->priv->handle && 
		    NJB_Get_Track( jukebox->priv->njb, daptrack->id, size, 
				NULL, nomad_jukebox_xfer_cb,
				jukebox ) != -1 ) {
			/* got */
			g_mutex_lock( jukebox->priv->job_queue_mutex );
			g_signal_emit( G_OBJECT( jukebox ),
				       nomad_jukebox_signals[ DOWNLOADED_TRACK_SIGNAL ],
				       0, jobid, jukebox->priv->transfer_file,
				       daptrack );
			g_mutex_unlock( jukebox->priv->job_queue_mutex );
		} else {
			gchar *errmsg;
			gchar *trackname;
			
			trackname = nomad_track_get_title( daptrack );

			errmsg = g_strdup_printf( "Failed to transfer %s from the jukebox",
						  trackname );
			g_print( "ERR: %s\nMSG: %s\n", errmsg,
					NJB_Error_Geterror( jukebox->priv->njb ) );
			g_mutex_lock( jukebox->priv->job_queue_mutex );
			g_signal_emit( G_OBJECT( jukebox ),
				       nomad_jukebox_signals[ ERROR_SIGNAL ], 0,
				       jobid, errmsg );
			g_mutex_unlock( jukebox->priv->job_queue_mutex );
			
			g_free( errmsg );
			g_free( trackname );
		}
		if( jukebox->priv->handle ) {
			nomad_close( jukebox->priv->handle );
		}
		nomad_track_free( daptrack );
		jukebox->priv->track = NULL;

		list = list->next;
	}
	/* signal downloading complete */
	g_mutex_lock( jukebox->priv->job_queue_mutex );
	g_signal_emit( G_OBJECT( jukebox ),
		       nomad_jukebox_signals[ DOWNLOAD_COMPLETE_SIGNAL ],
		       0, jobid );
	g_mutex_unlock( jukebox->priv->job_queue_mutex );
}

guint nomad_jukebox_play_track( NomadJukebox *jukebox, guint trackid )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );

	return nomad_jukebox_add_job( jukebox, PLAY_TRACK,
			GUINT_TO_POINTER( trackid ), NULL, NULL, 0 );
}

static void nomad_jukebox_play_track_real( NomadJukebox *jukebox, 
		guint jobid, guint trackid )
{
	NomadTrack *track;

	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->priv->njb != NULL );
	g_return_if_fail( jukebox->priv->connected );

	track = g_hash_table_lookup( jukebox->priv->tracks, GUINT_TO_POINTER( trackid ) );
	g_assert( track );

	NJB_Stop_Play( jukebox->priv->njb );

	jukebox->priv->track = track;

	if( NJB_Play_Track( jukebox->priv->njb, track->id ) != -1 ) {
		g_mutex_lock( jukebox->priv->job_queue_mutex );
		g_signal_emit( G_OBJECT( jukebox ),
			       nomad_jukebox_signals[ PLAY_BEGIN_SIGNAL ],
			       0, jobid, track );
		g_mutex_unlock( jukebox->priv->job_queue_mutex );
	} else {
		gchar *errmsg;
		gchar *trackname;
		
		trackname = nomad_track_get_title( track );
		
		errmsg = g_strdup_printf( "Failed to play %s",
					  trackname );
		g_mutex_lock( jukebox->priv->job_queue_mutex );
		g_signal_emit( G_OBJECT( jukebox ),
			       nomad_jukebox_signals[ ERROR_SIGNAL ], 0,
			       jobid, errmsg );
		g_mutex_unlock( jukebox->priv->job_queue_mutex );
		
		g_free( errmsg );
		g_free( trackname );
	}
}

guint nomad_jukebox_play_elapsed( NomadJukebox *jukebox )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );

	return nomad_jukebox_add_job( jukebox, PLAY_ELAPSED, NULL, NULL,
			NULL, 0 );
}

static void nomad_jukebox_play_elapsed_real( NomadJukebox *jukebox, 
		guint jobid )
{
	guint16 elapsed;
	gint changed;

	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->priv->njb != NULL );
	g_return_if_fail( jukebox->priv->connected );

	if( NJB_Elapsed_Time( jukebox->priv->njb, &elapsed, &changed ) != 0 ) {
		changed = 1;
		elapsed = 0;
	}
	g_mutex_lock( jukebox->priv->job_queue_mutex );
	g_signal_emit( G_OBJECT( jukebox ),
		       nomad_jukebox_signals[ PLAY_PROGRESS_SIGNAL ],
		       0, jobid, elapsed, changed, jukebox->priv->track );
	g_mutex_unlock( jukebox->priv->job_queue_mutex );
}

guint nomad_jukebox_play_stop( NomadJukebox *jukebox )
{
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );

	return nomad_jukebox_add_job( jukebox, PLAY_STOP, NULL, NULL,
			NULL, 0 );
}

static void nomad_jukebox_play_stop_real( NomadJukebox *jukebox, 
		guint jobid )
{
	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->priv->njb != NULL );
	g_return_if_fail( jukebox->priv->connected );

	NJB_Stop_Play( jukebox->priv->njb );

	jukebox->priv->track = NULL;

	g_mutex_lock( jukebox->priv->job_queue_mutex );
	g_signal_emit( G_OBJECT( jukebox ),
		       nomad_jukebox_signals[ PLAY_FINISHED_SIGNAL ],
		       0, jobid );
	g_mutex_unlock( jukebox->priv->job_queue_mutex );
}


void nomad_jukebox_abort( NomadJukebox *jukebox )
{
	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );

	jukebox->priv->abort = -1;
}

gint16 nomad_jukebox_get_volume( NomadJukebox *jukebox,
		gint16 *min, gint16 *max )
{
	const NJBEffects *effect;
	gint16 ret;
	
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );
	g_return_val_if_fail( jukebox->priv->njb != NULL, 0 );
	g_return_val_if_fail( jukebox->priv->connected, 0 );

	effect = &jukebox->priv->eax[ NJB_SOUND_SET_VOLUME ];
	
	ret = ( effect->current );
	
	if( min ) {
		*min = effect->min;
	}
	if( max ) {
		*max = effect->max;
	}
	
	return ret;
}

gboolean nomad_jukebox_get_muting( NomadJukebox *jukebox )
{
	gint16 ret;
	
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );
	g_return_val_if_fail( jukebox->priv->njb != NULL, 0 );
	g_return_val_if_fail( jukebox->priv->connected, 0 );

	ret = jukebox->priv->eax[ NJB_SOUND_SET_MUTING ].current;

	return ( ret != 0 );
}

gboolean nomad_jukebox_get_eq_active( NomadJukebox *jukebox )
{
	gint16 ret;
	
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );
	g_return_val_if_fail( jukebox->priv->njb != NULL, 0 );
	g_return_val_if_fail( jukebox->priv->connected, 0 );

	ret = jukebox->priv->eax[ NJB_SOUND_SET_EQSTATUS ].current;

	return ( ret != 0 );
}

gint16 nomad_jukebox_get_bass( NomadJukebox *jukebox,
		gint16 *min, gint16 *max )
{
	const NJBEffects *effect;
	gint16 ret;
	
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );
	g_return_val_if_fail( jukebox->priv->njb != NULL, 0 );
	g_return_val_if_fail( jukebox->priv->connected, 0 );

	effect = &jukebox->priv->eax[ NJB_SOUND_SET_BASS ];
	
	ret = effect->current;

	if( min ) {
		*min = effect->min;
	}
	if( max ) {
		*max = effect->max;
	}
	
	return ret;
}

gint16 nomad_jukebox_get_midrange( NomadJukebox *jukebox,
		gint16 *min, gint16 *max )
{
	const NJBEffects *effect;
	gint16 ret;
	
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );
	g_return_val_if_fail( jukebox->priv->njb != NULL, 0 );
	g_return_val_if_fail( jukebox->priv->connected, 0 );

	effect = &jukebox->priv->eax[ NJB_SOUND_SET_MIDRANGE ];
	
	ret = effect->current;

	if( min ) {
		*min = effect->min;
	}
	if( max ) {
		*max = effect->max;
	}
	
	return ret;
}

gint16 nomad_jukebox_get_treble( NomadJukebox *jukebox,
		gint16 *min, gint16 *max )
{
	const NJBEffects *effect;
	gint16 ret;
	
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );
	g_return_val_if_fail( jukebox->priv->njb != NULL, 0 );
	g_return_val_if_fail( jukebox->priv->connected, 0 );

	effect = &jukebox->priv->eax[ NJB_SOUND_SET_TREBLE ];
	
	ret = effect->current;

	if( min ) {
		*min = effect->min;
	}
	if( max ) {
		*max = effect->max;
	}
	
	return ret;
}

gint16 nomad_jukebox_get_effamt( NomadJukebox *jukebox,
		gint16 *min, gint16 *max )
{
	const NJBEffects *effect;
	gint16 ret;
	
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );
	g_return_val_if_fail( jukebox->priv->njb != NULL, 0 );
	g_return_val_if_fail( jukebox->priv->connected, 0 );

	effect = &jukebox->priv->eax[ NJB_SOUND_SET_EAXAMT ];
	
	ret = effect->current;

	if( min ) {
		*min = effect->min;
	}
	if( max ) {
		*max = effect->max;
	}

	return ret;
}

gint16 nomad_jukebox_get_midfreq( NomadJukebox *jukebox,
		const gchar ***names )
{
	const NJBEffects *effect;
	gint16 ret;
	
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );
	g_return_val_if_fail( jukebox->priv->njb != NULL, 0 );
	g_return_val_if_fail( jukebox->priv->connected, 0 );

	effect = &jukebox->priv->eax[ NJB_SOUND_SET_MIDFREQ ];

	ret = effect->current;
	
	if( names ) {
		*names = (const gchar**)effect->names;
	}
	
	return ret;
}

gint16 nomad_jukebox_get_effect( NomadJukebox *jukebox,
		const gchar ***names )
{
	const NJBEffects *effect;
	gint16 ret;
	
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );
	g_return_val_if_fail( jukebox->priv->njb != NULL, 0 );
	g_return_val_if_fail( jukebox->priv->connected, 0 );

	effect = &jukebox->priv->eax[ NJB_SOUND_SET_EAX ];

	ret = effect->current;
	
	if( names ) {
		*names = (const gchar**)effect->names;
	}
	
	return ret;
}

gint16 nomad_jukebox_get_headphone( NomadJukebox *jukebox,
		const gchar ***names )
{
	const NJBEffects *effect;
	gint16 ret;
	
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );
	g_return_val_if_fail( jukebox->priv->njb != NULL, 0 );
	g_return_val_if_fail( jukebox->priv->connected, 0 );

	effect = &jukebox->priv->eax[ NJB_SOUND_SET_HEADPHONE ];

	ret = effect->current;
	
	if( names ) {
		*names = (const gchar**)effect->names;
	}
	
	return ret;
}

gint16 nomad_jukebox_get_rearmode( NomadJukebox *jukebox,
		const gchar ***names )
{
	const NJBEffects *effect;
	gint16 ret;
	
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );
	g_return_val_if_fail( jukebox->priv->njb != NULL, 0 );
	g_return_val_if_fail( jukebox->priv->connected, 0 );

	effect = &jukebox->priv->eax[ NJB_SOUND_SET_REAR ];

	ret = effect->current;
	
	if( names ) {
		*names = (const gchar**)effect->names;
	}
	
	return ret;
}

GHashTable* nomad_jukebox_request_tracks( NomadJukebox *jukebox )
{
	return jukebox->priv->tracks;
}

GHashTable *nomad_jukebox_request_playlists( NomadJukebox *jukebox )
{
	return jukebox->priv->playlists;
}

NomadTrack *nomad_jukebox_request_track( NomadJukebox *jukebox,
					       guint id )
{
	NomadTrack *track;
	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), NULL );

	track = (NomadTrack*)
		g_hash_table_lookup( jukebox->priv->tracks, 
				     GUINT_TO_POINTER( id ) );
	if( track ) {
		track = nomad_track_copy( track );
	}

	return track;
}

/* static stuff */
static void nomad_jukebox_refresh_id( NomadJukebox *jukebox )
{
	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	g_return_if_fail( jukebox->priv->njb != NULL );
	g_return_if_fail( jukebox->priv->connected );

	NJB_Ping( jukebox->priv->njb );
}

static int nomad_jukebox_xfer_cb( u_int64_t sent, u_int64_t total,
		      const char *buf, unsigned int len, void *data )
{
	NomadJukebox *jukebox;
	NomadTrack *track;
	int ret;
	gchar tag[ 128 ];
	gboolean mp3;
	guint newlen;
	
	jukebox = NOMAD_JUKEBOX( data );
	
	track = jukebox->priv->track;
	ret = jukebox->priv->abort;

	if( jukebox->priv->handle && ! ret ) {
		mp3 = FALSE;
		
		if( ( sent == total || 
		      nomad_tell( jukebox->priv->handle ) == 0 ) && 
				ret != -1 ) {
			mp3 = ( ! strcmp( track->codec,
					NJB_CODEC_MP3 ) );
			jukebox->priv->id3len = 0;
		} 
		if( mp3 && sent != total ) {
			if( ! strncmp( "ID3", buf, 3 ) ) {
				if( len < ID3V2HeaderSize ) {
					g_warning( "not enough data to strip ID3" );
				} else {
					jukebox->priv->id3len = ID3V2HeaderSize;
					newlen = nomad_id3_get_length_from_header( (ID3V2Header*)buf );
					jukebox->priv->id3len += newlen;

				}
			}
			/* ID3 v2 tag */
			nomad_id3_tag_v2_with_handle( jukebox->priv->handle,
					track->title,
					track->artist,
					track->album,
					track->year,
					track->track,
					track->genre,
					track->length );
		}
		
		if( jukebox->priv->id3len > 0 )  {
			if( jukebox->priv->id3len < len ) {
				len -= jukebox->priv->id3len;
				buf += jukebox->priv->id3len;
				jukebox->priv->id3len = 0;
			} else {
				jukebox->priv->id3len -= len;
				len = 0;
			}
		}
		if( len > 0 ) {
			if( ! nomad_write( jukebox->priv->handle, buf, 
						len ) ) {
				ret = -1;
			}
		} 
		/* id3v1 tag */
		if( mp3 && sent == total ) {
			/* remove possible current tag */
			if( nomad_seek( jukebox->priv->handle, 
						SEEK_END, -128 ) &&
			    nomad_read( jukebox->priv->handle, tag, 128 ) ) {
				if( ! strncmp( "TAG", tag, 3 ) ) {
					nomad_seek( jukebox->priv->handle,
							SEEK_END,
							-128 );
				}
			}
			/* retag */
			nomad_id3_tag_v1dot1_with_handle( jukebox->priv->handle,
					track->title, track->artist,
					track->album, track->year,
					track->track, track->genre );
		}
	} 

	g_mutex_lock( jukebox->priv->job_queue_mutex );
	g_signal_emit( G_OBJECT( jukebox ),
		       nomad_jukebox_signals[ TRANSFER_SIGNAL ], 0,
		       jukebox->priv->jobid, 
		       jukebox->priv->transfer_file, buf == NULL,
		       sent, total );
	g_mutex_unlock( jukebox->priv->job_queue_mutex );
	
	return ret;
}


static gpointer nomad_jukebox_transfer_thread( gpointer data )
{
	NomadJukebox *jukebox;
	GQueue *jobs;
	gboolean terminate;

	jukebox = NOMAD_JUKEBOX( data );
	jobs = jukebox->priv->job_queue;

	terminate = FALSE;

	while( ! terminate ) {
		NomadJukeboxJob *job;

		/* wait till we have something to do */
		sem_wait( &jukebox->priv->transfer_semaphore );

		/* pull job off the queue */
		g_mutex_lock( jukebox->priv->job_queue_mutex );

		job = (NomadJukeboxJob *)g_queue_pop_head( jobs );
		
		g_mutex_unlock( jukebox->priv->job_queue_mutex );

		/* hmm, shouldn't happen, but does! */
		if( ! job ) {
			g_warning( "NO JOB!\n" );
			continue;
		}
		
		jukebox->priv->jobid = job->id;
		
		nomad_jukebox_refresh_id( jukebox );

		/* process job */
		switch( job->type ) {
		case GET_TIME:
			nomad_jukebox_get_time_real( jukebox, job->id );
			break;
		case BUILD_TRACKLIST:
			nomad_jukebox_build_tracklist_real( jukebox,
					job->id );
			break;
	
		case SET_OWNER_STRING:
			nomad_jukebox_set_ownerstring_real( jukebox,
					job->id, job->data );
			nomad_jukebox_getusage_real( jukebox, job->id );
			g_free( job->data );
		case GET_USAGE:
			nomad_jukebox_getusage_real( jukebox, job->id );
			break;
		case UPLOAD:
			nomad_jukebox_upload_real( jukebox, job->id,
					job->data );
			nomad_jukebox_getusage_real( jukebox, job->id );
			g_list_free( job->data );
			break;
		case DOWNLOAD:
			nomad_jukebox_download_real( jukebox, job->id,
					job->data, job->data2,
					job->data3 );
			g_free( job->data );
			g_free( job->data2 );
			g_list_free( job->data3 );
			break;
	
		case DELETE_TRACKS:
			nomad_jukebox_delete_tracks_real( jukebox,
					job->id, job->data );
			nomad_jukebox_getusage_real( jukebox, job->id );
			g_list_free( job->data );
			break;
		case SET_METADATA:
			nomad_jukebox_set_metadata_real( jukebox,
					job->id, job->data );
			/* job->data is either already free, or
			   it is in use */
			break;
		case SET_EAX:
			nomad_jukebox_eax_set( jukebox, job->id,
					GPOINTER_TO_INT( job->data ),
					GPOINTER_TO_INT( job->data2 ) );
			break;
		case PLAY_TRACK:
			nomad_jukebox_play_track_real( jukebox,
					job->id,
					GPOINTER_TO_UINT( job->data ) );
			break;
		case PLAY_ELAPSED:
			nomad_jukebox_play_elapsed_real( jukebox,
					job->id );
			break;
		case PLAY_STOP:
			nomad_jukebox_play_stop_real( jukebox, 
					job->id );
			break;
		case TERMINATE:
			terminate = TRUE;
			break;

		case BUILD_PLAYLIST:
			nomad_jukebox_build_playlist_real( jukebox,
					job->id );
			break;
		case DELETE_TRACKS_FROM_PLAYLIST:
                        nomad_jukebox_delete_tracks_from_playlist_real( jukebox,
					job->id, job->data, 
					(const gchar*)job->data2 );
                        g_list_free( job->data );
			if( job->data2 ) {
				g_free( job->data2 );
			}
                        break;
                case CREATE_PLAYLIST:
                        nomad_jukebox_create_playlist_real( jukebox,
					job->id, job->data );
                        nomad_jukebox_getusage_real( jukebox, job->id );
                        g_free( job->data );
                        break;
                case DELETE_PLAYLIST:
                        nomad_jukebox_delete_playlist_real( jukebox,
					job->id,
					(const gchar*)job->data );
			g_free( job->data );
                        nomad_jukebox_getusage_real( jukebox, job->id );
                        break;
                case RENAME_PLAYLIST:
                        nomad_jukebox_rename_playlist_real( jukebox,
					job->id,
					(const gchar*)job->data,
					(const gchar*)job->data2 );
                        nomad_jukebox_getusage_real( jukebox, job->id );
			g_free( job->data );
                        g_free( job->data2 );
                        break;
                case ADD_TO_PLAYLIST:
                        nomad_jukebox_add_tracks_to_playlist_real( jukebox,
					job->id,
					(const gchar*)job->data,
					job->data2 );
                        nomad_jukebox_getusage_real( jukebox, job->id );
			g_free( job->data );
                        g_list_free( job->data2 );
                        break;
		default:
			if( job->type >= MAX_JOB_NUM ) {
				g_warning( "Invalid NomadJukeboxJobType: %i\n",
					   job->type );
			} else {
				g_warning( "Unimplemented NomadJukeboxJobType: %i\n",
					   job->type );
			}
			break;
		}
		g_free( job );
	}

	return NULL;
}

static gchar *nomad_jukebox_filename_format( const gchar *dest, 
						const gchar *format,
						NomadTrack *track )
{
	GString *string;
	const gchar *temp;
	gchar *ret;

	string = g_string_new( dest );
	g_string_append_c( string, '/' );

	for( temp = format; *temp; ++ temp ) {
		if( *temp == '%' ) {
			gchar *esc;

			temp ++;
			switch( *temp ) {
			case 'a': /* artist */
				esc = nomad_filename_escape(track->artist );
				g_string_append( string, esc );
				g_free( esc );
				break;
			case 't': /* title */
				esc = nomad_filename_escape( track->title );
				g_string_append( string, esc );
				g_free( esc );
				break;
			case 'b': /* album */
				esc = nomad_filename_escape( track->album );
				g_string_append( string, esc );
				g_free( esc );
				break;
			case 'g': /* genre */
				esc = nomad_filename_escape( track->genre );
				g_string_append( string, esc );
				g_free( esc );
				break;
			case 'y': /* year */
				esc = nomad_filename_escape( track->year );
				g_string_append( string, esc );
				g_free( esc );
				break;
			case 'n': /* track */		
				esc = nomad_filename_escape( track->track );
				g_string_append( string, esc );

				g_free( esc );
				break;
			default:
				temp --;
				break;
			}
		} else {
			g_string_append_c( string, *temp );
		}
	}

	/* append extension */
	if( ! strcmp( NJB_CODEC_MP3, track->codec ) ) {
		g_string_append( string, ".mp3" );
	} else if( ! strcmp( NJB_CODEC_WMA, track->codec ) ) {
		g_string_append( string, ".wma" );
	} else if( ! strcmp( NJB_CODEC_WAV, track->codec ) ) {
		g_string_append( string, ".wav" );
	}

	ret = string->str;

	g_string_free( string, FALSE );

	return ret;
}

static njb_playlist_t *nomad_jukebox_fetch_playlist( NomadJukebox *jukebox,
						    const gchar *name )
{
	njb_playlist_t *ret;

        g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), NULL );
        g_return_val_if_fail( jukebox->priv->njb != NULL, NULL );
        g_return_val_if_fail( jukebox->priv->connected, NULL );
        g_return_val_if_fail( name != NULL, NULL );

	NJB_Reset_Get_Playlist( jukebox->priv->njb );
	for( ret = NULL; ! ret; ) {
		ret = NJB_Get_Playlist( jukebox->priv->njb );
		if( ret ) {
			if( ! strcmp( name, ret->name ) ) {
				break;
			}
			NJB_Playlist_Destroy( ret );
			ret = NULL;
		}
	}

	return ret;
}

static guint nomad_jukebox_add_job( NomadJukebox *jukebox,
				      NomadJukeboxJobType type,
				      gpointer data,
				      gpointer data2,
				      gpointer data3,
				      guint priority )
{
	NomadJukeboxJob *job;
	GList *ins;
	NomadJukeboxJob *j;
	static guint curid = 1;

	g_return_val_if_fail( NOMAD_IS_JUKEBOX( jukebox ), 0 );
	g_return_val_if_fail( jukebox->priv->njb != NULL, 0 );
	g_return_val_if_fail( jukebox->priv->connected, 0 );

	curid ++;
	/* handle wrap round, 1 is a reserved id for fake jobs */
	if( curid < 2 ) {
		curid ++;
	}
	
	job = g_new0( NomadJukeboxJob, 1 );
	job->type = type;
	job->id = curid;
	job->data = data;
	job->data2 = data2;
	job->data3 = data3;
	job->priority = priority;

	g_mutex_lock( jukebox->priv->job_queue_mutex );

	/* insert in priority order */
	ins = jukebox->priv->job_queue->tail;
	while( ins ) {
		j = (NomadJukeboxJob*)ins->data;
		
		if( j->priority >= priority ) {
			/* insert here */
			GList *newitem;
			
			newitem = g_list_alloc();
			newitem->data = job;
			
			if( ins->next ) {
				ins->next->prev = newitem;
			} else {
				jukebox->priv->job_queue->tail = newitem;
			}
			newitem->next = ins->next;
			ins->next = newitem;
			newitem->prev = ins;
			
			jukebox->priv->job_queue->length ++;
			break;
		}
		ins = ins->prev;
	}
	
	if( ! ins ) {
		g_queue_push_head( jukebox->priv->job_queue, job );
	}

	g_mutex_unlock( jukebox->priv->job_queue_mutex );
	sem_post( &jukebox->priv->transfer_semaphore );

	return job->id;
}

static void nomad_jukebox_eax_init( NomadJukebox *jukebox )
{
	NomadJukeboxPrivate *priv;
	gint type;
	njb_eax_t *eax;
	NJBEffects *effect;
	gint i;
	gint min;
	gint max;
	gint size;
	gint j;
	
	g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
	
	priv = jukebox->priv;
	
	effect = priv->eax;
	memset( effect, 0, sizeof( NJBEffects ) * NJB_SOUND_NUM_EFFECTS);
	
	type = priv->njb->device_type;
	
	if( type >= 0 && type < MAX_MODEL_NUM && njb_effects[ type ] ) {
		memcpy( effect, njb_effects[ type ], sizeof( NJBEffects ) * NJB_SOUND_NUM_EFFECTS );
	
		/* fill in current values */
		NJB_Reset_Get_EAX_Type( priv->njb );
		while( ( eax = NJB_Get_EAX_Type( priv->njb ) ) ) {
			for( i = 0; i < NJB_SOUND_NUM_EFFECTS; ++ i ) {
				if( effect[ i ].id == eax->number ) {
					effect[ i ].current = eax->current_value;
					break;
				}
			}
			if( i == NJB_SOUND_NUM_EFFECTS ) {
				g_print( "Unknown effect %i, please provide output from the dumpeax sample application in libnjb / jukebox type\n", eax->number );
			} else if( eax->type == NJB_EAX_FIXED_OPTION_CONTROL ) {
				/* need labels for supported option controls, so make a copy of eax->option_names */
				min = effect[ i ].min;
				max = effect[ i ].max;
		
				size = max - min + 1;
				
				/* do size + 1 so we can ensure the
				 * array will end with a NULL, as
				 * if min is 0 then size will only be
				 * the exact number of entries */
				effect[ i ].names = g_new0( gchar*, size + 1 );
				for( j = 0; j < size; ++ j ) {
					effect[ i ].names[ j ] = g_strdup( eax->option_names[ j ] );
				}
			}
			NJB_Destroy_EAX_Type( eax );
		}
	}
}


/* G Object stuff */

static GObjectClass *parent_class = NULL;

GType nomad_jukebox_get_type( void )
{
	static GType type = 0;

	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( NomadJukeboxClass ),
			NULL, /* base init */
			NULL, /* base_finalize */
			(GClassInitFunc) nomad_jukebox_class_init,
			NULL, /* class_finalize */
			NULL, /* class_data */
			sizeof( NomadJukebox ),
			0, /* n_preallocs */
			(GInstanceInitFunc) nomad_jukebox_instance_init
		};

		type = g_type_register_static( G_TYPE_OBJECT,
				"NomadJukebox", &info, 0 );
	}

	return type;
}

static void
nomad_jukebox_class_init( NomadJukeboxClass *klass)
{
        GObjectClass *object_class = G_OBJECT_CLASS( klass );

	parent_class = g_type_class_peek_parent( klass );

	object_class->finalize = nomad_jukebox_finalize;
	object_class->get_property = nomad_jukebox_get_prop;
	object_class->set_property = nomad_jukebox_set_prop;

	nomad_jukebox_signals[ TIME_SIGNAL ] = 
		g_signal_new( "time",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       time ),
			      NULL, NULL,
			      nomad_marshal_VOID__UINT_STRING,
			      G_TYPE_NONE, 2,
			      G_TYPE_UINT,
			      G_TYPE_STRING );

	nomad_jukebox_signals[ USAGE_SIGNAL ] = 
		g_signal_new( "usage",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       usage ),
			      NULL, NULL,
			      nomad_marshal_VOID__UINT_UINT64_UINT64_UINT64,
			      G_TYPE_NONE, 4,
			      G_TYPE_UINT,
			      G_TYPE_UINT64,
			      G_TYPE_UINT64,
			      G_TYPE_UINT64 );

	nomad_jukebox_signals[ SCANNING_TRACKS_SIGNAL ] = 
		g_signal_new( "scanning_tracks",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       scanning_tracks ),
			      NULL, NULL,
			      nomad_marshal_VOID__UINT,
			      G_TYPE_NONE, 1,
			      G_TYPE_UINT );

	nomad_jukebox_signals[ SCANNED_TRACKS_SIGNAL ] = 
		g_signal_new( "scanned_tracks",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       scanned_tracks ),
			      NULL, NULL,
			      nomad_marshal_VOID__UINT_UINT,
			      G_TYPE_NONE, 2,
			      G_TYPE_UINT,
			      G_TYPE_UINT );

	nomad_jukebox_signals[ SCANNING_PLAYLISTS_SIGNAL ] = 
		g_signal_new( "scanning_playlists",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       scanning_playlists ),
			      NULL, NULL,
			      nomad_marshal_VOID__UINT,
			      G_TYPE_NONE, 1,
			      G_TYPE_UINT );

	nomad_jukebox_signals[ PLAYLIST_ADD_SIGNAL ] = 
		g_signal_new( "playlist_add",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       playlist_add ),
			      NULL, NULL,
			      nomad_marshal_VOID__UINT_UINT_STRING,
			      G_TYPE_NONE, 3,
			      G_TYPE_UINT,
			      G_TYPE_UINT,
			      G_TYPE_STRING );

	nomad_jukebox_signals[ SCANNED_PLAYLISTS_SIGNAL ] = 
		g_signal_new( "scanned_playlists",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       scanned_playlists ),
			      NULL, NULL,
			      nomad_marshal_VOID__UINT_UINT,
			      G_TYPE_NONE, 2,
			      G_TYPE_UINT,
			      G_TYPE_UINT );

	nomad_jukebox_signals[ TRANSFER_SIGNAL ] = 
		g_signal_new( "transfer",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       transfer ),
			      NULL, NULL,
			      nomad_marshal_VOID__UINT_STRING_BOOLEAN_UINT64_UINT64,
			      G_TYPE_NONE, 5,
			      G_TYPE_UINT,
			      G_TYPE_STRING,
			      G_TYPE_BOOLEAN,
			      G_TYPE_UINT64,
			      G_TYPE_UINT64 );

	nomad_jukebox_signals[ UPLOAD_COMPLETE_SIGNAL ] = 
		g_signal_new( "upload_complete",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       upload_complete ),
			      NULL, NULL,
			      nomad_marshal_VOID__UINT,
			      G_TYPE_NONE, 1,
			      G_TYPE_UINT );
	nomad_jukebox_signals[ DOWNLOAD_COMPLETE_SIGNAL ] = 
		g_signal_new( "download_complete",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       download_complete ),
			      NULL, NULL,
			      nomad_marshal_VOID__UINT,
			      G_TYPE_NONE, 1,
			      G_TYPE_UINT );
	nomad_jukebox_signals[ DOWNLOADED_TRACK_SIGNAL ] = 
		g_signal_new( "downloaded_track",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       downloaded_track ),
			      NULL, NULL,
			      nomad_marshal_VOID__UINT_STRING_POINTER,
			      G_TYPE_NONE, 3,
			      G_TYPE_UINT,
			      G_TYPE_STRING,
			      G_TYPE_POINTER );

	nomad_jukebox_signals[ PLAYLIST_TRACK_ERASED_SIGNAL ] = 
		g_signal_new( "playlist_track_erased",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       playlist_track_erased ),
			      NULL, NULL,
			      nomad_marshal_VOID__UINT_STRING_STRING_STRING_STRING,
			      G_TYPE_NONE, 5,
			      G_TYPE_UINT,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING );
	nomad_jukebox_signals[ PLAYLIST_TRACKS_ERASED_SIGNAL ] = 
		g_signal_new( "playlist_tracks_erased",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       playlist_tracks_erased ),
			      NULL, NULL,
			      nomad_marshal_VOID__UINT_STRING,
			      G_TYPE_NONE, 2,
			      G_TYPE_UINT,
			      G_TYPE_STRING );
	nomad_jukebox_signals[ PLAYLIST_REMOVE_SIGNAL ] = 
		g_signal_new( "playlist_remove",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       playlist_remove ),
			      NULL, NULL,
			      nomad_marshal_VOID__UINT_STRING,
			      G_TYPE_NONE, 2,
			      G_TYPE_UINT, G_TYPE_STRING );
	nomad_jukebox_signals[ PLAYLIST_RENAMED_SIGNAL ] = 
		g_signal_new( "playlist_renamed",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       playlist_renamed ),
			      NULL, NULL,
			      nomad_marshal_VOID__UINT_STRING_STRING,
			      G_TYPE_NONE, 3,
			      G_TYPE_UINT,
			      G_TYPE_STRING,
			      G_TYPE_STRING );
	nomad_jukebox_signals[ PLAY_BEGIN_SIGNAL ] = 
		g_signal_new( "play_begin",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       play_begin ),
			      NULL, NULL,
			      nomad_marshal_VOID__UINT_POINTER,
			      G_TYPE_NONE, 2,
			      G_TYPE_UINT,
			      G_TYPE_POINTER );
	nomad_jukebox_signals[ PLAY_PROGRESS_SIGNAL ] = 
		g_signal_new( "play_progress",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       play_progress ),
			      NULL, NULL,
			      nomad_marshal_VOID__UINT_UINT_BOOLEAN_POINTER,
			      G_TYPE_NONE, 4,
			      G_TYPE_UINT,
			      G_TYPE_UINT,
			      G_TYPE_BOOLEAN,
			      G_TYPE_POINTER );	
	nomad_jukebox_signals[ PLAY_FINISHED_SIGNAL ] = 
		g_signal_new( "play_finished",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       play_finished ),
			      NULL, NULL,
			      nomad_marshal_VOID__UINT,
			      G_TYPE_NONE, 1,
			      G_TYPE_UINT );

	nomad_jukebox_signals[ TRACK_ADD_SIGNAL ] = 
		g_signal_new( "track_add",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       track_add ),
			      NULL, NULL,
			      nomad_marshal_VOID__UINT_UINT_POINTER,
			      G_TYPE_NONE, 3,
			      G_TYPE_UINT,
			      G_TYPE_UINT,
			      G_TYPE_POINTER );
	nomad_jukebox_signals[ TRACK_REMOVE_SIGNAL ] = 
		g_signal_new( "track_remove",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       track_remove ),
			      NULL, NULL,
			      nomad_marshal_VOID__UINT_POINTER,
			      G_TYPE_NONE, 2,
			      G_TYPE_UINT,
			      G_TYPE_POINTER );	
	nomad_jukebox_signals[ TRACK_CHANGED_SIGNAL ] = 
		g_signal_new( "track_changed",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       track_changed ),
			      NULL, NULL,
			      nomad_marshal_VOID__UINT_POINTER_POINTER,
			      G_TYPE_NONE, 3,
			      G_TYPE_UINT,
			      G_TYPE_POINTER,
			      G_TYPE_POINTER );

	nomad_jukebox_signals[ ERROR_SIGNAL ] = 
		g_signal_new( "error",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( NomadJukeboxClass, 
					       error ),
			      NULL, NULL,
			      nomad_marshal_VOID__UINT_STRING,
			      G_TYPE_NONE, 2,
			      G_TYPE_UINT,
			      G_TYPE_STRING );

}

static void
nomad_jukebox_instance_init( NomadJukebox *jukebox )
{
	NomadJukeboxPrivate *priv;

	priv = g_new0( NomadJukeboxPrivate, 1 );
	jukebox->priv = priv;
	
	priv->tracks = g_hash_table_new_full( NULL, NULL,
			NULL, (GDestroyNotify)nomad_track_free );
	/* the key here is the same as that used in the next hashtable
	 * therefore there is nothing to free */
	priv->playlistids = g_hash_table_new( NULL, NULL );
	priv->playlists = g_hash_table_new_full( g_str_hash, 
			g_str_equal,
			(GDestroyNotify)g_free, 
			(GDestroyNotify)g_slist_free );
	
	priv->job_queue = g_queue_new();
	priv->job_queue_mutex = g_mutex_new();

	priv->lock = g_mutex_new();

	sem_init( &priv->transfer_semaphore, 0, 0 );
	priv->transfer_thread = 
		g_thread_create( nomad_jukebox_transfer_thread,
				 jukebox, TRUE, NULL );
}

static void
nomad_jukebox_set_prop( GObject *object, guint prop_id, 
			 const GValue *value, GParamSpec *spec )
{
	NomadJukebox *jukebox;

	jukebox = NOMAD_JUKEBOX( object );

	switch( prop_id ) {
	default:
		break;
	}
}

static void
nomad_jukebox_get_prop( GObject *object, guint prop_id, 
			 GValue *value, GParamSpec *spec )
{
	NomadJukebox *jukebox;

	jukebox = NOMAD_JUKEBOX( object );

	switch( prop_id ) {
	default:
		break;
	}
}

static void
nomad_jukebox_finalize( GObject *object )
{
	NomadJukebox *jukebox;
	NomadJukeboxJob *job;

	jukebox = NOMAD_JUKEBOX( object );

	job = g_new0( NomadJukeboxJob, 1 );
	job->type = TERMINATE;

	g_mutex_lock( jukebox->priv->job_queue_mutex );
	g_queue_push_head( jukebox->priv->job_queue, job );
	g_mutex_unlock( jukebox->priv->job_queue_mutex );
	sem_post( &jukebox->priv->transfer_semaphore );

	g_thread_join( jukebox->priv->transfer_thread );

	g_mutex_free( jukebox->priv->job_queue_mutex );
	g_mutex_free( jukebox->priv->lock );
	
	g_hash_table_destroy( jukebox->priv->tracks );
	g_hash_table_destroy( jukebox->priv->playlistids );
	g_hash_table_destroy( jukebox->priv->playlists );

	g_free( jukebox->priv );
	
	G_OBJECT_CLASS( parent_class )->finalize( object );
}


