/* iproute.c -- IPv4 userspace routing.

   Copyright (C) 2007, 2008 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <dnet.h>

#include "missing.h"
#include "usr.h"
#include "usr-priv.h"
#include "util.h"

static int
rt_handler(const struct route_entry *entry, void *arg)
{
    struct rtable *rtable;

    /* For now we ignore non-IPv4 routing table entries. */
    if (entry->route_dst.addr_type != ADDR_TYPE_IP)
	return 0;

    rtable = arg;

    rtable->nentries++;

    rtable->entries = realloc(rtable->entries,
			      rtable->nentries*sizeof(struct route_entry) );
    if (rtable->entries == NULL)
	return -1;

    rtable->entries[rtable->nentries - 1] = *entry;

    return 0;
}

/*
 * Synchronizes a routing table with the system's routing table.
 */
static int
rtable_sync(struct rtable *rt)
{
    route_t *rh;

    /* Only IPv4 routing tables are supported for now */
    if (rt->type != RTABLE_IPV4)
	return 0;

    rh = route_open();
    if (rh == NULL)
	return -1;

    /* Start fresh */
    rt->nentries = 0;
    free(rt->entries);
    rt->entries = NULL;

    /*
     * Read system routing table. All work is done by the callback function
     * rt_handler()
     */
    if (route_loop(rh, rt_handler, rt) == -1) {
	route_close(rh);
	return -1;
    }

    route_close(rh);

    return 0;
}

/*
 * Creates a routing table object. "type" contains the type of routing
 * table to create (available types are defined in the rtabletype enum 
 * defined in usr.h) and "sync" indicates that the routing table needs
 * to be initially synchronized with the contents of the system routing
 * table.
 */
struct rtable *
rtable_new(enum rtabletype type, int sync)
{
    struct rtable *rtable;

    rtable = (struct rtable *) malloc(sizeof(struct rtable) ); 
    if (rtable == NULL)
	return NULL;

    rtable->type = type; 
    rtable->nentries = 0; 
    rtable->entries = NULL; 

    if (sync)
	if (rtable_sync(rtable) == -1) {
	    free(rtable);
	    return NULL;
	}

    return rtable; 
}

/*
 * Destroys a routing table object by releasing the memory used by the
 * object.
 */
void
rtable_destroy(struct rtable *rt)
{
    free(rt->entries);
    free(rt);
}

/*
 * Prints a routing table. If the pointer to the routing table is NULL then
 * print the system routing table.
 */
void
rtable_print(struct rtable *rt)
{
    int i, n;
    char dest[16], gw[16], mask[16];

    printf("Destination\tGateway \tMask\n");

    if (rt == NULL)
	rt = default_ipv4rt;

    for (i = 0; i < rt->nentries; i++) {
	strlcpy(dest, ipaddr2str(rt->entries[i].route_dst.addr_ip),
		sizeof dest);

	strlcpy(gw, ipaddr2str(rt->entries[i].route_gw.addr_ip), sizeof gw);

	addr_btom(rt->entries[i].route_dst.addr_bits, &n, IP_ADDR_LEN);
	strlcpy(mask, ipaddr2str(n), sizeof mask);

	printf("%-15s\t%-15s\t%-15s\n", dest, gw, mask);
    }
}

