/*

Copyright (C) 2000, 2001, 2002 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/

#include <netdude/nd.h>
#include <netdude/nd_debug.h>
#include <netdude/nd_gettext.h>
#include <netdude/nd_gui.h>
#include <netdude/nd_packet_iterator.h>
#include <netdude/nd_dialog.h>

#include <nd_icmp.h>
#include <nd_icmp_callbacks.h>


void    
nd_icmp_type_cb(ND_Packet   *packet,
		guchar      *header,
		guchar      *data)
{
  static GtkWidget *menu = NULL;

  if (!menu)
    menu = nd_gui_create_menu(icmp_menu_type_data);
  
  gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, 0, 0);
  
  return;
  TOUCH(header);
  TOUCH(packet);  
  TOUCH(data);
}


void    
nd_icmp_code_cb(ND_Packet   *packet,
		guchar      *header,
		guchar      *data)
{
  static GtkWidget *unreach_menu = NULL;
  static GtkWidget *redirect_menu = NULL;

  struct icmphdr *icmphdr = (struct icmphdr *) header;
  
  switch (icmphdr->type)
    {
    case ICMP_DEST_UNREACH:
      if (!unreach_menu)
	unreach_menu = nd_gui_create_menu(icmp_menu_unreach_code_data);
      gtk_menu_popup(GTK_MENU(unreach_menu), NULL, NULL, NULL, NULL, 0, 0);
      return;

    case ICMP_REDIRECT:
      if (!redirect_menu)
	redirect_menu = nd_gui_create_menu(icmp_menu_redirect_code_data);
      gtk_menu_popup(GTK_MENU(redirect_menu), NULL, NULL, NULL, NULL, 0, 0);
      return;

    default:
    }
  
  nd_dialog_number(_("Enter ICMP code:"),
		   ND_BASE_DEC,
		   icmphdr->code,
		   255,
		   (ND_NumberCallback) nd_icmp_code_value_cb,
		   NULL,
		   packet, header);
  
  return;
  TOUCH(header);
  TOUCH(packet);  
  TOUCH(data);
}


static void
icmp_sum_ok_cb(ND_Packet *packet, void *user_data, guint value)
{
  struct icmphdr       *icmphdr;
  ND_PacketIterator     pit;

  for (nd_pit_init(&pit, packet->trace, TRUE); nd_pit_get(&pit); nd_pit_next(&pit))
    {
      icmphdr = (struct icmphdr *) nd_packet_get_data(nd_pit_get(&pit), nd_icmp_get(), 0);
      if (!icmphdr)
	continue;
  
      icmphdr->checksum = htons(value);
      nd_packet_modified_at_index(nd_pit_get(&pit), nd_pit_get_index(&pit));
    }

  return;
  TOUCH(user_data);
}


void    
nd_icmp_cksum_cb(ND_Packet   *packet,
		 guchar      *header,
		 guchar      *data)
{
  struct icmphdr *icmphdr = (struct icmphdr *) header;

  nd_dialog_number(_("Enter ICMP checksum:"),
		   ND_BASE_HEX,
		   ntohs(icmphdr->checksum), 65535,
		   icmp_sum_ok_cb,
		   NULL,
		   packet, data);
  
}


static void
icmp_id_ok_cb(ND_Packet *packet, void *user_data, guint value)
{
  struct icmphdr       *icmphdr;
  ND_PacketIterator     pit;

  for (nd_pit_init(&pit, packet->trace, TRUE); nd_pit_get(&pit); nd_pit_next(&pit))
    {
      icmphdr = (struct icmphdr *) nd_packet_get_data(nd_pit_get(&pit), nd_icmp_get(), 0);
      if (!icmphdr)
	continue;
  
      switch (icmphdr->type)
	{
	case ICMP_ECHO:
	case ICMP_ECHOREPLY:
	case ICMP_ADDRESS:
	case ICMP_ADDRESSREPLY:
	case ICMP_TIMESTAMP:
	case ICMP_TIMESTAMPREPLY:
	case ICMP_INFO_REQUEST:
	case ICMP_INFO_REPLY:
	  break;
	  
	default:
	  continue;
	}

      icmphdr->un.echo.id = htons(value);
      nd_packet_modified_at_index(nd_pit_get(&pit), nd_pit_get_index(&pit));
    }

  return;
  TOUCH(user_data);
}


void    
nd_icmp_id_cb(ND_Packet   *packet,
	      guchar      *header,
	      guchar      *data)
{
  struct icmphdr *icmphdr = (struct icmphdr *) header;

  nd_dialog_number(_("Enter ICMP identifier:"),
		   ND_BASE_DEC,
		   ntohs(icmphdr->un.echo.id),
		   65535,
		   (ND_NumberCallback) icmp_id_ok_cb,
		   NULL,
		   packet, header);

  return;
  TOUCH(data);
}


static void
icmp_seq_ok_cb(ND_Packet *packet, void *user_data, guint value)
{
  struct icmphdr       *icmphdr;
  ND_PacketIterator     pit;

  for (nd_pit_init(&pit, packet->trace, TRUE); nd_pit_get(&pit); nd_pit_next(&pit))
    {
      icmphdr = (struct icmphdr *) nd_packet_get_data(nd_pit_get(&pit), nd_icmp_get(), 0);
      if (!icmphdr)
	continue;

      switch (icmphdr->type)
	{
	case ICMP_ECHO:
	case ICMP_ECHOREPLY:
	case ICMP_ADDRESS:
	case ICMP_ADDRESSREPLY:
	case ICMP_TIMESTAMP:
	case ICMP_TIMESTAMPREPLY:
	case ICMP_INFO_REQUEST:
	case ICMP_INFO_REPLY:
	  break;
	  
	default:
	  continue;
	}
  
      icmphdr->un.echo.sequence = htons(value);
      nd_packet_modified_at_index(nd_pit_get(&pit), nd_pit_get_index(&pit));
    }

  return;
  TOUCH(user_data);
}


void    
nd_icmp_seq_cb(ND_Packet   *packet,
	       guchar      *header,
	       guchar      *data)
{
  struct icmphdr *icmphdr = (struct icmphdr *) header;

  nd_dialog_number(_("Enter ICMP sequence number:"),
		   ND_BASE_DEC,
		   ntohs(icmphdr->un.echo.sequence),
		   65535,
		   (ND_NumberCallback) icmp_seq_ok_cb,
		   NULL,
		   packet, header);

  return;
  TOUCH(data);
}


static void 
icmp_ip_ok_cb(guchar     *address,
	      int         address_len,
	      ND_Packet  *packet,
	      void       *user_data)
{
  struct nd_icmphdr    *icmphdr;
  ND_PacketIterator     pit;

  for (nd_pit_init(&pit, packet->trace, TRUE); nd_pit_get(&pit); nd_pit_next(&pit))
    {
      icmphdr = (struct nd_icmphdr *) nd_packet_get_data(nd_pit_get(&pit), nd_icmp_get(), 0);
      if (!icmphdr)
	continue;

      if (icmphdr->type != ICMP_ADDRESS &&
	  icmphdr->type != ICMP_ADDRESSREPLY)
	continue;

      memcpy(&icmphdr->un.in.in, address, sizeof(guchar) * 4);
      nd_packet_modified_at_index(nd_pit_get(&pit), nd_pit_get_index(&pit));
    }

  return;
  TOUCH(address_len);
  TOUCH(user_data);
}


void    
nd_icmp_ip_cb(ND_Packet   *packet,
	      guchar      *header,
	      guchar      *data)
{
  nd_dialog_ip(_("Enter IP subnet mask:"),
	       (guchar*) data,
	       icmp_ip_ok_cb,
	       NULL,
	       packet, data);  

  return;
  TOUCH(header);
}


static void
icmp_ts_orig_ok_cb(ND_Packet *packet, void *user_data, guint value)
{
  struct nd_icmphdr    *icmphdr;
  ND_PacketIterator     pit;

  for (nd_pit_init(&pit, packet->trace, TRUE); nd_pit_get(&pit); nd_pit_next(&pit))
    {
      icmphdr = (struct nd_icmphdr *) nd_packet_get_data(nd_pit_get(&pit), nd_icmp_get(), 0);
      if (!icmphdr)
	continue;
      
      if (icmphdr->type != ICMP_TIMESTAMP &&
	  icmphdr->type != ICMP_TIMESTAMPREPLY)
	continue;

      icmphdr->un.ts.orig_ts = htonl(value);
      nd_packet_modified_at_index(nd_pit_get(&pit), nd_pit_get_index(&pit));
    }

  return;
  TOUCH(user_data);
}


void    
nd_icmp_ts_orig_cb(ND_Packet   *packet,
		   guchar      *header,
		   guchar      *data)
{
  struct nd_icmphdr *icmphdr_ts = (struct nd_icmphdr *) header;

  nd_dialog_number(_("Enter ICMP receive timestamp:"),
		   ND_BASE_DEC,
		   ntohl(icmphdr_ts->un.ts.orig_ts), (guint) -1,
		   icmp_ts_orig_ok_cb,
		   NULL, packet, data);  
}


static void
icmp_ts_recv_ok_cb(ND_Packet *packet, void *user_data, guint value)
{
  struct nd_icmphdr    *icmphdr;
  ND_PacketIterator     pit;

  for (nd_pit_init(&pit, packet->trace, TRUE); nd_pit_get(&pit); nd_pit_next(&pit))
    {
      icmphdr = (struct nd_icmphdr *) nd_packet_get_data(nd_pit_get(&pit), nd_icmp_get(), 0);
      if (!icmphdr)
	continue;
  
      if (icmphdr->type != ICMP_TIMESTAMP &&
	  icmphdr->type != ICMP_TIMESTAMPREPLY)
	continue;

      icmphdr->un.ts.recv_ts = htonl(value);
      nd_packet_modified_at_index(nd_pit_get(&pit), nd_pit_get_index(&pit));
    }

  return;
  TOUCH(user_data);
}



void    
nd_icmp_ts_recv_cb(ND_Packet   *packet,
		   guchar      *header,
		   guchar      *data)
{
  struct nd_icmphdr *icmphdr_ts = (struct nd_icmphdr *) header;

  nd_dialog_number(_("Enter ICMP receive timestamp:"),
		   ND_BASE_DEC,
		   ntohl(icmphdr_ts->un.ts.recv_ts), (guint) -1,
		   icmp_ts_recv_ok_cb,
		   NULL, packet, data);  
}


static void
icmp_ts_trans_ok_cb(ND_Packet *packet, void *user_data, guint value)
{
  struct nd_icmphdr    *icmphdr;
  ND_PacketIterator     pit;

  for (nd_pit_init(&pit, packet->trace, TRUE); nd_pit_get(&pit); nd_pit_next(&pit))
    {
      icmphdr = (struct nd_icmphdr *) nd_packet_get_data(nd_pit_get(&pit), nd_icmp_get(), 0);
      if (!icmphdr)
	continue;

      if (icmphdr->type != ICMP_TIMESTAMP &&
	  icmphdr->type != ICMP_TIMESTAMPREPLY)
	continue;
  
      icmphdr->un.ts.trans_ts = htonl(value);
      nd_packet_modified_at_index(nd_pit_get(&pit), nd_pit_get_index(&pit));
    }

  return;
  TOUCH(user_data);
}


void    
nd_icmp_ts_trans_cb(ND_Packet   *packet,
		    guchar      *header,
		    guchar      *data)
{
  struct nd_icmphdr *icmphdr_ts = (struct nd_icmphdr *) header;

  nd_dialog_number(_("Enter ICMP transmit timestamp:"),
		   ND_BASE_DEC,
		   ntohl(icmphdr_ts->un.ts.trans_ts), (guint) -1,
		   icmp_ts_trans_ok_cb,
		   NULL, packet, data);  
}


static void
icmp_error_ok_cb(ND_Packet *packet, void *user_data, guint value)
{
  struct icmphdr       *icmphdr;
  ND_PacketIterator     pit;
  
  for (nd_pit_init(&pit, packet->trace, TRUE); nd_pit_get(&pit); nd_pit_next(&pit))
    {
      icmphdr = (struct icmphdr *) nd_packet_get_data(nd_pit_get(&pit), nd_icmp_get(), 0);
      if (!icmphdr)
	continue;
  
      icmphdr->un.gateway = htonl(value);
      nd_packet_modified_at_index(nd_pit_get(&pit), nd_pit_get_index(&pit));
    }

  return;
  TOUCH(user_data);
}


void    
nd_icmp_error_cb(ND_Packet   *packet,
		 guchar      *header,
		 guchar      *data)
{
  struct icmphdr *icmphdr = (struct icmphdr *) header;

  if (!nd_icmp_header_is_error(icmphdr) && icmphdr->type != 10)
    return;

  nd_dialog_number(_("Enter 32-bit ICMP data:"),
		   ND_BASE_DEC,
		   ntohl(icmphdr->un.gateway), (guint) -1,
		   icmp_error_ok_cb,
		   NULL, packet, data);    
}


static void
icmp_adv_num_addr_ok_cb(ND_Packet *packet, void *user_data, guint value)
{
  struct nd_icmphdr    *icmphdr;
  ND_PacketIterator     pit;
  
  for (nd_pit_init(&pit, packet->trace, TRUE); nd_pit_get(&pit); nd_pit_next(&pit))
    {
      icmphdr = (struct nd_icmphdr *) nd_packet_get_data(nd_pit_get(&pit), nd_icmp_get(), 0);
      if (!icmphdr)
	continue;

      if (icmphdr->type != 9)
	continue;

      icmphdr->un.adv.num_addr = value;
      nd_packet_modified_at_index(nd_pit_get(&pit), nd_pit_get_index(&pit));
    }

  return;
  TOUCH(user_data);
}


void    
nd_icmp_adv_num_addr_cb(ND_Packet   *packet,
			guchar      *header,
			guchar      *data)
{
  struct nd_icmphdr *icmphdr = (struct nd_icmphdr *) header;

  nd_dialog_number(_("Enter number of addresses in ICMP router advertisement:"),
		   ND_BASE_DEC,
		   icmphdr->un.adv.num_addr, 255,
		   icmp_adv_num_addr_ok_cb,
		   NULL, packet, data);      
}


static void
icmp_adv_addr_entry_ok_cb(ND_Packet *packet, void *user_data, guint value)
{
  struct nd_icmphdr    *icmphdr;
  ND_PacketIterator     pit;
  
  for (nd_pit_init(&pit, packet->trace, TRUE); nd_pit_get(&pit); nd_pit_next(&pit))
    {
      icmphdr = (struct nd_icmphdr *) nd_packet_get_data(nd_pit_get(&pit), nd_icmp_get(), 0);
      if (!icmphdr)
	continue;
  
      if (icmphdr->type != 9)
	continue;

      icmphdr->un.adv.addr_size = value;
      nd_packet_modified_at_index(nd_pit_get(&pit), nd_pit_get_index(&pit));
    }

  return;
  TOUCH(user_data);
}


void    
nd_icmp_adv_addr_entry_cb(ND_Packet   *packet,
			  guchar      *header,
			  guchar      *data)
{
  struct nd_icmphdr *icmphdr = (struct nd_icmphdr *) header;

  nd_dialog_number(_("Enter ICMP router advertisement entry size:"),
		   ND_BASE_DEC,
		   icmphdr->un.adv.addr_size, 255,
		   icmp_adv_addr_entry_ok_cb,
		   NULL, packet, data);    
}


static void
icmp_adv_lifetime_ok_cb(ND_Packet *packet, void *user_data, guint value)
{
  struct nd_icmphdr    *icmphdr;
  ND_PacketIterator     pit;
  
  for (nd_pit_init(&pit, packet->trace, TRUE); nd_pit_get(&pit); nd_pit_next(&pit))
    {
      icmphdr = (struct nd_icmphdr *) nd_packet_get_data(nd_pit_get(&pit), nd_icmp_get(), 0);
      if (!icmphdr)
	continue;
      
      if (icmphdr->type != 9)
	continue;

      icmphdr->un.adv.lifetime = htons(value);
      nd_packet_modified_at_index(nd_pit_get(&pit), nd_pit_get_index(&pit));
    }

  return;
  TOUCH(user_data);
}


void    
nd_icmp_adv_lifetime_cb(ND_Packet   *packet,
			guchar      *header,
			guchar      *data)
{
  struct nd_icmphdr *icmphdr = (struct nd_icmphdr *) header;

  nd_dialog_number(_("Enter ICMP router advertisement lifetime:"),
		   ND_BASE_DEC,
		   ntohs(icmphdr->un.adv.lifetime), 65535,
		   icmp_adv_lifetime_ok_cb,
		   NULL, packet, data);    
}


static void 
icmp_adv_ip_ok_cb(guchar     *address,
		  int         address_len,
		  ND_Packet  *packet,
		  void       *user_data)
{
  int                   entry_num;
  ND_PacketIterator     pit;
  guchar               *data;
  struct icmphdr       *icmphdr;

  entry_num = GPOINTER_TO_INT(user_data);

  for (nd_pit_init(&pit, packet->trace, TRUE); nd_pit_get(&pit); nd_pit_next(&pit))
    {
      data = nd_packet_get_data(nd_pit_get(&pit), nd_icmp_get(), 0);
      if (!data)
	continue;
  
      icmphdr = (struct icmphdr *) data;
      if (icmphdr->type != 9)
	continue;

      memcpy(data + 8 + entry_num * 8, address, sizeof(guchar) * 4);
      nd_packet_modified_at_index(nd_pit_get(&pit), nd_pit_get_index(&pit));
    }
  
  return;
  TOUCH(address_len);
}


void    
nd_icmp_adv_ip_cb(ND_Packet   *packet,
		  guchar      *header,
		  guchar      *data)
{
  int entry_num = (data - header - 8) / 8;

  nd_dialog_ip(_("Enter router IP address:"),
	       (guchar*) data,
	       icmp_adv_ip_ok_cb,
	       NULL,
	       packet, GINT_TO_POINTER(entry_num));
}


static void
icmp_adv_pref_ok_cb(ND_Packet *packet, void *user_data, guint value)
{
  int                   entry_num;
  guchar               *data;
  struct icmphdr       *icmphdr;
  ND_PacketIterator     pit;
  guint32               val = htonl(value);

  entry_num = GPOINTER_TO_INT(user_data);  

  for (nd_pit_init(&pit, packet->trace, TRUE); nd_pit_get(&pit); nd_pit_next(&pit))
    {
      data = nd_packet_get_data(nd_pit_get(&pit), nd_icmp_get(), 0);
      if (!data)
	continue;
  
      icmphdr = (struct icmphdr *) data;
      if (icmphdr->type != 9)
	continue;
      
      memcpy(data + 12 + entry_num * 8, &val, sizeof(guint32));
      nd_packet_modified_at_index(nd_pit_get(&pit), nd_pit_get_index(&pit));
    }

  return;
  TOUCH(user_data);
}


void    
nd_icmp_adv_pref_cb(ND_Packet   *packet,
		    guchar      *header,
		    guchar      *data)
{
  guint32 pref;
  int     entry_num;

  if (!data)
    return;

  entry_num = (data - header - 12) / 8;
  pref = * ((guint32*) data);

  nd_dialog_number(_("Enter ICMP router advertisement preference:"),
		   ND_BASE_DEC,
		   ntohl(pref), (guint) -1,
		   icmp_adv_pref_ok_cb,
		   NULL, packet, GINT_TO_POINTER(entry_num));    
}


/* Menu item callbacks */

void    
nd_icmp_type_value_cb(ND_Packet   *packet,
		      guchar      *header,
		      int          value)
{
  struct icmphdr       *icmphdr;
  ND_PacketIterator     pit;

  for (nd_pit_init(&pit, packet->trace, TRUE); nd_pit_get(&pit); nd_pit_next(&pit))
    {
      icmphdr = (struct icmphdr *) nd_packet_get_data(nd_pit_get(&pit), nd_icmp_get(), 0);
      if (!icmphdr)
	continue;
  
      icmphdr->type = value;
      nd_packet_init(packet);
      nd_packet_modified_at_index(nd_pit_get(&pit), nd_pit_get_index(&pit));
    }

  return;
  TOUCH(header);
}


void    
nd_icmp_type_custom_cb(ND_Packet   *packet,
		       guchar      *header,
		       int          value)
{
  struct icmphdr *icmphdr = (struct icmphdr *) header;

  nd_dialog_number(_("Enter ICMP type:"),
		   ND_BASE_DEC,
		   icmphdr->type,
		   255,
		   (ND_NumberCallback) nd_icmp_type_value_cb,
		   NULL,
		   packet, header);

  return;
  TOUCH(value);
}


void    
nd_icmp_code_value_cb(ND_Packet   *packet,
		      guchar      *header,
		      int          value)
{
  struct icmphdr       *icmphdr;
  ND_PacketIterator     pit;

  for (nd_pit_init(&pit, packet->trace, TRUE); nd_pit_get(&pit); nd_pit_next(&pit))
    {
      icmphdr = (struct icmphdr *) nd_packet_get_data(nd_pit_get(&pit), nd_icmp_get(), 0);
      if (!icmphdr)
	continue;
  
      icmphdr->code = value;
      nd_packet_modified_at_index(nd_pit_get(&pit), nd_pit_get_index(&pit));
    }

  return;
  TOUCH(header);
}


void    
nd_icmp_code_custom_cb(ND_Packet   *packet,
		       guchar      *header,
		       int          value)
{
  struct icmphdr *icmphdr = (struct icmphdr *) header;

  nd_dialog_number(_("Enter ICMP code:"),
		   ND_BASE_DEC,
		   icmphdr->code,
		   255,
		   (ND_NumberCallback) nd_icmp_code_value_cb,
		   NULL,
		   packet, header);

  return;
  TOUCH(value);
}


void    
nd_icmp_cksum_fix_cb(ND_Packet   *packet,
		     guchar      *header,
		     int          value)
{
  guint16               correct_sum;
  struct icmphdr       *icmphdr;
  ND_PacketIterator     pit;

  for (nd_pit_init(&pit, packet->trace, TRUE); nd_pit_get(&pit); nd_pit_next(&pit))
    {
      icmphdr = (struct icmphdr *) nd_packet_get_data(nd_pit_get(&pit), nd_icmp_get(), 0);
      if (!icmphdr)
	continue;
      
      if (!nd_icmp_csum_correct(nd_pit_get(&pit), &correct_sum))
	{
	  icmphdr->checksum = correct_sum;
	  nd_packet_modified_at_index(nd_pit_get(&pit), nd_pit_get_index(&pit));
	}
    }

  return;
  TOUCH(header);
  TOUCH(value);
}


