/*************************************************************************
***	Authentication, authorization, accounting + firewalling package
***	Copyright 1998-2002 Anton Vinokurov <anton@netams.com>
***	Copyright 2002-2008 NeTAMS Development Team
***	This code is GPL v3
***	For latest version and more info, visit this project web page
***	located at http://www.netams.com
***
*************************************************************************/
/* $Id: users.c,v 1.52 2009-08-01 09:23:55 anton Exp $ */

#include "netams.h"

//////////////////////////////////////////////////////////////////////////
UsersList *Users=NULL;
//////////////////////////////////////////////////////////////////////////
User::User(char *n, u_char h):Object(){
	if(n) name=set_string(n);
	else name=NULL;
	real_name=email=password=sms=NULL;
	permissions=UPERM_NONE;
#ifdef HAVE_PAM
	pam_auth=0;
#endif
	hidden=h;
	mode=USER_HUMAN;
}

User::~User(){
	if(name) aFree(name);
	if(real_name) aFree(real_name);
	if(email) aFree(email);
	if(password) aFree(password);
	if(sms) aFree(sms);
}

void User::setName(char *s) {
	if(name) aFree(name);
	name=set_string(s);
	Users->changed_user=1;
}
void User::setRealName(char *s) {
	if(real_name) aFree(real_name);
	real_name=set_string(s);
}
void User::setEmail(char *s) {
	if(email) aFree(email);
	email=set_string(s);
}
void User::setSms(char *s) {
	if(sms) aFree(sms);
	sms=set_string(s);
}
void User::setPermissions(char *s) {
	for (u_short i=0; i<256; i++)
		if (STREQ(s, permissions_list[i])) {
			permissions=i;
			break;
		}
}

u_char User::getPermissions() { return permissions; }
//////////////////////////////////////////////////////////////////////////
UsersList::UsersList(): List(){
	changed_pw=changed_user=1;
}

User *UsersList::getUser(char *name) {
	User *d=NULL;
	if (name==NULL) return NULL;
	netams_rwlock_rdlock(&rwlock);
	for(d=(User*)root; d!=NULL; d=(User*)d->next)
		if (STREQ(d->name, name))
			break;
	netams_rwlock_unlock(&rwlock);

	if(!d) {
		oid id=strtol(name, NULL, 16);
		if(id) d=(User*)getById(id);
	}
	return d;
}

void UsersList::listUsers(struct cli_def *cli){
	User *t;
	const char *type;

	cli_print(cli, "%6s | %4s | %10s | %20s | %10s", "OID", "MODE", "NAME", "REAL NAME", "PERMIT");

	netams_rwlock_rdlock(&rwlock);
	for(t=(User*)root; t!=NULL; t=(User*)t->next){
		if (!t->hidden) {
			switch (t->mode) {
				case USER_HUMAN:	type="U"; break;
				case USER_DATASOURCE:	type="DS";break;
				case USER_STORAGE:	type="ST";break;
				default: 		type="\?\?";break;
			}
			cli_print(cli, "%06X | %4s | %10s | %20s | %10s",
				t->id, type, t->name,
				t->real_name?t->real_name:"<>",
				permissions_list[t->permissions]);
		}
	}
	netams_rwlock_unlock(&rwlock);
}

void UsersList::listUsersCfg(struct cli_def *cli, u_char flags){
	User *t;
	netams_rwlock_rdlock(&rwlock);
	for(t=(User*)root; t!=NULL; t=(User*)t->next){
		if (!t->hidden) {
			cli_bufprint(cli, "user oid %06X", t->id);
			if (t->name) cli_bufprint(cli, " name %s", t->name);
			if (t->real_name) cli_bufprint(cli, " real-name \"%s\"", t->real_name);
			if (t->password) cli_bufprint(cli, " crypted %s", t->password);
			if (t->email) cli_bufprint(cli, " email %s", t->email);
			if (t->sms) cli_bufprint(cli, " sms %s", t->sms);
#ifdef HAVE_PAM
			if (t->pam_auth) cli_bufprint(cli," pam_auth");
#endif
			cli_bufprint(cli, " permit %s\n", permissions_list[t->permissions]);
		}
	}
	netams_rwlock_unlock(&rwlock);
}

void UsersList::listUsersHtml(FILE *f){
	User *t;
	netams_rwlock_rdlock(&rwlock);
	for (t=(User*)root; t!=NULL; t=(User*)t->next)
		if (!t->hidden && t->permissions >= UPERM_SHOWSTAT_HIS)
			fprintf(f, " %s", t->name);
	netams_rwlock_unlock(&rwlock);
}

void UsersList::listPasswordsHtml(FILE *f){
	User *t;
	netams_rwlock_rdlock(&rwlock);
	for (t=(User*)root; t!=NULL; t=(User*)t->next)
		if (!t->hidden && t->permissions >= UPERM_SHOWSTAT_HIS) {
			if (t->password)
				fprintf(f, "%s:%s\n", t->name, t->password);
		}
	netams_rwlock_unlock(&rwlock);
}

//////////////////////////////////////////////////////////////////////////
int cUser(struct cli_def *cli, const char *cmd, char **argv, int argc) {
	User *u=NULL;
	oid id=0;
	u_char i=0;
	u_char no_flag=0;

	no_flag = CLI_CHECK_NO(argv[i]);
	if(no_flag) i++;
	i++;

	if (STRARG(argv[i], "name")) {
		u=Users->getUser(argv[i+1]);
	} else if (STRARG(argv[i], "oid")) {
		id=strtol(argv[i+1], NULL, 16);
		u=(User*)Users->getById(id);
	}
	else {
		cli_error(cli, "user: %s command unknown", argv[i]?argv[i]:"");
		return CLI_OK;
	}

	if (is_running) gettimeofday(&when_config_changed, NULL);

	if (!u && !no_flag) {
		u= new User(NULL);
		u->id=newOid(id);
		Users->Insert(u);
		cli_error(cli, "user %s %06X created", u->name?u->name:"", u->id);
	}

	if (u && no_flag){
		cli_error(cli, "user %s %06X deleted", u->name?u->name:"", u->id);
		Users->Delete(u);
		return CLI_OK;
	}

	for(; i<argc; i+=2) {
	 	if (STRARG(argv[i], "real-name")) {
			u->setRealName(argv[i+1]);
			cli_error(cli, "user %s real-name set: %s", u->name, u->real_name);

		} else if (STRARG(argv[i], "name")) {
			u->setName(argv[i+1]);
			cli_error(cli, "user %s name set: %s", u->name, u->name);

		} else if (STRARG(argv[i], "email")) {
			u->setEmail(argv[i+1]);
			cli_error(cli, "user %s email set: %s", u->name, u->email);

		} else if (STRARG(argv[i], "sms")) {
			u->setSms(argv[i+1]);
			cli_error(cli, "user %s sms number set: %s", u->name, u->sms);

		} else if (STRARG(argv[i], "crypted")) {
			if(u->password) aFree(u->password);
			u->password=set_string(argv[i+1]);
			cli_error(cli, "user %s crypted set: %s", u->name, u->password?u->password:"<>");

	 	} else if (STRARG(argv[i], "password")) {
			if(u->password) aFree(u->password);
			u->password=CryptWord(argv[i+1]);
			cli_error(cli, "user %s crypted set: %s", u->name, u->password?u->password:"<>");

		} else if (STRARG(argv[i], "permit")) {
			u->setPermissions(argv[i+1]);
			cli_error(cli, "user %s permissions set: %s", u->name, permissions_list[u->getPermissions()]);
#ifdef HAVE_PAM
		} else if (STREQ(argv[i], "pam_auth")) {
			u->pam_auth=1;
			i--;
#endif
		} else
			cli_error(cli, "user %s: %s command unknown", u->name, argv[i]);
	}
	return CLI_OK;
}
//////////////////////////////////////////////////////////////////////////
int cShowUsers(struct cli_def *cli, const char *cmd, char **argv, int argc){
	Users->listUsers(cli);
	return CLI_OK;
}
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
