/*************************************************************************
***	Authentication, authorization, accounting + firewalling package
***	Copyright 1998-2002 Anton Vinokurov <anton@netams.com>
***	Copyright 2002-2008 NeTAMS Development Team
***	This code is GPL v3
***	For latest version and more info, visit this project web page
***	located at http://www.netams.com
***
*************************************************************************/
/* $Id: attrlist.c,v 1.32 2008-02-23 08:35:03 anton Exp $ */

#include "netams.h"
#include "attrlist.h"

static const char *attr_type_name[] = { "", "ip", "mac", "email", NULL };

Attribute::Attribute(attr_type atype) {
	type=atype;
}

const char* Attribute::name() {
	return attr_type_name[type];
}
/////////////////////////////////////////////////////////////////////////////////////

//ATTR_IPV4
class Attr_IPv4 : public Attribute {
	public:
		u_char num;
		struct in_addr *addr;
		struct in_addr *mask;

		Attr_IPv4();
		~Attr_IPv4();
				
		char* getAttr(char *buf);
		int setAttr(char **str, char *errbuf=NULL);
		match Check(Flow *);
};

Attr_IPv4::Attr_IPv4() : Attribute(ATTR_IPV4) {
	addr=NULL;
	mask=NULL;
	num=0;
}

Attr_IPv4::~Attr_IPv4() {
	if(addr) aFree(addr);
	if(mask) aFree(mask);
}

char* Attr_IPv4::getAttr(char *buf) {
	char tmp[32];
		
	for(u_char i=0; i<num; i++) {
		inet_ntop(AF_INET, &addr[i], tmp, 32);
		sprintf(buf+strlen(buf), "%s%s", (i)?" ":"", tmp);
		if(MASK2MLEN(mask[i])!=32)
			sprintf(buf+strlen(buf), "/%u", MASK2MLEN(mask[i]));
	}
	return buf;
}

int Attr_IPv4::setAttr(char **str, char *errbuf) {
	u_char i;
	struct in_addr taddr;
	struct in_addr tmask;
		
	for(i=0; str[i]!=NULL; i++) {
		if(getAddr(str[i], &taddr, &tmask)<0)
			break;
	}
	if(i==0) {
		if(errbuf) sprintf(errbuf, "Can't determine IP address");
		return -1;
	}
	num=i;
	addr=(struct in_addr*)aMalloc(sizeof(struct in_addr)*num);
	mask=(struct in_addr*)aMalloc(sizeof(struct in_addr)*num);
	
	for(i=0; i<num; i++) {
		getAddr(str[i], &addr[i], &mask[i]);
	}	
	return i;
}

match Attr_IPv4::Check(Flow *flow) {
	match mf=MATCH_NONE;
	struct ipv4_info_value  *ipv4_info = (struct ipv4_info_value*)flow->get(ATTR_IPV4_INFO);

	if(ipv4_info == NULL) {
		return MATCH_NONE;
	}
	for(u_char i=0; i<num; i++) {
		if (addr[i].s_addr==(ipv4_info->ip_src.s_addr&mask[i].s_addr)) mf|=MATCH_SRC;
		if (addr[i].s_addr==(ipv4_info->ip_dst.s_addr&mask[i].s_addr)) mf|=MATCH_DST;
	}
	return mf;	
}
/////////////////////////////////////////////////////////////////////////////////////
//ATTR_MAC
class Attr_MAC : public Attribute {
	public:
		struct ether_addr mac;
		
		Attr_MAC();
		~Attr_MAC() {};
				
		char* getAttr(char *buf);
		int setAttr(char **str, char *errbuf=NULL);
};

Attr_MAC::Attr_MAC() : Attribute(ATTR_MAC) {
}

char* Attr_MAC::getAttr(char *buf) {
	sprintf(buf, "%s", ether_ntoa(&mac));
	return buf;  
}

int Attr_MAC::setAttr(char **str, char *errbuf) {
	struct ether_addr *m;
	if((m=ether_aton(str[0]))==NULL) {
		if(errbuf) sprintf(errbuf, "Can't determine MAC address");
		return -1;
	} else memcpy(&mac, m, sizeof (mac));
	return 1;
}
/////////////////////////////////////////////////////////////////////////////////////
//ATTR_EMAIL
class Attr_EMAIL : public Attribute {
	public:
		char *email;
		
		Attr_EMAIL();
		~Attr_EMAIL();
				
		char* getAttr(char *buf);
		int setAttr(char **str, char *errbuf=NULL);
};

Attr_EMAIL::Attr_EMAIL() : Attribute(ATTR_EMAIL) {
	email=NULL;
}

Attr_EMAIL::~Attr_EMAIL() {
	if(email) aFree(email);
}

char* Attr_EMAIL::getAttr(char *buf) {
	sprintf(buf, "%s", email);
	return buf;
}

int Attr_EMAIL::setAttr(char **str, char *errbuf) {
	email=set_string(str[0]);
	return 1;
}
/////////////////////////////////////////////////////////////////////////////////////

Attribute* getAttr(Attribute **attrlist, attr_type atype) {
	Attribute *a;
	ELIST_FOR_EACH(attrlist, a) {
		if(atype == a->type)
			return a;
	}
	return NULL;
}
char* getAttr(Attribute **attrlist, attr_type atype, char *buf) {
	Attribute *a;
	if((a=getAttr(attrlist, atype))!=NULL)
		return a->getAttr(buf);
	return NULL;
}

int setAttr(Attribute **attrlist, struct cli_def *cli, char **param) {
	char errbuf[256];
	u_char no_flag;
	u_char atype;
	Attribute *olda, *newa;
	
	for(u_char i=0; param[i]!=NULL; i++) {
		no_flag=0;
		atype=ATTR_NONE;
		
 		if (STREQ(param[i], "no")) { 
			no_flag=1; 
			i++;
		}
		
		for(u_char t=1; attr_type_name[t]!=NULL; t++) {
			if(strcmp(param[i], attr_type_name[t])==NULL) {
				atype=t;
				break;
			}
		}
		if(atype==ATTR_NONE) {
			cli_error(cli, "No such attribute %s", param[i]);
			continue;
		}	

		if(!no_flag) {
			switch(atype) {
				case ATTR_IPV4:
					newa = new Attr_IPv4();
					break;
				case ATTR_MAC:
					newa = new Attr_MAC();
					break;
				case ATTR_EMAIL:
					newa = new Attr_EMAIL();
					break;
				default:
					break;
			}
			
			int ret = newa->setAttr(&param[i+1], errbuf);
			if(ret<0) {
				cli_error(cli, "Can't set %s attribute: %s", newa->name(), errbuf);
				continue;
			}
			i+=ret;
		}
		
		if((olda=getAttr(attrlist, (attr_type)atype))) {
			ELIST_REMOVE(attrlist, olda);
			delete olda;
		}
		
		if(!no_flag) {
			ELIST_ADD(attrlist, newa);
		}
	}					
	return 1;
}
/////////////////////////////////////////////////////////////////////////////////////	
