// Copyright (c) 2003-2005 The University of Wroclaw.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
//    1. Redistributions of source code must retain the above copyright
//       notice, this list of conditions and the following disclaimer.
//    2. Redistributions in binary form must reproduce the above copyright
//       notice, this list of conditions and the following disclaimer in the
//       documentation and/or other materials provided with the distribution.
//    3. The name of the University may not be used to endorse or promote
//       products derived from this software without specific prior
//       written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
// NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
// TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//

using System;
using System.IO;
using System.Collections;
using Nemerle.IO;

namespace Nemerle.CSharp
{
    /**
     *  Module used to emit Nemerle code
     */
    public module Emit
    {
        /* -- PUBLIC METHODS -------------------------------------------------- */

        /**
         *  Opens output buffer
         */
        public static Initialize (file_name : string, need_output_filter : bool) : void
        {                        
            out_stream = FileStream (file_name, FileMode.Create);

            def output_filter (s) {
              if (s.Trim () == "/*ASPX-REMOVE-BRACE*/}") ""
              else {
                def s = s.Replace ("class AspxRemoveMe {", "");
                if (s.Trim ().StartsWith ("/*ASPX-UNCOMMENT:"))
                  s.Replace ("/*ASPX-UNCOMMENT:", "").Replace ("*/", "")
                else s
              }
            }

            if (need_output_filter)
              writer = PipeWriter (StreamWriter (out_stream), output_filter);
            else
              writer = StreamWriter (out_stream);            
            buffer = System.Text.StringBuilder ("");
        }

        /**
         *  Closes output buffer
         */
        public End () : void
        {
            when (ExtendedToken.LastToken != null) {
              writer.Write (ExtendedToken.getWhitespaces (ExtendedToken.LastToken));
              ExtendedToken.LastToken = null;
            }
            writer.Close ();
            out_stream.Close ();
        }
        
        /**
         *  Emits string s to output stream
         */
        public EmitString (s : string) : void
        {
            when(is_buffered)
                _ = buffer.Append (s);
            writer.Write(s);
        }


        /**
         *  Emits string s to output stream
         */
        public EmitChar (s : char) : void
        {
            when(is_buffered)
                _ = buffer.Append (s);
            writer.Write(s);
        }

        
        /**
         *  Emits text of token to output stream
         */
        public EmitToken (t : antlr.IToken) : void
        {
            EmitString ( t.getText());
        }                

        /**
         *  Begins buffering emit commands
         */
        public BeginBuffer () : void
        {
            buffer = System.Text.StringBuilder ("");
            is_buffered = true;
        }

        /**
         *  Ends buffering emit commands
         */
        public EndBuffer () : void
        {
            is_buffered = false;
        }

        /**
         *  Gets buffered emit commands
         */
        public Buffer : string
        {
            get
            {
                buffer.ToString ()
            }
        }

        /* -- PRIVATE FIELDS -------------------------------------------------- */
        
        private mutable static out_stream : FileStream;
        private mutable static writer : TextWriter;
        
        private mutable static buffer : System.Text.StringBuilder;        
        private mutable static is_buffered : bool = false;
    }    
}
