/*
 * NASPRO - NASPRO Architecture for Sound Processing
 * LADSPA bridge
 *
 * Copyright (C) 2007-2010 Stefano D'Angelo <zanga.mail@gmail.com>
 *
 * See the COPYING file for license conditions.
 */

#include <stdlib.h>

#include <ladspa.h>

#include <lv2.h>

#include <NASPRO/core/lib.h>

#include "pluglib.h"
#include "lv2api.h"

struct ll_desc
  {
	LV2_Descriptor		 lv2_desc;
	LADSPA_Descriptor	*l_desc;
  };

struct instance
  {
	LADSPA_Handle		 l_handle;
	LADSPA_Descriptor	*l_desc;
  };

static struct ll_desc *descs = NULL;
static size_t descs_count = 0;

LV2_Handle
_naladspa_lv2api_instantiate(const LV2_Descriptor *descriptor,
			     double sample_rate, const char *bundle_path,
			     const LV2_Feature * const *features)
{
	LADSPA_Descriptor *ldesc;
	LADSPA_Handle *handle;
	struct instance *instance;

	instance = malloc(sizeof(struct instance));
	if (instance == NULL)
		return NULL;

	/* This should work correctly according to the ANSI C standard. */
	ldesc = ((struct ll_desc *)descriptor)->l_desc;

	handle = ldesc->instantiate(ldesc, sample_rate);
	if (handle == NULL)
	  {
		free(instance);
		return NULL;
	  }

	instance->l_desc = ldesc;
	instance->l_handle = handle;

	return (LV2_Handle)instance;
}

void
_naladspa_lv2api_connect_port(LV2_Handle instance, uint32_t port,
			      void *data_location)
{
	struct instance *i;

	/* This is needed to avoid a little incompatibility beetween LADSPA and
	 * LV2. LADSPA's connect_port() does not specify whether data_location
	 * is valid at the time it is being run, while LV2 mandates the plugin
	 * not to trust the memory location indicated by the pointer at the time
	 * connect_port() is run. */
	if (data_location == NULL)
		return;

	i = ((struct instance *)instance);

	i->l_desc->connect_port(i->l_handle, port, data_location);
}

void
_naladspa_lv2api_activate(LV2_Handle instance)
{
	struct instance *i;

	i = (struct instance *)instance;

	i->l_desc->activate(i->l_handle);
}

void
_naladspa_lv2api_run(LV2_Handle instance, uint32_t sample_count)
{
	struct instance *i;

	i = (struct instance *)instance;

	i->l_desc->run(i->l_handle, sample_count);
}

void
_naladspa_lv2api_deactivate(LV2_Handle instance)
{
	struct instance *i;

	i = (struct instance *)instance;

	i->l_desc->deactivate(i->l_handle);
}

void
_naladspa_lv2api_cleanup(LV2_Handle instance)
{
	struct instance *i;

	i = (struct instance *)instance;

	i->l_desc->cleanup(i->l_handle);
	free(i);
}

static void
generate_desc(void *content, void *data)
{
	struct nacore_descriptor *ndesc;
	LADSPA_Descriptor *ldesc;
	struct ll_desc *lldesc;
	size_t *n;

	n = (size_t *)data;
	lldesc = descs + *n;
	ndesc = (struct nacore_descriptor *)content;
	ldesc = (LADSPA_Descriptor *)ndesc->data;

	nacore_lv2api_fill_desc(&lldesc->lv2_desc, ndesc);
	lldesc->l_desc = ldesc;

	(*n)++;
}

int
_naladspa_lv2api_generate_descs()
{
	size_t n;

	n = nacore_avl_tree_get_nodes_count(_naladspa_pluglib_desc_tree);
	descs = malloc(n * sizeof(struct ll_desc));
	if (descs == NULL)
		return -1;

	descs_count = n;

	n = 0;
	nacore_avl_tree_for_each(_naladspa_pluglib_desc_tree, generate_desc,
				 &n);

	return 0;
}

void
_naladspa_lv2api_free_descs()
{
	free(descs);
	descs = NULL;
	descs_count = 0;
}

const LV2_Descriptor *
lv2_descriptor(uint32_t index)
{
	if (index >= descs_count)
		return NULL;

	return &descs[index].lv2_desc;
}
