--------------------------------------------------------------------------
--                                                                      --
--           Copyright: Copyright (C) 2000-2010 CNRS/IN2P3              --
--                                                                      --
-- Narval framework is free  software; you can redistribute  it and/or  --
-- modify  it   under  terms  of  the  GNU General  Public  License as  --
-- published  by  the  Free Software Foundation; either version  2, or  --
-- (at your option) any later version. Narval framework is distributed  --
-- in the hope  that  they will  be useful, but  WITHOUT ANY WARRANTY;  --
-- without even the implied warranty of  MERCHANTABILITY or FITNESS FOR --
-- A PARTICULAR PURPOSE. See the  GNU. General Public License for more  --
-- details. You should have received  a copy of the GNU General Public  --
-- License distributed with Narval; see file COPYING. If not, write to  --
-- the Free Software  Foundation,  Inc., 51 Franklin St,  Fifth Floor,  --
-- Boston, MA 02110-1301 USA.                                           --
--------------------------------------------------------------------------
with Ada.Unchecked_Deallocation;
with System;
with Ada.Exceptions;
with Ada.Strings.Unbounded;
with Ada.Containers.Vectors;

with Start_Stop_Handling;
with Log4ada.Loggers;
with Log4ada.Appenders.Consoles;
with Log4ada.Appenders.Annex_E;
with Shared_Library;

with Narval.Communication;
with Narval.Protected_Memory;
with Narval.Parameters;
with Narval.Configurator.Actors_Description;

package Narval.Actors.Actives is

   type Active_Actor_Type (Kind : Actor_Kind_Type) is abstract new
     Actors.Actor with private;
   subtype Active_Actor_Class is Active_Actor_Type'Class;
   type Actor_Access is access all Actors.Actor;
   type Active_Actor_Class_Access is access all Active_Actor_Type'Class;
   type Active_Actor_Access is access all Active_Actor_Type;

   procedure Trig_Event (Active_Actor : access Active_Actor_Type;
                         Event : String);
   procedure Set (Active_Actor : access Active_Actor_Type;
                  Parameter : String;
                  Value : String);
   function Get_Xml (Active_Actor : access Active_Actor_Type;
                     Parameter : String) return String;
   function Get_Image (Active_Actor : access Active_Actor_Type;
                       Parameter_Name : String) return String;
   function Arguments (Active_Actor : access Active_Actor_Type;
                       Xml : Boolean := True)
                      return String;

   procedure Change_State (Active_Actor : access Active_Actor_Type;
                           Order : Action);
   procedure Initialise (Active_Actor : access Active_Actor_Type;
                         Actor_Name : String);
   procedure Log_Message (Active_Actor : access Active_Actor_Type;
                          Message : String;
                          Level : Log4ada.Level_Type);
   procedure Log_Message
     (Active_Actor : access Active_Actor_Type;
      Message : String;
      Level : Log4ada.Level_Type;
      Exception_To_Send : Ada.Exceptions.Exception_Occurrence);
   function Get_Available_Link
     (Active_Actor : access Active_Actor_Type;
      Asker : String;
      Port : String;
      Service_Name : String)
     return Link_Descriptor_Type;

   procedure Buffer_Handling (Active_Actor : access Active_Actor_Type)
      is abstract;

   Actor_Not_Ready : exception;
   Initialisation_Failed : exception;
   Incoherent_Links : exception;

   P_NAME : constant String := "name";
   P_KIND : constant String := "kind";
   P_LOG_LEVEL : constant String := "log_level";
   P_CLIENTS : constant String := "clients";
   P_SERVERS : constant String := "servers";
   P_BYTES_OUT : constant String := "bytes_out";
   P_BYTES_IN : constant String := "bytes_in";
   P_PORT : constant String := "port";
   P_RANK : constant String := "rank";
   P_HOST_NAME : constant String := "host_name";
   P_ACTIVE : constant String := "actif";
   P_PRELOAD_LIBRARY : constant String := "preload_library";
   P_LIBRARIES : constant String := "libraries";
   P_REMOVE_LIBRARIES : constant String := "remove_libraries";
   P_CURRENT_DIRECTORY : constant String := "current_directory";
   P_FILES : constant String := "files";
   P_WATCHER_DELAY : constant String := "watcher_delay";
   P_WATCHER_PORT : constant String := "watcher_port";
   P_WATCHER_OUTPUT_NUMBER : constant String := "watcher_output_number";
   P_WATCHER_RUNNING : constant String := "watcher_running";
   P_RUN_NUMBER : constant String := "run_number";

private
   type Type_Interface is tagged limited record
      Name : Ada.Strings.Unbounded.Unbounded_String;
      Memory : Protected_Memory.Memory_Access;
   end record;
   procedure Free (Interface_Arg : in out Type_Interface);
   type Input_Interface_Type is new Type_Interface with record
      Input_Task : Communication.Input_Task_Access := null;
      Link : Communication.Link_Access := null;
      Swapped_Link : Boolean := False;
   end record;
   type Input_Interface_Array is array (Positive range <>)
     of Input_Interface_Type;
   type Input_Interface_Array_Access is access Input_Interface_Array;
   procedure Connect (Interface_Arg : in out Input_Interface_Type);
   procedure Initialise
     (Interface_Arg : in out Input_Interface_Type;
      Asker : String;
      Input : Configurator.Actors_Description.Receiver_Descriptor_Type;
      Start_Stop : Start_Stop_Handling.Synchro_Start_Access;
      Logger : Log4ada.Loggers.Logger_Class_Access);
   procedure Free (Interface_Arg : in out Input_Interface_Type);
   procedure Free is new Ada.Unchecked_Deallocation
     (Input_Interface_Array, Input_Interface_Array_Access);
   type Output_Interface_Type is new Type_Interface with record
      Position_Fifo : Positive := 1;
      Output_Task : Communication.Output_Task_Access := null;
      Links : Communication.Link_Array_Access := null;
   end record;
   type Output_Interface_Array is array (Positive range <>)
     of Output_Interface_Type;
   type Output_Interface_Array_Access is access Output_Interface_Array;
   procedure Connect (Interface_Arg : in out Output_Interface_Type;
                      Bit_Order_To_Send : System.Bit_Order);
   procedure Initialise
     (Interface_Arg : in out Output_Interface_Type;
      Output : Configurator.Actors_Description.Sender_Descriptor_Type;
      Start_Stop : Start_Stop_Handling.Synchro_Start_Access;
      Logger : Log4ada.Loggers.Logger_Class_Access);
   procedure Free (Interface_Arg : in out Output_Interface_Type);
   procedure Free is new Ada.Unchecked_Deallocation
     (Output_Interface_Array, Output_Interface_Array_Access);

   task type Work_Task_Type (Wrapper : access Active_Actor_Type) is
     entry Start;
     entry Stop;
   end Work_Task_Type;

   task type Task_Buffer_Dispatcher_Type
     (Wrapper : access Active_Actor_Type) is
      entry Start;
      entry Stop;
   end Task_Buffer_Dispatcher_Type;

   type Library_Full_Description_Type is record
      Name : Ada.Strings.Unbounded.Unbounded_String :=
        Ada.Strings.Unbounded.Null_Unbounded_String;
      Handle : Shared_Library.Handle_Type;
   end record;

   package Library_Vector_Package is new Ada.Containers.Vectors
     (Positive, Library_Full_Description_Type, "=");

   type Active_Actor_Type (Kind : Actor_Kind_Type) is abstract new
     Actors.Actor (Kind) with record
        Name : Ada.Strings.Unbounded.Unbounded_String :=
          Ada.Strings.Unbounded.Null_Unbounded_String;
        Description : Configurator.Actors_Description.Actor_Description_Type;
        Waiting_Delay : Duration := Default_Waiting_Duration;
        Start_Stop : Start_Stop_Handling.Synchro_Start_Access := null;
        Inputs : Input_Interface_Array_Access := null;
        Outputs : Output_Interface_Array_Access := null;
        Working : Work_Task_Type (Active_Actor_Type'Access);
        Reverse_Endian : Boolean := False;
        Can_Be_Stopped : Boolean := True;
        Logger : aliased Log4ada.Loggers.Logger_Type;
        Console : aliased Log4ada.Appenders.Consoles.Console_Type;
        Remote_Appender :
          aliased Log4ada.Appenders.Annex_E.Remote_Appender_Type;
        Parameters_List : Parameters.Parameter_Vector_Package.Vector;
        Libraries : Library_Vector_Package.Vector;
        Duplicate : Boolean := False;
        Duplication_Port : Positive := 10201;
        Duplication_Output : Positive := 1;
        Buffer_Dispatcher : Task_Buffer_Dispatcher_Type
          (Active_Actor_Type'Access);
        Watcher_Delay : Duration := 0.1;
   end record;
   procedure Parameter_Changed (Active_Actor : access Active_Actor_Type;
                                Parameter : Parameters.Parameter_Access);
   procedure On_Unload (Active_Actor : access Active_Actor_Type);
   procedure On_Start (Active_Actor : access Active_Actor_Type);
   procedure On_Stop (Active_Actor : access Active_Actor_Type);
   procedure On_Suspend (Active_Actor : access Active_Actor_Type);
   procedure On_Resume (Active_Actor : access Active_Actor_Type);
   procedure On_Initialise (Active_Actor : access Active_Actor_Type);
   procedure On_Reset_Com (Active_Actor : access Active_Actor_Type);

   procedure Distribute_Work
     (Active_Actor : access Active_Actor_Type);

   procedure Put_Sub_System_In_Error (Active_Actor : access Active_Actor_Type;
                                      Message : String);

end Narval.Actors.Actives;
