/* 
 * mumudvb - UDP-ize a DVB transport stream.
 * 
 * (C) 2008-2009 Brice DUBOST
 * 
 * The latest version can be found at http://mumudvb.braice.net
 * 
 * Copyright notice:
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/**@file
 * @brief Autoconfiguration structures
 */

#ifndef _AUTOCONF_H
#define _AUTOCONF_H

#include "mumudvb.h"
#include "unicast_http.h"
#include "ts.h"

/**find the audio and video pids from the PMT*/
#define AUTOCONF_MODE_PIDS 1
/**find the pmt pids and the channels from the pat, and go to AUTOCONF_MODE_PIDS*/
#define AUTOCONF_MODE_FULL 2

//timeout for autoconfiguration
#define AUTOCONFIGURE_TIME 10

/**@brief chained list of services for autoconfiguration
 *
*/
typedef struct mumudvb_service_t{
  /**The channel name*/
  char name[MAX_NAME_LEN];  
  /**Is the channel running ? Not used for the moment*/
  int running_status; 
  /**The service type : television, radio, data, ...*/
  int type; 
  /**The PMT pid of the service*/
  int pmt_pid; 
  /**The program ID, also called program number in the PAT or in ATSC*/
  int id;
  /**Tell if this service is scrambled*/
  int free_ca_mode;
  /**The next service in the chained list*/
  struct mumudvb_service_t *next;
}mumudvb_service_t;

/**@brief The different parameters used for autoconfiguration*/
typedef struct autoconf_parameters_t{
  /**Do we use autoconfiguration ?

Possible values for this variable

 0 : none (or autoconf finished)

 1 or AUTOCONF_MODE_PIDS : we have the PMT pids and the channels, we search the audio and video

 2 or AUTOCONF_MODE_FULL : we have only the tuning parameters, we search the channels and their pmt pids*/
  int autoconfiguration;
  /**do we autoconfigure the radios ?*/
  int autoconf_radios;
  /** The beginning of autoconfigured multicast addresses*/
  char autoconf_ip_header[10];
  /**When did we started autoconfiguration ?*/
  long time_start_autoconfiguration; 
  /**The transport stream id (used to read ATSC PSIP tables)*/
  int transport_stream_id;
  /** Do we autoconfigure scrambled channels ? */
  int autoconf_scrambled;
  /** Do we follow pmt changes*/
  int autoconf_pid_update;
  
  //Different packets used by autoconfiguration
  mumudvb_ts_packet_t *autoconf_temp_pat;
  mumudvb_ts_packet_t *autoconf_temp_sdt;
  /**For ATSC Program and System Information Protocol*/
  mumudvb_ts_packet_t *autoconf_temp_psip; /**@todo : see if it's really necesarry to split it from the sdt*/
  mumudvb_service_t   *services;

  /**the starting http unicast port */
  int autoconf_unicast_start_port;
  /**The unicast "HTTP" ip address*/
  char unicast_ipOut[20];
}autoconf_parameters_t;



int autoconf_init(autoconf_parameters_t *autoconf_vars, mumudvb_channel_t *channels,int number_of_channels);
int autoconf_read_pmt(mumudvb_ts_packet_t *pmt, mumudvb_channel_t *channel, int card, uint8_t *asked_pid, uint8_t *number_chan_asked_pid, fds_t *fds);
int autoconf_read_sdt(unsigned char *buf, int len, mumudvb_service_t *services);
int autoconf_read_psip(autoconf_parameters_t *);
void autoconf_freeing(autoconf_parameters_t *);
int autoconf_read_pat(autoconf_parameters_t *);
int autoconf_services_to_channels(autoconf_parameters_t parameters, mumudvb_channel_t *channels, int port, int card, unicast_parameters_t *unicast_vars, fds_t *fds);
int autoconf_finish_full(int *number_of_channels, mumudvb_channel_t *channels, autoconf_parameters_t *autoconf_vars, int common_port, int card, fds_t *fds,uint8_t *asked_pid, uint8_t *number_chan_asked_pid, int multicast_ttl, unicast_parameters_t *unicast_vars);
void autoconf_end(int card, int number_of_channels, mumudvb_channel_t *channels, uint8_t *asked_pid, uint8_t *number_chan_asked_pid, fds_t *fds);
void autoconf_free_services(mumudvb_service_t *services);
int pmt_need_update(mumudvb_channel_t *channel, unsigned char *buf, int ts_header);
void update_pmt_version(mumudvb_channel_t *channel);


#endif
