using System;
using System.Text;
using Muine.PluginLib;

namespace Muine.Audioscrobbler
{
  public enum SongStatus { Init, Loaded, Cached, Submitted }

  public class Song {

    public string Artist = "";
    public string Title = "";
    public string Album = "";
    public string MBID = "";
    public int Duration = 0;
    public DateTime Played;
    public SongStatus Status;
    public bool Processed; // true if the status hasn't changed

    public Song ()
    {
      Played = DateTime.MinValue;
      Status = SongStatus.Init;
      Processed = false;
    }

    public Song (ISong s)
    {
      if (s.Artists.Length > 0)
        Artist = s.Artists[0].Trim();
      Title = s.Title.Trim();
      Album = s.Album.Trim();
      Duration = s.Duration;
      Played = DateTime.MinValue;
      Status = SongStatus.Init;
      Processed = false;
    }

    public override string ToString()
    {
      return Artist + "\t" +
        Title + "\t" +
        Album + "\t" + 
        MBID + "\t" +
        Duration + "\t" +
        Played.ToString("s");
    }

    public string ToShortString()
    {
      StringBuilder s = new StringBuilder();
      if (Title != "")
        s.Append(Title);
      else
        s.Append("(Untitled)");

      if (Artist != "") {
        s.Append(" - " + Artist);
      }

      if (Album != "")
        s.Append(" (" + Album + ")");

      return s.ToString();
    }

    // Can throw Int32.Parse and DateTime.Parse exceptions
    public static Song Parse(string s) {
      string[] arr = s.Split('\t');
      if (arr.Length == 6) {
        Song song = new Song();
        song.Artist = arr[0];
        song.Title = arr[1];
        song.Album = arr[2];
        song.MBID = arr[3];
        song.Duration = Int32.Parse(arr[4]);
        song.Played = DateTime.Parse(arr[5]);
        return song;
      }
      else {
        throw new Exception("Bad song format: " + s);
      }
    }

    public string GetPostData(int index)
    {                
      // Generate POST data for updates:
      //	u - username
      //	s - md5 response
      //	a - artist
      //	t - title
      //	b - album
      //	m - musicbrainz id
      //	l - length (secs)
      //	i - time (UTC)

      string queueTime = String.Format("{0:0000}-{1:00}-{2:00} {3:00}:{4:00}:{5:00}", 
                                       Played.Year,
                                       Played.Month,
                                       Played.Day,
                                       Played.Hour,
                                       Played.Minute,
                                       Played.Second);
    
      return String.Format("a[{0}]={1}&t[{0}]={2}&b[{0}]={3}&m[{0}]={4}&l[{0}]={5}&i[{0}]={6}",
                           index, 
                           System.Web.HttpUtility.UrlEncode(Artist),
                           System.Web.HttpUtility.UrlEncode(Title),
                           System.Web.HttpUtility.UrlEncode(Album),
                           System.Web.HttpUtility.UrlEncode(MBID),
                           Duration,
                           System.Web.HttpUtility.UrlEncode(queueTime));
    }
  }
}
