/* Time-stamp: <2007-10-05 02:01:08 poser> */

/*
 * Copyright (C) 1993-2007 William J. Poser.
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of version 3 of the GNU General Public License
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"
#include "compdefs.h"
#include <stdlib.h>
#ifdef HAVE_UNINUM_UNICODE_H
#include <uninum/unicode.h>
#else
#include "unicode.h"
#endif
#ifdef HAVE_STDINT_H
#include <stdint.h>
#endif
#include <stdio.h>
#include <string.h>
#ifdef LOCALE_GETTEXT
#include <libintl.h>
#define _(x) gettext(x)
#else
#define _(x) (x)
#endif
#include <time.h>
#include <wchar.h>
#include <errno.h>
#include "retcodes.h"

#ifdef HAVE_LONGDOUBLE
#define DOUBLE (long double)
#else
#define DOUBLE double
#endif


/*
 * Return the difference between UTC and the local time zone, in seconds.
 * Add this value to local time to obtain UTC.
 *
 * We call localtime() rather than tzset because localtime is guaranteed by
 * POSIX to set timezone while tzset is not.
 */

static long
TimezoneOffset(void) {
  time_t lt;

  lt = time(NULL);
  (void) localtime(&lt);
  return((long) timezone);
}


/* 
 * Given a time as a string in one of the following formats, return the value in seconds:
 * 	hh
 * 	hhmm
 *	hh:mm
 * If the expression is ill-formed, set the error flag.
 */

static long BrokenTimeToSeconds (char *s, short *eflag) {
  int hours = 0;
  int minutes = 0;
  int FieldsFound;
  short status = 1;		/* No valid return value yet */


  FieldsFound = sscanf((char *)s,"%2u:%2u",&hours,&minutes);
  if(FieldsFound == 2) {
    status = 0;
    goto end;
  }

  FieldsFound = sscanf((char *)s,"%2u%2u",&hours,&minutes);
  if(FieldsFound == 2) {
    status = 0;
    goto end;
  }
  if(FieldsFound == 1) status = 0;

end:
  if (minutes > 59) status = 1;
  if (hours > 24) status = 1;
  *eflag = status;
  if (status == 0)  return (((hours * 60) + minutes) * 60);
  else return 0;
}

/* Extract the time zone info if any and return the number of seconds
 * that must be added to normalize the time to UTC. 
 * The time zone marker (Z,z,-,+) is ovewritten with a null,
 * modifying the original string.
 */

long ExtractTimeZone(UTF8 *s, short *eflag) {
  char *sep;
  long Offset;
  short err = 0;

  /* Z/z means the value is already UTC  */
  if ((sep = strchr((char *) s,'Z'))) {
    *sep = '\0';
    return 0;
  }

  if ((sep = strchr((char *) s,'z'))) {
    *sep = '\0';
    return 0;
  }

  /* + means subtract the following offset to get UTC  */
  if ((sep = strchr((char *) s,'+'))) {
    Offset = BrokenTimeToSeconds(sep+1,&err);
    *eflag = err;
    *sep = '\0';
    return (-Offset);
  }
  
  /* - means add the following offset to get UTC  */
  if ((sep = strchr((char *) s,'-'))) {
    Offset = BrokenTimeToSeconds(sep+1,&err);
    *eflag = err;
    *sep = '\0';
    return (Offset);
  }

  /*
   * If there is no time zone indicator, we default to the time zone of the
   * environment of this process.
   */

  return(TimezoneOffset());
}

/*
 * Remove hyphen-minus and colon from time-date strings
 * Return the adjusted location of the T.
 */

char *
StripTimeDateSeparators(char *s, char *tloc) {
  char *src;
  char *tgt;
  char c;
  int cnt = 0;			/* Number of hyphens stripped */

  src = tgt = s;
  while (src < tloc) {
    c = *src++;
    if (c == '-') {cnt++;continue;}
    *tgt++ = c;
  }
  while ( (c = *src++) != '\0') {
    if (c == ':') continue;
    *tgt++ = c;
  }
  *tgt = '\0';
  return(tloc-cnt);
}
