/*
 *   Written by Bradley Broom (2003).
 *
 *   Copyright (c) 2003 Bradley Broom
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "MRW_Private.h"
#include "MRW_Loader.h"
#include <math.h>

static char *
MRW_GetCameraName (const MRI *mri)
{
    const char *retval = MRW_CameraName ((MRW_Header *)mri->hdr);
    return (char *)retval;
}

static int
MRW_HasPackedPixels (const MRI *mri)
{
    char * name = MRW_CameraName ((const MRW_Header *)mri->hdr);
    return strcmp (name, "DiMAGE A1") == 0
           || strcmp (name, "DiMAGE A2") == 0;
}

static int
MRW_GetFlash (const MRI *mri)
{
    int retval = MRW_Flash ((MRW_Header *)mri->hdr);
    return retval;
}

static int
MRW_GetWidth (const MRI *mri)
{
    int retval = MRW_Width ((MRW_Header *)mri->hdr);
    return retval;
}

static int
MRW_GetHeight (const MRI *mri)
{
    int retval = MRW_Height ((MRW_Header *)mri->hdr);
    return retval;
}

static char *
MRW_GetTimestamp (const MRI *mri)
{
    const char *retval = MRW_DateTime ((MRW_Header *)mri->hdr);
    return (char *)retval;
}

static double
MRW_GetShutter (const struct _MRI *mri)
{
    double retval = MRW_Shutter ((MRW_Header *)mri->hdr);
    return retval;
}

static double
MRW_GetAperture (const struct _MRI *mri)
{
    double retval = MRW_Aperture ((MRW_Header *)mri->hdr);
    return retval;
}

static double
MRW_GetISO (const struct _MRI *mri)
{
    double retval = MRW_ISO ((MRW_Header *)mri->hdr);
    return retval;
}

static double
MRW_GetFocalLen (const struct _MRI *mri)
{
    double retval = MRW_FocalLen35mm ((MRW_Header *)mri->hdr);
    return retval;
}

static int
MRW_GetFocusMode (const struct _MRI *mri)
{
	MRW_Header *	hdr = (MRW_Header *)mri->hdr;
	return MRW_FocusMode (hdr);
}

static double
MRW_GetFocusLen (const struct _MRI *mri)
{
    double retval;
    retval = MRW_FocusLen ((MRW_Header *)mri->hdr);
    if (retval == 0.0 && MRW_GetFocusMode (mri) == MRI_FOCUS_AUTO)
    	retval = -1;  /* Don't know how to find this (yet). */
    return retval;
}

static cmsCIEXYZTRIPLE dimagePrimaries = {
	{ 0.581818, 0.241455, 0.123322 },
	{ 0.333633, 0.924133,-0.236435 },
	{ 0.048737,-0.165588, 0.938004 }
};

void
MRW_AdjustBalanceLuminance (const struct _MRI *mri, MRI_balance *balanceSpec, double *luma)
{
   unsigned short maxgain = balanceSpec->rgain;
   double scale;
#if 0
   fprintf (stderr, "Balance in: %d %d %d\n", balanceSpec->rgain,
	    balanceSpec->ggain, balanceSpec->bgain);
#endif
   if (balanceSpec->ggain > maxgain) maxgain = balanceSpec->ggain;
   *luma = balanceSpec->ggain;
#if 0
   if (balanceSpec->bgain > maxgain) maxgain = balanceSpec->bgain;
#endif
   scale = maxgain / (balanceSpec->ggain / (3965.0/4095.0));
   balanceSpec->rgain = balanceSpec->rgain / scale + 0.5;
   balanceSpec->ggain = balanceSpec->ggain / scale + 0.5;
   balanceSpec->bgain = balanceSpec->bgain / scale + 0.5;
   *luma /= (dimagePrimaries.Red.Y * balanceSpec->rgain + dimagePrimaries.Green.Y * balanceSpec->ggain /* + dimagePrimaries.Blue.Y * balanceSpec->bgain */ );
#if 0
   fprintf (stderr, "Balance out: %d %d %d (lumascale = %g)\n", balanceSpec->rgain,
	    balanceSpec->ggain, balanceSpec->bgain, *luma);
#endif
}

static cmsCIEXYZ dimageWhitePt = { 0.963577, 1.000000, 0.823654 };
static cmsCIEXYZ dimageWhitePt_d7i = { 0.96909, 1.000000, 0.87569 };
/* static cmsCIEXYZ dimageIlluminant = { 0.964188, 1.000000, 0.824875 }; */

static double bTRC[512] = {
#include "btrc.dat"
};
static double gTRC[512] = {
#include "gtrc.dat"
};
static double rTRC[512] = {
#include "rtrc.dat"
};

LPcmsCIEXYZ
MRW_GetWhitePoint (const struct _MRI *mri)
{
	const char *cameraName = MRW_GetCameraName (mri);

	if (strcmp (cameraName, "DiMAGE 7i") == 0)
		return &dimageWhitePt_d7i;
	else
		return &dimageWhitePt;
}

cmsHPROFILE
MRW_GetNativeProfile (const struct _MRI *mri)
{
	LPcmsCIEXYZ whitePoint = MRW_GetWhitePoint (mri);
	cmsCIExyY wPt;
	cmsCIExyYTRIPLE primaries;
	cmsHPROFILE dimageProfile;
	LPGAMMATABLE transferFunction[3];
	SAMPLEDCURVE rcrv, gcrv, bcrv;

	cmsXYZ2xyY (&wPt, whitePoint);
	cmsXYZ2xyY (&primaries.Red, &dimagePrimaries.Red);
	cmsXYZ2xyY (&primaries.Green, &dimagePrimaries.Green);
	cmsXYZ2xyY (&primaries.Blue, &dimagePrimaries.Blue);

	/* These are [rgb]TRC */
	rcrv.nItems = 512; rcrv.Values = rTRC;
	gcrv.nItems = 512; gcrv.Values = gTRC;
	bcrv.nItems = 512; bcrv.Values = bTRC;
	transferFunction[0] = cmsConvertSampledCurveToGamma(&rcrv, 1.0);
	transferFunction[1] = cmsConvertSampledCurveToGamma(&gcrv, 1.0);
	transferFunction[2] = cmsConvertSampledCurveToGamma(&bcrv, 1.0);
	dimageProfile = cmsCreateRGBProfile(&wPt, &primaries, transferFunction);
	_cmsAddTextTag(dimageProfile, icSigProfileDescriptionTag, "dimage (built-in)");
	return dimageProfile;
}

static unsigned short
applyTone (double *trc, unsigned short val)
{
	double v = (val * 511.0) / 65535.0;
	unsigned idx = v;
	unsigned result;
	double w;

	if (idx > 510) { idx= 510; w = 1; }
	else { w = v - idx; }
	result = ((trc[idx] + w*(trc[idx+1]-trc[idx])) * 65535.0 + 0.5);
	return result > 65535 ? 65535 : result;
}

static float
applyToneF (double *trc, float val)
{
	double v = (val * 511.0) / 65535.0;
	unsigned idx = v;
	unsigned result;
	double w;

	if (idx > 510) { idx= 510; w = 1; }
	else { w = v - idx; }
	result = ((trc[idx] + w*(trc[idx+1]-trc[idx])) * 65535.0);
	return result;
}

/* f(t) = t1/3      		for t > 0.008856
 * f(t) = 7.787 * t + 16/116    otherwise
 */
static double f(double t)
{
	if (t > 0.008856)
		return pow (t, 1.0/3.0);
	else
		return 7.787 * t + 16.0/116.0;
}

/* L* = 116 * (Y/Yn)1/3 - 16    for Y/Yn > 0.008856
 * L* = 903.3 * Y/Yn             otherwise
 *
 * a* = 500 * ( f(X/Xn) - f(Y/Yn) )
 * b* = 200 * ( f(Y/Yn) - f(Z/Zn) )
 */
static void
myXYZ2Lab (LPcmsCIEXYZ wPt, LPcmsCIELab Lab, LPcmsCIEXYZ XYZ)
{
	double YoverYn = XYZ->Y / wPt->Y;
	double fYoverYn = f (YoverYn);

	if (YoverYn > 0.008856)
		Lab->L = 116.0 * pow (YoverYn, 1.0/3.0) - 16.0;
	else
		Lab->L = 903.3 * YoverYn;

	Lab->a = 500.0 * ( f(XYZ->X/wPt->X) - fYoverYn );
	Lab->b = 200.0 * ( fYoverYn - f(XYZ->Z/wPt->Z) );

	if (Lab->L < 0.0) {
		Lab->L = 0.0;
		Lab->a = Lab->b = 0.0;
	}
	else if (Lab->L > 100.0) {
		Lab->L = 100.0;
		Lab->a = Lab->b = 0.0;
	}
	if (Lab->a < -128.0) {
		Lab->b *= -128.0/Lab->a;
		Lab->a = -128.0;
	}
	else if (Lab->a > 127.96) {
		Lab->b *= 127.97/Lab->a;
		Lab->a = 127.96;
	}
	if (Lab->b < -128.0) {
		Lab->a *= -128.0/Lab->b;
		Lab->b = -128.0;
	} else if (Lab->b > 127.96) {
		Lab->a *= 127.96/Lab->b;
		Lab->b = 127.96;
	}
}

static void
MRW_ConvertScanLineToLABShort (const MRI *mri, struct MRI_ScanLine *sl)
{
	LPcmsCIEXYZ whitePoint = MRW_GetWhitePoint (mri);
	unsigned short tin[3], tout[3];
	unsigned short *R = sl->R;
	unsigned short *G = sl->G;
	unsigned short *B = sl->B;
	cmsCIELab Lab;
	int j;
	cmsCIEXYZ XYZ;
#if 0
	unsigned short xyz[3];
	cmsCIEXYZ adaptedXYZ;
#endif

	for (j = 0; j < sl->sl_Width; j++) {
#if 1
		tin[0] = applyTone (rTRC, R[j]);
		tin[1] = applyTone (gTRC, G[j]);
		tin[2] = applyTone (bTRC, B[j]);
#else
		tin[0] = R[j];
		tin[1] = G[j];
		tin[2] = B[j];
#endif
#if 0
		if (sl->masks[j] & 7) {
		    unsigned short m = tin[0] > tin[1] ? tin[0] : tin[1];
		    if (tin[2] > m) m = tin[2];
		    if (sl->masks[j] & 1) tin[0] = m;
		    if (sl->masks[j] & 2) tin[1] = m;
		    if (sl->masks[j] & 4) tin[2] = m;
		}
#endif
#if 0
		cmsDoTransform(wd->hTransform, tin, tout, 1);
		cmsDoTransform (wd->hTransform, tin, xyz, 1);
		cmsXYZEncoded2Float (&XYZ, xyz);
		cmsAdaptToIlluminant (&adaptedXYZ, wd->whitePoint, &dimageIlluminant, &XYZ);
		cmsXYZ2Lab (wd->whitePoint, &Lab, &adaptedXYZ);
#else
		XYZ.X = dimagePrimaries.Red.X * tin[0] / 65535.0 +
			dimagePrimaries.Green.X * tin[1] / 65535.0 +
			dimagePrimaries.Blue.X * tin[2] / 65535.0;
		XYZ.Y = dimagePrimaries.Red.Y * tin[0] / 65535.0 +
			dimagePrimaries.Green.Y * tin[1] / 65535.0 +
			dimagePrimaries.Blue.Y * tin[2] / 65535.0;
		XYZ.Z = dimagePrimaries.Red.Z * tin[0] / 65535.0 +
			dimagePrimaries.Green.Z * tin[1] / 65535.0 +
			dimagePrimaries.Blue.Z * tin[2] / 65535.0;
		myXYZ2Lab (whitePoint, &Lab, &XYZ);
#endif
		cmsFloat2LabEncoded (tout, &Lab);
		R[j] = tout[0];
		G[j] = tout[1];
		B[j] = tout[2];
	}
}

static void
MRW_ConvertScanLineToLABFloat (const MRI *mri, struct MRI_ScanLine *sl)
{
	LPcmsCIEXYZ whitePoint = MRW_GetWhitePoint (mri);
	float tin[3];
	unsigned short tout[3];
	float *R = sl->R;
	float *G = sl->G;
	float *B = sl->B;
	cmsCIELab Lab;
	int j;
	cmsCIEXYZ XYZ;
#if 0
	unsigned short xyz[3];
	cmsCIEXYZ adaptedXYZ;
#endif

	for (j = 0; j < sl->sl_Width; j++) {
#if 1
		tin[0] = applyToneF (rTRC, R[j]);
		tin[1] = applyToneF (gTRC, G[j]);
		tin[2] = applyToneF (bTRC, B[j]);
#else
		tin[0] = R[j];
		tin[1] = G[j];
		tin[2] = B[j];
#endif
#if 0
		if (sl->masks[j] & 7) {
		    unsigned short m = tin[0] > tin[1] ? tin[0] : tin[1];
		    if (tin[2] > m) m = tin[2];
		    if (sl->masks[j] & 1) tin[0] = m;
		    if (sl->masks[j] & 2) tin[1] = m;
		    if (sl->masks[j] & 4) tin[2] = m;
		}
#endif
#if 0
		cmsDoTransform(wd->hTransform, tin, tout, 1);
		cmsDoTransform (wd->hTransform, tin, xyz, 1);
		cmsXYZEncoded2Float (&XYZ, xyz);
		cmsAdaptToIlluminant (&adaptedXYZ, wd->whitePoint, &dimageIlluminant, &XYZ);
		cmsXYZ2Lab (wd->whitePoint, &Lab, &adaptedXYZ);
#else
		XYZ.X = dimagePrimaries.Red.X * tin[0] / 65535.0 +
			dimagePrimaries.Green.X * tin[1] / 65535.0 +
			dimagePrimaries.Blue.X * tin[2] / 65535.0;
		XYZ.Y = dimagePrimaries.Red.Y * tin[0] / 65535.0 +
			dimagePrimaries.Green.Y * tin[1] / 65535.0 +
			dimagePrimaries.Blue.Y * tin[2] / 65535.0;
		XYZ.Z = dimagePrimaries.Red.Z * tin[0] / 65535.0 +
			dimagePrimaries.Green.Z * tin[1] / 65535.0 +
			dimagePrimaries.Blue.Z * tin[2] / 65535.0;
		myXYZ2Lab (whitePoint, &Lab, &XYZ);
#endif
		cmsFloat2LabEncoded (tout, &Lab);
		R[j] = tout[0];
		G[j] = tout[1];
		B[j] = tout[2];
	}
}

static void
MRW_ConvertScanLineToLAB (const MRI *mri, struct MRI_ScanLine *sl)
{
    if (sl->scanLineType == LINETYPE_SHORT)
        MRW_ConvertScanLineToLABShort (mri, sl);
    else if (sl->scanLineType == LINETYPE_FLOAT)
        MRW_ConvertScanLineToLABFloat (mri, sl);
}

static MRI_Methods mrw_methods = {
        MRW_GetCameraName,
	MRW_GetHeight,
	MRW_GetWidth,
	MRW_GetPresetBalance,
	MRW_AdjustBalanceLuminance,
	MRW_GetFlash,
        MRW_GetTimestamp,
	MRW_GetShutter,
	MRW_GetAperture,
	MRW_GetISO,
	MRW_GetFocalLen,
	MRW_GetFocusLen,
	MRW_GetFocusMode,
	MRW_PresetIterator,
	MRW_GetWhitePoint,
	MRW_GetNativeProfile,
	MRW_ConvertScanLineToLAB,
};

static void
SetColorSpace (MRI *mri)
{
#if 1
        mri->colorSpace = MRI_NATIVE_PROFILE;
#else
	if (strcmp (cameraName, "DiMAGE 7") == 0) {
	     mri->colorSpace = "MLTDim7r.icc";
	}
	else if (strcmp (cameraName, "DiMAGE 7i") == 0) {
	     mri->colorSpace = "DiMAGE 7i_rprof.icc"; /* Not sure about these. */
	}
	else if (strcmp (cameraName, "DiMAGE 7Hi") == 0) {
	     mri->colorSpace = "AdobeRGB1998.icc"; /* Should be based on mode?? */
	}
	else if (strcmp (cameraName, "DiMAGE 5") == 0) {
	     mri->colorSpace = "MLTDim5r.icc";
	}
	else {
	    extern char *progname;
	    const char *cameraName = MRW_CameraName ((MRW_Header *)mri->hdr);

	    fprintf (stderr, "%s: Warning: Minolta raw file appears to come from unknown camera %s\n", progname, cameraName);
	    mri->colorSpace = "MLTDim5r.icc";
	}
#endif
}

unsigned short
iscale (unsigned short x)
{
#if 1
    return x << 4;
#else
    if (x > 3965)
        return 65535;
    else
        return (unsigned short)(((unsigned int)x * 1083204) >> 16);
#endif
}

static void
ReadRasterDataFromFile (MRI *mri, FILE *f)
{
	int y, height = MRI_GetHeight (mri);
	int x, width = MRI_GetWidth (mri);
	unsigned short **R, **G, **B;
	unsigned char *buf;
	int truncate;
	extern char *progname;

	mri->height = height;
	mri->width = width;
	MRI_Allocate (mri);
	R = mri->R;
	G = mri->G;
	B = mri->B;
	truncate = 0;
	buf = (unsigned char *)malloc (2 * width);
	if (buf == (unsigned char *)0) {
		fprintf (stderr, "%s: Unable to allocate temporary memory.\n", progname);
		exit (1);
	}
	for (y = 0; y < height; y+=2) {
		int i;
		unsigned char *p;

		/* Read a row of RGR..G pixels. */
		if ((i = fread (buf, 2, width, f)) != width) {
			truncate = 1;
			memset (&buf[2*i], '\0', (width-i)*2);
		}
		p = buf;
		memset (B[y], '\0', width*2);
		for (x = 0; x < width; x += 2) {
			/* Copy in a red pixel. */
			R[y][x] = iscale ((p[0] << 8) | p[1]);
			G[y][x] = 0;
			/* Copy in a green pixel. */
			R[y][x+1] = 0;
			G[y][x+1] = iscale ((p[2] << 8) | p[3]);
			/* Advance. */
			p += 4;
		}
		if (truncate) break;

		/* Read a row of GB..GB pixels. */
		if ((i = fread (buf, 2, width, f)) != width) {
			truncate = 1;
			memset (&buf[2*i], '\0', (width-i)*2);
		}
		p = buf;
		/* Assume compiler can optimize away use of x. */
		memset (R[y+1], '\0', width*2);
		for (x = 0; x < width; x += 2) {
			/* Copy in a green pixel. */
			G[y+1][x] = iscale ((p[0] << 8) | p[1]);
			B[y+1][x] = 0;
			/* Copy in a blue pixel. */
			G[y+1][x+1] = 0;
			B[y+1][x+1] = iscale ((p[2] << 8) | p[3]);
			/* Advance. */
			p += 4;
		}
		if (truncate) { y++; break; }
	}
	if (truncate) {
		fprintf (stderr, "%s: Warning: Unable to read entire image. Truncating at line %d...\n", progname, y);
		for (; y < mri->height; y++)
			for (x = 0; x < mri->width; x++)
				R[y][x] = G[y][x] = B[y][x] = 0;
	}

	free (buf);
}

void
unpack_pixels (unsigned char *b, unsigned char *p, int width)
{
	int i;
	int v;

	for (i = 0; i < width; i += 2) {
		v = (p[0] << 4) | (p[1] >> 4);
		b[0] = v >> 8;
		b[1] = v & 0xFF;
		b[2] = p[1] & 0x0F;
		b[3] = p[2];
		p += 3;
		b += 4;
	}
}

static void
ReadPackedRasterDataFromFile (MRI *mri, FILE *f)
{
	int y, height = MRI_GetHeight (mri);
	int x, width = MRI_GetWidth (mri);
	unsigned short **R, **G, **B;
	unsigned char *pbuf;
	unsigned char *buf;
	int truncate;
	int rawlength;
	extern char *progname;

	mri->height = height;
	mri->width = width;
	MRI_Allocate (mri);
	R = mri->R;
	G = mri->G;
	B = mri->B;
	truncate = 0;
	rawlength = 3 * width/2;
	pbuf = (unsigned char *)malloc (rawlength);
	buf = (unsigned char *)malloc (width*2);
	if (pbuf == NULL || buf == NULL) {
		fprintf (stderr, "%s: Unable to allocate temporary memory.\n", progname);
		exit (1);
	}
	for (y = 0; y < height; y+=2) {
		int i;
		unsigned char *p;

		/* Read a row of RGR..G pixels. */
		if ((i = fread (pbuf, 1, rawlength, f)) != rawlength) {
			truncate = 1;
			memset (&pbuf[i], '\0', rawlength-i);
		}
		unpack_pixels (buf, pbuf, width);
		p = buf;
		memset (B[y], '\0', width*2);
		for (x = 0; x < width; x += 2) {
			/* Copy in a red pixel. */
			R[y][x] = iscale ((p[0] << 8) | p[1]);
			G[y][x] = 0;
			/* Copy in a green pixel. */
			R[y][x+1] = 0;
			G[y][x+1] = iscale ((p[2] << 8) | p[3]);
			/* Advance. */
			p += 4;
		}
		if (truncate) break;

		/* Read a row of GB..GB pixels. */
		if ((i = fread (pbuf, 1, rawlength, f)) != rawlength) {
			truncate = 1;
			memset (&pbuf[i], '\0', rawlength-i);
		}
		unpack_pixels (buf, pbuf, width);
		p = buf;
		/* Assume compiler can optimize away use of x. */
		memset (R[y+1], '\0', width*2);
		for (x = 0; x < width; x += 2) {
			/* Copy in a green pixel. */
			G[y+1][x] = iscale ((p[0] << 8) | p[1]);
			B[y+1][x] = 0;
			/* Copy in a blue pixel. */
			G[y+1][x+1] = 0;
			B[y+1][x+1] = iscale ((p[2] << 8) | p[3]);
			/* Advance. */
			p += 4;
		}
		if (truncate) { y++; break; }
	}
	if (truncate) {
		fprintf (stderr, "%s: Warning: Unable to read entire image. Truncating at line %d...\n", progname, y);
		for (; y < mri->height; y++)
			for (x = 0; x < mri->width; x++)
				R[y][x] = G[y][x] = B[y][x] = 0;
	}

	free (buf);
	free (pbuf);
}


MRI *
MRW_Loader (FILE *f, char **errmsg)
{
	MRW_Header *hdr ;
	MRI *mri;

	if ((hdr = (MRW_Header *)malloc(sizeof(*hdr))) == (MRW_Header *)0) {
		*errmsg = "MRW_LoadImage: unable to allocate memory for MRW header.";
		return (MRI *)0;
	}
	if (!LoadMRW (hdr, f, errmsg)) {
		free (hdr);
		*errmsg = "MRW_LoadImage: not a Minolta RAW file.";
		return (MRI *)0;
	}

	mri = (MRI *)malloc(sizeof(*mri));
	if (mri == (MRI *)0) {
		*errmsg = "MRW_LoadImage: out of memory!";
		return (MRI *)0;
	}
	mri->methods = &mrw_methods;
	mri->hdr = (void *)hdr;
	mri->data = NULL;
	mri->dFactor = 0;
	mri->wBalance.rgain = mri->wBalance.ggain = mri->wBalance.bgain = 256;
	SetColorSpace (mri);
	if (MRW_HasPackedPixels (mri))
	    ReadPackedRasterDataFromFile (mri, f);
	else
	    ReadRasterDataFromFile (mri, f);

	{
		int Rmax = 0, Gmax = 0, Bmax = 0;
		double Ravg = 0.0, Gavg = 0.0, Bavg = 0.0;
		int x, y;
		int width = MRW_GetWidth (mri);
		int height = MRW_GetHeight (mri);

		for (x = 0; x < width; x++)
		    for (y = 0; y < height; y++) {
			Ravg += mri->R[y][x];
		    	if (mri->R[y][x] > Rmax) Rmax = mri->R[y][x];
			Gavg += mri->G[y][x];
		    	if (mri->G[y][x] > Gmax) Gmax = mri->G[y][x];
			Bavg += mri->B[y][x];
		    	if (mri->B[y][x] > Bmax) Bmax = mri->B[y][x];
		    }
		mri->Rmax = Rmax;
		mri->Gmax = Gmax;
		mri->Bmax = Bmax;
		Ravg /= (width * height);
		Gavg /= (width * height);
		Bavg /= (width * height);
		Ravg *= 4;
		Gavg *= 2;
		Bavg *= 4;
		mri->Ravg = Ravg;
		mri->Gavg = Gavg;
		mri->Bavg = Bavg;
#if 0
		fprintf (stderr, "Rmax=%d, Gmax=%d, Bmax=%d\n", Rmax, Gmax, Bmax);
		fprintf (stderr, "Ravg=%g, Gavg=%g, Bavg=%g\n", Ravg, Gavg, Bavg);
#endif
	}
	return mri;
}
