/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2011  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/slam.h>
#include <mrpt/gui.h>

using namespace mrpt;
using namespace mrpt::slam;
using namespace mrpt::poses;
using namespace mrpt::opengl;
using namespace mrpt::system;
using namespace mrpt::math;
using namespace mrpt::utils;
using namespace std;


template <class GRAPHTYPE>
void display_graph(const GRAPHTYPE & g)
{
	// Convert into a 3D representation:
	TParametersDouble  params;
	CSetOfObjectsPtr objGraph = mrpt::opengl::graph_tools::graph_visualize(g,params);

	// Show in a window:
	mrpt::gui::CDisplayWindow3D  win("graph-slam - Graph visualization",700,600);

	// Set camera pointing to the bottom-center of all nodes.
	//  Since a CGridPlaneXY object is generated by graph_visualize()
	//  already centered at the bottom, use that information to avoid
	//  recomputing it again here:
	win.setCameraElevationDeg(75);
	{
		opengl::CGridPlaneXYPtr obj_grid = objGraph->CSetOfObjects::getByClass<CGridPlaneXY>();
		if (obj_grid)
		{
			float x_min,x_max, y_min,y_max;
			obj_grid->getPlaneLimits(x_min,x_max, y_min,y_max);
			const float z_min = obj_grid->getPlaneZcoord();
			win.setCameraPointingToPoint( 0.5*(x_min+x_max), 0.5*(y_min+y_max), z_min );
			win.setCameraZoom( 2.0f * std::max(10.0f, std::max(x_max-x_min, y_max-y_min) ) );
		}
	}

	mrpt::opengl::COpenGLScenePtr &scene = win.get3DSceneAndLock();
	scene->insert(objGraph);

	win.addTextMessage(2, 2+14*0,"'q': Quit", TColorf(0.8,0.8,0.8), 1000, MRPT_GLUT_BITMAP_HELVETICA_12 );
	win.addTextMessage(2, 2+14*1,"'i': Switch view/hide node IDs", TColorf(1,1,1), 1001, MRPT_GLUT_BITMAP_HELVETICA_12 );
	win.addTextMessage(2, 2+14*2,"'e': Switch view/hide edges", TColorf(1,1,1), 1002, MRPT_GLUT_BITMAP_HELVETICA_12 );
	win.addTextMessage(2, 2+14*3,"'c': Switch view/hide node corners", TColorf(1,1,1), 1003, MRPT_GLUT_BITMAP_HELVETICA_12 );
	win.addTextMessage(2, 2+14*4,"+/-: Increase/reduce size of node points", TColorf(1,1,1), 1004, MRPT_GLUT_BITMAP_HELVETICA_12 );
	win.addTextMessage(2, 2+14*5,"o/p: Increase/reduce length of node corners", TColorf(1,1,1), 1005, MRPT_GLUT_BITMAP_HELVETICA_12 );

	win.unlockAccess3DScene();
	win.repaint();

	// Create a feedback class for keystrokes:
	struct Win3D_observer : public mrpt::utils::CObserver
	{
		const GRAPHTYPE   &m_graph;
		CSetOfObjectsPtr   m_new_3dobj;
		TParametersDouble  params;  // for "graph_visualize()"

		bool  request_to_refresh_3D_view;
		bool  request_to_quit;

		Win3D_observer(const GRAPHTYPE &g) :
			m_graph(g),
			request_to_refresh_3D_view(false),
			request_to_quit(false)
		{
			params["show_edges"] = 1;
			params["show_node_corners"] = 1;
			params["nodes_corner_scale"] = 0.7;
		}

	protected:
		/** This virtual function will be called upon receive of any event after starting listening at any CObservable object. */
		virtual void OnEvent(const mrptEvent &e)
		{
			if (e.isOfType<mrpt::gui::mrptEventWindowChar>())
			{
				const mrpt::gui::mrptEventWindowChar * ev = e.getAs<mrpt::gui::mrptEventWindowChar>();
				switch (ev->char_code)
				{
				case 'q':
				case 'Q':
					request_to_quit=true;
					break;
				case 'i':
				case 'I':
					params["show_ID_labels"] = params["show_ID_labels"]!=0 ? 0:1;
					m_new_3dobj = mrpt::opengl::graph_tools::graph_visualize(m_graph,params);
					request_to_refresh_3D_view = true;
					break;
				case 'e':
				case 'E':
					params["show_edges"] = params["show_edges"]!=0 ? 0:1;
					m_new_3dobj = mrpt::opengl::graph_tools::graph_visualize(m_graph,params);
					request_to_refresh_3D_view = true;
					break;
				case 'c':
				case 'C':
					params["show_node_corners"] = params["show_node_corners"]!=0 ? 0:1;
					m_new_3dobj = mrpt::opengl::graph_tools::graph_visualize(m_graph,params);
					request_to_refresh_3D_view = true;
					break;
				case '+':
					params["nodes_point_size"]+=1.0;
					m_new_3dobj = mrpt::opengl::graph_tools::graph_visualize(m_graph,params);
					request_to_refresh_3D_view = true;
					break;
				case '-':
					params["nodes_point_size"]=std::max(0.0, params["nodes_point_size"] - 1.0 );
					m_new_3dobj = mrpt::opengl::graph_tools::graph_visualize(m_graph,params);
					request_to_refresh_3D_view = true;
					break;
				case 'p':
					params["nodes_corner_scale"]*=1.2;
					m_new_3dobj = mrpt::opengl::graph_tools::graph_visualize(m_graph,params);
					request_to_refresh_3D_view = true;
					break;
				case 'o':
					params["nodes_corner_scale"]*=1.0/1.2;
					m_new_3dobj = mrpt::opengl::graph_tools::graph_visualize(m_graph,params);
					request_to_refresh_3D_view = true;
					break;
				};
			}
		}
	};
	// Create an instance of the observer:
	Win3D_observer  win_feedback(g);
	win_feedback.observeBegin(win);

	cout << "Close the window to exit.\n";
	while (win.isOpen())
	{
		if (win_feedback.request_to_refresh_3D_view)
		{
			// Replace 3D representation:
			win.get3DSceneAndLock();
			*objGraph = *win_feedback.m_new_3dobj;  // This overwrites the object POINTER BY the smart pointer "objGraph".
			win.unlockAccess3DScene();
			win.repaint();
			win_feedback.request_to_refresh_3D_view = false;
		}
		if (win_feedback.request_to_quit)
			break;
		mrpt::system::sleep(10);
	}
}

// Explicit instantiations:
template void display_graph<CNetworkOfPoses2D>(const CNetworkOfPoses2D & g);
template void display_graph<CNetworkOfPoses3D>(const CNetworkOfPoses3D & g);
template void display_graph<CNetworkOfPoses2DInf>(const CNetworkOfPoses2DInf & g);
template void display_graph<CNetworkOfPoses3DInf>(const CNetworkOfPoses3DInf & g);
