/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.

#include <mrpt/topography/conversions.h>

#include <mrpt/poses/CPoint3D.h>
#include <mrpt/poses/CPose3D.h>
#include <mrpt/math_mrpt.h>

using namespace std;
using namespace mrpt;
using namespace mrpt::math;
using namespace mrpt::utils;
using namespace mrpt::poses;


/*---------------------------------------------------------------
					coordinatesTransformation_WGS84
 ---------------------------------------------------------------*/
void  mrpt::topography::coordinatesTransformation_WGS84(
	double		in_longitude_degrees,
	double		in_latitude_degrees,
	double		in_height_meters,
	double		&out_x_meters,
	double		&out_y_meters,
	double		&out_z_meters,
	double		in_longitude_reference_degrees,
	double		in_latitude_reference_degrees,
	double		in_height_reference_meters
	)
{
	// --------------------------------------------------------------------
	//  Explanation: We compute the earth-centric coordinates first,
	//    then make a system transformation to local XYZ coordinates
	//    using a system of three orthogonal vectors as local reference.
	//
	// See: http://en.wikipedia.org/wiki/Reference_ellipsoid
	// (JLBC 21/DEC/2006)  (Fixed: JLBC 9/JUL/2008)
	// --------------------------------------------------------------------
	// Generate 3D point:
	TPoint3D	P_geocentric;
	coordinatesTransformation_WGS84_geocentric(
		in_longitude_degrees, in_latitude_degrees, in_height_meters,
		P_geocentric.x, P_geocentric.y, P_geocentric.z );

	// Generate reference 3D point:
	vector_double   P_ref(3);
	coordinatesTransformation_WGS84_geocentric(
		in_longitude_reference_degrees, in_latitude_reference_degrees, in_height_reference_meters,
		P_ref[0], P_ref[1], P_ref[2] );

	// Z axis -> In direction out-ward the center of the Earth:
	vector_double	REF_X(3),REF_Y(3),REF_Z(3);
	math::normalize(P_ref, REF_Z);

	// 1st column: Starting at the reference point, move in the tangent direction
	//   east-ward: I compute this as the derivative of P_ref wrt "longitude":
	//      A_east[0] =-(N+in_height_meters)*cos(lat)*sin(lon);  --> -Z[1]
	//      A_east[1] = (N+in_height_meters)*cos(lat)*cos(lon);  -->  Z[0]
	//      A_east[2] = 0;                                       -->  0
	// ---------------------------------------------------------------------------
	vector_double AUX_X(3);
	AUX_X[0]=-REF_Z[1];
	AUX_X[1]= REF_Z[0];
	AUX_X[2]= 0;
	math::normalize(AUX_X, REF_X);

	// 2nd column: The cross product:
	math::crossProduct3D(REF_Z, REF_X, REF_Y);

	// Compute the resulting relative coordinates:
	// For using smaller numbers:
	P_geocentric.x -= P_ref[0];
	P_geocentric.y -= P_ref[1];
	P_geocentric.z -= P_ref[2];

	// Optimized calculation: Local transformed coordinates of P_geo(x,y,z)
	//   after rotation given by the three vectors REF_X,Y,Z:
	TPoint3D P_local;
	P_local.x = REF_X[0]*P_geocentric.x+REF_X[1]*P_geocentric.y+REF_X[2]*P_geocentric.z;
	P_local.y = REF_Y[0]*P_geocentric.x+REF_Y[1]*P_geocentric.y+REF_Y[2]*P_geocentric.z;
	P_local.z = REF_Z[0]*P_geocentric.x+REF_Z[1]*P_geocentric.y+REF_Z[2]*P_geocentric.z;

	// Relative position:
	out_x_meters	= P_local.x;
	out_y_meters	= P_local.y;
	out_z_meters	= P_local.z;
}


/*---------------------------------------------------------------
					ENU_axes_from_WGS84
 ---------------------------------------------------------------*/
void mrpt::topography::ENU_axes_from_WGS84(
	double		in_longitude_reference_degrees,
	double		in_latitude_reference_degrees,
	double		in_height_reference_meters,
	mrpt::math::TPose3D		&out_ENU,
	bool					only_angles
	)
{
	// See "coordinatesTransformation_WGS84" for more comments.

	vector_double   P_ref(3);
	coordinatesTransformation_WGS84_geocentric(
		in_longitude_reference_degrees, in_latitude_reference_degrees, in_height_reference_meters,
		P_ref[0], P_ref[1], P_ref[2] );

	// Z axis -> In direction out-ward the center of the Earth:
	vector_double	REF_X(3),REF_Y(3),REF_Z(3);
	math::normalize(P_ref, REF_Z);

	vector_double AUX_X(3);
	AUX_X[0]=-REF_Z[1];
	AUX_X[1]= REF_Z[0];
	AUX_X[2]= 0;
	math::normalize(AUX_X, REF_X);

	math::crossProduct3D(REF_Z, REF_X, REF_Y);

	CMatrixDouble44	 HM;

	for (size_t i=0;i<3;i++)  
	{
		HM(i,0) = REF_X[i];
		HM(i,1) = REF_Y[i];
		HM(i,2) = REF_Z[i];
		HM(i,3) = only_angles ? 0 : P_ref[i];
	}
	HM(3,3)=1;

	const CPose3D  p= CPose3D(HM);
	out_ENU = p;
}

/*---------------------------------------------------------------
					coordinatesTransformation_WGS84_geocentric
 ---------------------------------------------------------------*/
void  mrpt::topography::coordinatesTransformation_WGS84_geocentric(
	double		in_longitude_degrees,
	double		in_latitude_degrees,
	double		in_height_meters,
	double		&out_x_meters,
	double		&out_y_meters,
	double		&out_z_meters )
{
	// --------------------------------------------------------------------
	// See: http://en.wikipedia.org/wiki/Reference_ellipsoid
	//  Constants are for WGS84
	// --------------------------------------------------------------------

	// The cos^2 of the angular eccentricity of the Earth:
	static const double	cos2_ae_earth = 0.993305619995739;

	// The sin^2 of the angular eccentricity of the Earth:
	static const double	sin2_ae_earth = 0.006694380004261;

	// Major axis of the Earth (meters)
	static const double	a = 6378137;

	double			lon  = DEG2RAD( in_longitude_degrees );
	double			lat  = DEG2RAD( in_latitude_degrees );

	// The radius of curvature in the prime vertical:
	// Using "square" here leads to VC6++ Internal compiler error!!!!!!!!!!!!!!!!!
	double			N = a / sqrt( 1 - sin2_ae_earth*square( sin(lat) ) );

	// Generate 3D point:
	out_x_meters = (N+in_height_meters)*cos(lat)*cos(lon);
	out_y_meters = (N+in_height_meters)*cos(lat)*sin(lon);
	out_z_meters = (cos2_ae_earth*N+in_height_meters)*sin(lat);
}


/*---------------------------------------------------------------
					LatLonToUTM
 ---------------------------------------------------------------*/
void  mrpt::topography::LatLonToUTM(
	double		la,
	double		lo,
	double    	&xx,
	double    	&yy,
	int    		&out_UTM_zone,
	char    	&out_UTM_latitude_band )
{
	// This method is based on public code by Gabriel Ruiz Martinez and Rafael Palacios.
	//  http://www.mathworks.com/matlabcentral/fileexchange/10915

	//const double sa = 6378137.000000;
	//const double sb = 6356752.314245;
	//const double e2 = 0.082094437950043;    // ( ( ( sa ^ 2 ) - ( sb ^ 2 ) ) ^ 0.5 ) / sb;

	const double e2cuadrada = 0.006739496742333; //  e2 ^ 2;
	const double c = 6399593.625758674;    // c = ( sa ^ 2 ) / sb;

	const double lat = DEG2RAD(la);
	const double lon = DEG2RAD(lo);

	const int Huso = mrpt::math::fix( ( lo / 6 ) + 31);
	double S = ( ( Huso * 6 ) - 183 );
	double deltaS = lon - DEG2RAD(S);

	char Letra;

	if (la<-72) Letra='C';
	else if (la<-64) Letra='D';
	else if (la<-56) Letra='E';
	else if (la<-48) Letra='F';
	else if (la<-40) Letra='G';
	else if (la<-32) Letra='H';
	else if (la<-24) Letra='J';
	else if (la<-16) Letra='K';
	else if (la<-8) Letra='L';
	else if (la<0) Letra='M';
	else if (la<8) Letra='N';
	else if (la<16) Letra='P';
	else if (la<24) Letra='Q';
	else if (la<32) Letra='R';
	else if (la<40) Letra='S';
	else if (la<48) Letra='T';
	else if (la<56) Letra='U';
	else if (la<64) Letra='V';
	else if (la<72) Letra='W';
	else Letra='X';

	const double a = cos(lat) * sin(deltaS);
	const double epsilon = 0.5 * log( ( 1 +  a) / ( 1 - a ) );
	const double nu = atan( tan(lat) / cos(deltaS) ) - lat;
	const double v = ( c / sqrt( ( 1 + ( e2cuadrada * square(cos(lat))  ) ) ) ) * 0.9996;
	const double ta = 0.5 * e2cuadrada * square(epsilon)  * square( cos(lat) );
	const double a1 = sin( 2 * lat );
	const double a2 = a1 * square( cos(lat) );
	const double j2 = lat + 0.5*a1;
	const double j4 = ( ( 3.0 * j2 ) + a2 ) / 4.0;
	const double j6 = ( ( 5.0 * j4 ) + ( a2 * square( cos(lat) ) ) ) / 3.0;
	const double alfa = 0.75  * e2cuadrada;
	const double beta = (5.0/3.0) * pow(alfa, 2.0);
	const double gama = (35.0/27.0) * pow(alfa,3.0);
	const double Bm = 0.9996 * c * ( lat - alfa * j2 + beta * j4 - gama * j6 );

	xx = epsilon * v * ( 1 + ( ta / 3.0 ) ) + 500000;
	yy = nu * v * ( 1 + ta ) + Bm;

	if (yy<0)
	   yy += 9999999;

	out_UTM_zone = Huso;
	out_UTM_latitude_band = Letra;
}
