/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.


#include <mrpt/opengl/CTexturedPlane.h>
#include <mrpt/math/utils.h>
#include "opengl_internals.h"
#include <mrpt/poses/CPose3D.h>
#include <mrpt/opengl/CSetOfTriangles.h>

using namespace mrpt;
using namespace mrpt::opengl;
using namespace mrpt::poses;
using namespace mrpt::utils;
using namespace mrpt::math;
using namespace std;

IMPLEMENTS_SERIALIZABLE( CTexturedPlane, CRenderizable, mrpt::opengl )

/*---------------------------------------------------------------
							CTexturedPlane
  ---------------------------------------------------------------*/
CTexturedPlane::CTexturedPlane(
	float				x_min,
	float				x_max,
	float				y_min,
	float				y_max
	) :
		m_glTextureName(0),
		m_texture_is_loaded(false),
		polygonUpToDate(false)
{
	// Copy data:
	m_xMin = x_min;
	m_xMax = x_max;
	m_yMin = y_min;
	m_yMax = y_max;

	m_enableTransparency = false;
}

/*---------------------------------------------------------------
							assignImage
  ---------------------------------------------------------------*/
void  CTexturedPlane::assignImage(
	const CImage& img,
	const CImage& imgAlpha )
{
	MRPT_START;

	unloadTexture();

	// Make a copy:
	m_textureImage = img;
	m_textureImageAlpha = imgAlpha;

	m_enableTransparency = true;

	MRPT_END;
}

/*---------------------------------------------------------------
							assignImage
  ---------------------------------------------------------------*/
void  CTexturedPlane::assignImage(
	const CImage& img )
{
	MRPT_START;

	unloadTexture();

	// Make a copy:
	m_textureImage = img;

	m_enableTransparency = false;

	MRPT_END;
}

/*---------------------------------------------------------------
							assignImage
  ---------------------------------------------------------------*/
void  CTexturedPlane::assignImage_fast(
	CImage& img,
	CImage& imgAlpha )
{
	MRPT_START;

	unloadTexture();

	// Make a copy:
	m_textureImage.copyFastFrom(img);
	m_textureImageAlpha.copyFastFrom(imgAlpha);

	m_enableTransparency = true;

	MRPT_END;
}

/*---------------------------------------------------------------
							assignImage
  ---------------------------------------------------------------*/
void  CTexturedPlane::assignImage_fast(
	CImage& img )
{
	MRPT_START;

	unloadTexture();

	// Make a copy:
	m_textureImage.copyFastFrom(img);

	m_enableTransparency = false;

	MRPT_END;
}


/*---------------------------------------------------------------
							loadTextureInOpenGL
  ---------------------------------------------------------------*/
void  CTexturedPlane::loadTextureInOpenGL() const
{
#if MRPT_HAS_OPENGL_GLUT
	unsigned char	*dataAligned=NULL;
	vector<unsigned char>		data;

	try
	{
		if (m_texture_is_loaded)
		{
			glBindTexture( GL_TEXTURE_2D, m_glTextureName );
			checkOpenGLError();
			return;
		}

		// Reserve the new one --------------------------

        // allocate texture names:
		m_glTextureName = getNewTextureNumber();

		// select our current texture
		glBindTexture( GL_TEXTURE_2D, m_glTextureName );
		checkOpenGLError();

/*		glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
		checkOpenGLError();
		glPixelStorei(GL_PACK_ALIGNMENT, 1);
		checkOpenGLError();
*/
		// select modulate to mix texture with color for shading
//		glTexEnvf( GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE );
//		checkOpenGLError();

		// when texture area is small, bilinear filter the closest mipmap
		glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR ); //_MIPMAP_NEAREST );
		checkOpenGLError();

		// when texture area is large, bilinear filter the first mipmap
		glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST); //GL_LINEAR );
	    checkOpenGLError();

		// if wrap is true, the texture wraps over at the edges (repeat)
		//       ... false, the texture ends at the edges (clamp)
		glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT );
		checkOpenGLError();

		glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T,GL_REPEAT );
		checkOpenGLError();

		// Assure that the images do not overpass the maximum dimensions allowed by OpenGL:
		// ------------------------------------------------------------------------------------
		GLint	texSize;
		glGetIntegerv(GL_MAX_TEXTURE_SIZE, &texSize);
		while ( m_textureImage.getHeight()>(unsigned int)texSize || m_textureImage.getWidth()>(unsigned int)texSize )
		{
			m_textureImage		= m_textureImage.scaleHalf();
			m_textureImageAlpha = m_textureImageAlpha.scaleHalf();
		}

		int		width = m_textureImage.getWidth();
		int		height = m_textureImage.getHeight();

		r_width = round2up( width );
		r_height = round2up( height );

		int		fill_x_left = (r_width - width)/2;
		int		fill_y_top = (r_height - height)/2;

		int		fill_x_right = (r_width - width) - fill_x_left;
		int		fill_y_bottom = (r_height - height) - fill_y_top;

		// Compute the exact texture coordinates:
		m_tex_x_min = ((float)fill_x_left) / r_width;
		m_tex_x_max = 1.0f-((float)fill_x_right) / r_width;
		m_tex_y_min = ((float)fill_y_top) / r_height;
		m_tex_y_max = 1.0f-((float)fill_y_bottom) / r_height;

        if (m_enableTransparency)
		{
			ASSERT_( !m_textureImageAlpha.isColor());
			ASSERT_(m_textureImageAlpha.getWidth() == m_textureImage.getWidth());
			ASSERT_(m_textureImageAlpha.getHeight() == m_textureImage.getHeight());
		}

		if (m_textureImage.isColor())
		{
			// Color texture:
			if (m_enableTransparency)
			{
				data.clear();
				data.resize( r_height*r_width*4 + 1000);
                dataAligned = ((unsigned char*)(((POINTER_TYPE)&data[0]) & (~((POINTER_TYPE)0x0F)) )) + 0x10;

				for (int y=0;y<height;y++)
				{
					unsigned char 	*ptrSrcCol = m_textureImage(0,y,2);
					unsigned char 	*ptrSrcAlfa = m_textureImageAlpha(0,y);
					unsigned char 	*ptr = dataAligned + (fill_x_left)*4 + (fill_y_top+y)*r_width*4;
					for (int x=0;x<width;x++)
					{
    					*ptr++ = *ptrSrcCol--;
    					*ptr++ = *ptrSrcCol--;
    					*ptr++ = *ptrSrcCol--;
    					*ptr++ = *ptrSrcAlfa++;
						ptrSrcCol+=6;
					}
				}

				// build our texture mipmaps
				gluBuild2DMipmaps( GL_TEXTURE_2D, 4, r_width, r_height,GL_RGBA, GL_UNSIGNED_BYTE, dataAligned );
//				glTexImage2D( GL_TEXTURE_2D, 0, 4, r_width, r_height,0,GL_RGBA, GL_UNSIGNED_BYTE, dataAligned );
				checkOpenGLError();
			} // End of color texture WITH trans.
			else
			{
				data.clear();
				data.resize( r_height*r_width*3+1000 );
                dataAligned = ((unsigned char*)(((POINTER_TYPE)&data[0]) & (~((POINTER_TYPE)0x0F)) )) + 0x10;

				for (int y=0;y<height;y++)
				{
					unsigned char 	*ptrSrcCol = m_textureImage(0,y,2);
					unsigned char 	*ptr = dataAligned + (fill_x_left)*3 + (fill_y_top+y)*r_width*3;
					for (int x=0;x<width;x++)
					{
    					*ptr++ = *ptrSrcCol--;
    					*ptr++ = *ptrSrcCol--;
    					*ptr++ = *ptrSrcCol--;
						ptrSrcCol+=6;
					}
				}

				// build our texture mipmaps
				gluBuild2DMipmaps( GL_TEXTURE_2D, 3, r_width, r_height,GL_RGB, GL_UNSIGNED_BYTE, dataAligned );
//				glTexImage2D( GL_TEXTURE_2D, 0, 3, r_width, r_height,0,GL_RGB, GL_UNSIGNED_BYTE, dataAligned );
				checkOpenGLError();
			} // End of color texture WITHOUT trans.
		}
		else
		{
			// Gray-scale texture:
			if (m_enableTransparency)
			{
				data.clear();
				data.resize( r_height*r_width*2+1000 );
                dataAligned = ((unsigned char*)(((POINTER_TYPE)&data[0]) & (~((POINTER_TYPE)0x0F)) )) + 0x10;

				for (int y=0;y<height;y++)
				{
					unsigned char 	*ptrSrcCol = m_textureImage(0,y);
					unsigned char 	*ptrSrcAlfa = m_textureImageAlpha(0,y);
					unsigned char 	*ptr = dataAligned + (fill_x_left)*2 + (fill_y_top+y)*r_width*2;
					for (int x=0;x<width;x++)
					{
    					*ptr++ = *ptrSrcCol++;
    					*ptr++ = *ptrSrcAlfa++;
					}
				}

				// build our texture mipmaps
				gluBuild2DMipmaps( GL_TEXTURE_2D, 2, r_width, r_height, GL_LUMINANCE_ALPHA, GL_UNSIGNED_BYTE, dataAligned );
//				glTexImage2D( GL_TEXTURE_2D, 0, 2, r_width, r_height,0,GL_LUMINANCE_ALPHA, GL_UNSIGNED_BYTE, dataAligned );
				checkOpenGLError();
			}// End of gray-scale texture WITH trans.
			else
			{
				data.clear();
				data.resize( r_height*r_width+1000 );
                dataAligned = ((unsigned char*)(((POINTER_TYPE)&data[0]) & (~((POINTER_TYPE)0x0F)) )) + 0x10;

				for (int y=0;y<height;y++)
				{
					unsigned char 	*ptrSrcCol = m_textureImage(0,y);
					unsigned char 	*ptr = dataAligned + (fill_x_left)*1 + (fill_y_top+y)*r_width*1;
					for (int x=0;x<width;x++)
					{
    					*ptr++ = *ptrSrcCol++;
					}
				}

				// build our texture mipmaps
				gluBuild2DMipmaps( GL_TEXTURE_2D, 1, r_width, r_height, GL_LUMINANCE, GL_UNSIGNED_BYTE, dataAligned );
//				glTexImage2D( GL_TEXTURE_2D, 0, 1, r_width, r_height,0,GL_LUMINANCE, GL_UNSIGNED_BYTE, dataAligned );
				checkOpenGLError();
			}// End of gray-scale texture WITHOUT trans.

		}

		m_texture_is_loaded = true;
	}
	catch(exception &e)
	{
		THROW_EXCEPTION(format("m_glTextureName=%i\n%s",m_glTextureName,e.what()));
	}
	catch(...)
	{
		THROW_EXCEPTION("Runtime error!");
	}
#endif
}

/*---------------------------------------------------------------
							~CTexturedPlane
  ---------------------------------------------------------------*/
CTexturedPlane::~CTexturedPlane()
{
	unloadTexture();
}

/*---------------------------------------------------------------
							unloadTexture
  ---------------------------------------------------------------*/
void CTexturedPlane::unloadTexture()
{
    if (m_texture_is_loaded)
    {
    	m_texture_is_loaded = false;
		releaseTextureName( m_glTextureName );
		m_glTextureName = 0;
	}
}

/*---------------------------------------------------------------
							render
  ---------------------------------------------------------------*/
void   CTexturedPlane::render() const
{
#if MRPT_HAS_OPENGL_GLUT
	try
	{
		glEnable(GL_TEXTURE_2D);
		checkOpenGLError();

		if (m_enableTransparency || m_color_A!=1.0)
		{
			glDisable(GL_DEPTH_TEST);
			glEnable(GL_BLEND);
			glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
		}
        else
        {
			glEnable(GL_DEPTH_TEST);
			glDisable(GL_BLEND);
        }

		// This will load and/or select our texture, only if "m_texture_is_loaded" is false
		loadTextureInOpenGL();

		glBegin(GL_QUADS);

		glTexCoord2d(m_tex_x_min,m_tex_y_min);
		glVertex3f( m_xMin, m_yMin,0 );

		glTexCoord2d(m_tex_x_max,m_tex_y_min);
		glVertex3f( m_xMax, m_yMin,0 );

		glTexCoord2d(m_tex_x_max,m_tex_y_max);
		glVertex3f( m_xMax, m_yMax,0 );

		glTexCoord2d(m_tex_x_min,m_tex_y_max);
		glVertex3f( m_xMin, m_yMax,0 );

		glEnd();
		checkOpenGLError();

        glDisable(GL_BLEND);
		checkOpenGLError();

        glBlendFunc(GL_ONE, GL_ZERO);

        glEnable(GL_DEPTH_TEST);
		checkOpenGLError();

		glDisable(GL_TEXTURE_2D);
		checkOpenGLError();

	}
	catch(exception &e)
	{
        THROW_EXCEPTION(format("m_glTextureName=%i\n%s",
			m_glTextureName,
			e.what()));
	}
	catch(...)
	{
		THROW_EXCEPTION("Runtime error!");
	}
#endif
}

/*---------------------------------------------------------------
   Implements the writing to a CStream capability of
     CSerializable objects
  ---------------------------------------------------------------*/
void  CTexturedPlane::writeToStream(CStream &out,int *version) const
{
	if (version)
		*version = 1;
	else
	{
		writeToStreamRender(out);

		out << m_xMin << m_xMax;
		out << m_yMin << m_yMax;

		out << m_enableTransparency; // Added in version #1
		out << m_textureImage;
		if (m_enableTransparency) out << m_textureImageAlpha;

	}
}

/*---------------------------------------------------------------
	Implements the reading from a CStream capability of
		CSerializable objects
  ---------------------------------------------------------------*/
void  CTexturedPlane::readFromStream(CStream &in,int version)
{

	switch(version)
	{
	case 0:
		{
			readFromStreamRender(in);
			in >> m_textureImage >> m_textureImageAlpha;
			in >> m_xMin >> m_xMax;
			in >> m_yMin >> m_yMax;

			assignImage( m_textureImage, m_textureImageAlpha );

		} break;
	case 1:
		{
			readFromStreamRender(in);

			in >> m_xMin >> m_xMax;
			in >> m_yMin >> m_yMax;

			in >> m_enableTransparency;

			in >> m_textureImage;

			if (m_enableTransparency)
			{
				in >> m_textureImageAlpha;
				assignImage( m_textureImage, m_textureImageAlpha );
			}
			else
			{
				assignImage( m_textureImage );
			}

		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};
}

bool CTexturedPlane::traceRay(const mrpt::poses::CPose3D &o,double &dist) const	{
	if (!polygonUpToDate) updatePoly();
	return math::traceRay(tmpPoly,o-CPose3D(m_x,m_y,m_z,DEG2RAD(m_yaw),DEG2RAD(m_pitch),DEG2RAD(m_roll)),dist);
}

void CTexturedPlane::updatePoly() const	{
	TPolygon3D poly(4);
	poly[0].x=poly[1].x=m_xMin;
	poly[2].x=poly[3].x=m_xMax;
	poly[0].y=poly[3].y=m_yMin;
	poly[1].y=poly[2].y=m_yMax;
	for (size_t i=0;i<4;i++) poly[i].z=0;
	tmpPoly.resize(1);
	tmpPoly[0]=poly;
	polygonUpToDate=true;
}
