/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef CMATRIX_H
#define CMATRIX_H

#include <mrpt/utils/CSerializable.h>
#include <mrpt/math/CMatrixTemplateNumeric.h>
#include <mrpt/math/CVectorTemplate.h>

namespace mrpt
{
	namespace poses
	{
		class MRPTDLLIMPEXP CPose2D;
		class MRPTDLLIMPEXP CPose3D;
		class MRPTDLLIMPEXP CPoint2D;
		class MRPTDLLIMPEXP CPoint3D;
	}

	namespace math
	{

		// This must be added to any CSerializable derived class:
		DEFINE_SERIALIZABLE_PRE( CMatrix )

		/**  This class is a "CSerializable" wrapper for "CMatrixFloat".
		 */
		class MRPTDLLIMPEXP CMatrix : public mrpt::utils::CSerializable, public CMatrixFloat
		{
			// This must be added to any CSerializable derived class:
			DEFINE_SERIALIZABLE( CMatrix )
		public:
			/** Constructor  */
			CMatrix() : CMatrixFloat(1,1)
			{ }

			/** Constructor  */
			CMatrix(size_t row, size_t col) : CMatrixFloat(row,col)
			{ }

			/** Copy constructor
			  */
			CMatrix( const CMatrixFloat &m ) : CMatrixFloat(m)
			{ }

			/** Copy constructor
			  */
			CMatrix( const CMatrixTemplateNumeric<double> &m ) : CMatrixFloat(0,0)
			{
				*this = m;
			}

			/** Constructor from a mrpt::poses::CPose2D, which generates a 3x1 matrix \f$ [x y \phi]^T \f$
			   */
			explicit CMatrix( const mrpt::poses::CPose2D &p) : CMatrixFloat(p) {}

			/** Constructor from a mrpt::poses::CPose6D, which generates a 6x1 matrix \f$ [x y z yaw pitch roll]^T \f$
			   */
			explicit CMatrix( const mrpt::poses::CPose3D &p) : CMatrixFloat(p) {}

			/** Constructor from a mrpt::poses::CPoint2D, which generates a 2x1 matrix \f$ [x y]^T \f$
			   */
			explicit CMatrix( const mrpt::poses::CPoint2D &p) : CMatrixFloat(p) {}

			/** Constructor from a mrpt::poses::CPoint3D, which generates a 3x1 matrix \f$ [x y z]^T \f$
			   */
			explicit CMatrix( const mrpt::poses::CPoint3D &p) : CMatrixFloat(p) {}

			/** Assignment operator for float matrixes
			*/
			CMatrix & operator = (const CMatrixFloat& m)
			{
				CMatrixFloat::operator =(m);
				return *this;
			}

			/** Assignment operator for double matrixes
			   */
			CMatrix &operator = (const CMatrixTemplateNumeric<double>& m)
			{
				setSize(m.getRowCount(),m.getColCount());

				for (size_t i=0;i<m.getRowCount();i++)
					for (size_t j=0;j<m.getColCount();j++)
						set_unsafe(i,j, static_cast<float>( m.get_unsafe(i,j) ) );
				return *this;
			}

		}; // end of class definition

	} // End of namespace
} // End of namespace

#endif
