#include <wx/filename.h>
#include <wx/log.h>
#include <wx/string.h>

#include <string>

#include "OutputList.h"
#include "Basic3DOutputter.h"
#include "SeriesHandler.h"
#include "HandlerFactory.h"
//#include "Basic3DConversion.h"
#include "BasicVolumeFormat.h"

using namespace std;
using namespace jcs;

vector<bool>
GetValuesToUse(set<VolId> SetOfVolIdVectors)
{
	set<VolId>::iterator viv = SetOfVolIdVectors.begin();
	set<VolId>::iterator viv_end = SetOfVolIdVectors.end();
 
	vector<set<string> > VectorOfVolIdValues;
	for (unsigned int i = 0; i < viv->ids.size(); ++i) {
		VectorOfVolIdValues.push_back(set<string> ());
	}

	while (viv != viv_end) {
		VolId::IdType::const_iterator vid = viv->ids.begin();
		vector<set<string> >::iterator insert_point = VectorOfVolIdValues.begin();
		while (vid != viv->ids.end()) {
			insert_point->insert(*vid);
			++vid;
			++insert_point;
		}
		++viv;
	}

	vector<bool> UseValue;
	vector<set<string> >::iterator vit = VectorOfVolIdValues.begin();
	while (vit != VectorOfVolIdValues.end()) {
		UseValue.push_back(vit->size() > 1);
		++vit;
	}

	return UseValue;
}

Basic3DOutputter::Basic3DOutputter(const Options& options):
	OutputterBase(options)
{
	dim_ = mOptions.intOptions["dim"];
	headerExtension = mOptions.stringOptions["header"];
	rawExtension = mOptions.stringOptions["raw"];
	skip_ = mOptions.intOptions["skip"];
	rescale = mOptions.boolOptions["rescale"];
}

Basic3DOutputter::~Basic3DOutputter()
{
	mOptions.intOptions["dim"] = dim_;
	mOptions.intOptions["skip"] = skip_;
	mOptions.stringOptions["header"] = headerExtension;
	mOptions.stringOptions["raw"] = rawExtension;
	mOptions.boolOptions["rescale"] = rescale;
}

Options
Basic3DOutputter::Get3DOptions()
{
	Options options = GetBaseOptions();
	options.intOptions["dim"] = 3;
	options.intOptions["skip"] = 0;
	options.stringOptions["header"] = _T("hdr");
	options.stringOptions["raw"] = _T("img");
	options.boolOptions["rescale"] = false;

	return options;

}

void
Basic3DOutputter::UpdateOutputForSeries(SeriesHandler* handler)
{
	string series_uid(handler->GetSeriesUid());
	RemoveSeries(series_uid);
	
	ImageFileName name;
	name.seriesUid = series_uid;
	name.SetPrefix(GenerateDefaultPrefix(handler));

	// fill in default dirs
	FillInDefaultDirs(name, handler);

	int dimensionality = GetDimensionality(series_uid);

	if (dimensionality == 4) {
		string output_file_uid = series_uid;
		name.SetExt(static_cast<const char*>(headerExtension.mb_str(wxConvLocal)));
		mOutputList.fileNameList.insert(make_pair(output_file_uid, name));
		if (rawExtension != _T("")) {
			name.SetExt(static_cast<const char*>(rawExtension.mb_str(wxConvLocal)));
			mOutputList.fileNameList.insert(make_pair(output_file_uid, name));
		}
	}

	else { // dimensionality == 3

		set<VolId> vol_ids = handler->GetVolIds();

		vector<bool> use_id;
		use_id = GetValuesToUse(vol_ids);		
		
		set<VolId>::iterator it = vol_ids.begin();

		int skip = GetSkip(series_uid);
		int skipVols = (static_cast<int>(vol_ids.size()) > skip) ? skip : 0;
		for (int i = 0; i < skipVols; ++i) ++it;

		while (it != vol_ids.end()) {
			string postfix;
			for (unsigned int i = 0; i < it->ids.size(); ++i) {
				if (use_id.at(i) && (it->ids.at(i) != "")) {
					postfix.append("_");
					postfix.append(it->ids.at(i));
				}
			}

			name.SetPostfix(postfix);

			string output_file_uid = series_uid + postfix;

			volKeyMap[*it] = output_file_uid;
			name.SetExt(static_cast<const char*>(headerExtension.mb_str(wxConvLocal)));
			mOutputList.fileNameList.insert(make_pair(output_file_uid, name));
			if (rawExtension != _T("")) {
				name.SetExt(static_cast<const char*>(rawExtension.mb_str(wxConvLocal)));
				mOutputList.fileNameList.insert(make_pair(output_file_uid, name));
			}

			++it;

		}
	}
	
	// if is moco series
	if (handler->IsMoCo()) {
		string series_uid(handler->GetSeriesUid());
		ImageFileName name;
		name.seriesUid = series_uid;
		name.SetPrefix(GenerateDefaultPrefix(handler));
		FillInDefaultDirs(name, handler);
		name.SetExt("txt");
		name.SetPostfix("_moco");
		mOutputList.fileNameList.insert(make_pair(series_uid+"_moco", name));
	}

}

void
Basic3DOutputter::RemoveSeries(const std::string& seriesUid)
{
	set<OutputList::ListType::key_type> keys;
	OutputList::ListType::iterator it = mOutputList.fileNameList.begin();
	while (it != mOutputList.fileNameList.end()) {
		if (it->second.seriesUid == seriesUid)
			keys.insert(it->first);
		++it;
	}

	set<OutputList::ListType::key_type>::iterator key_it = keys.begin();
	while (key_it != keys.end()) {
		mOutputList.fileNameList.erase(*key_it);
		++key_it;
	}
}

wxFileName
Basic3DOutputter::GetFileNameFromVolId(const VolId& vol_id)
{
	// first element in vol id must be series uid!
	if (GetDimensionality(vol_id.ids.front()) == 4) {
		return GetFileName(vol_id.ids.front());
	}


	string key = volKeyMap[vol_id];
	OutputList::ListType::iterator pos;
	pos = mOutputList.fileNameList.find(key);


	if (pos == mOutputList.fileNameList.end()) {
		wxLogError(_T("Error finding output file name"));
		for (unsigned int i = 0; i < vol_id.ids.size(); ++i) {
			wxLogError(_T("vol_id.ids[%d]: %s"), i, vol_id.ids[i].c_str());
		}

		wxLogError(_T("Key: %s\n"), key.c_str());

		OutputList::ListType::iterator test_it = mOutputList.fileNameList.begin();
		while (test_it != mOutputList.fileNameList.end() ){
			wxLogMessage(_T("Output list: %s\t%s"), test_it->first.c_str(), test_it->second.GetFullName().c_str());
			++test_it;
		}

		return wxFileName(_T("error"));
	}


	wxFileName name(mOutputList.rootDir + wxFileName::GetPathSeparator());
	wxFileName fname(wxString(pos->second.GetFullPath().c_str(), wxConvLocal));
	wxArrayString dirs = fname.GetDirs();
	for (unsigned int i = 0; i < dirs.size(); ++i) 
		name.AppendDir(dirs[i]);
	

	name.SetFullName(fname.GetFullName());

	assert(name.IsOk());


//	name.PrependDir(mOutputList.rootDir);

	// temp
/*	wxLogMessage(name.GetName());
	for (unsigned int i = 0; i < vol_id.size(); ++i)
		wxLogMessage(vol_id[i].c_str());
	wxLogMessage(_T("Key: %s"), key.c_str());
*/
	// end temp

	return name;

}


int
Basic3DOutputter::GetDimensionality(const string& series_uid) const
{
	map<string, Options>::const_iterator pos = overrides.find(series_uid);
	if (pos != overrides.end()) {
		map <string, int>::const_iterator find_dim = pos->second.intOptions.find("dim");
		if (find_dim != pos->second.intOptions.end())
			return find_dim->second;
	}

	return dim_;
}

void 
Basic3DOutputter::SetDimensionality(const std::string& series_uid, int dim)
{
	overrides[series_uid].intOptions["dim"] = dim;
}

int
Basic3DOutputter::GetSkip(const string& series_uid) const
{
	map<string, Options>::const_iterator pos = overrides.find(series_uid);
	if (pos != overrides.end()) {
		map <string, int>::const_iterator find_skip = pos->second.intOptions.find("skip");
		if (find_skip != pos->second.intOptions.end())
			return find_skip->second;
	}

	return skip_;
}

void 
Basic3DOutputter::SetSkip(const std::string& series_uid, int skip)
{
	overrides[series_uid].intOptions["skip"] = skip;
}


void
Basic3DOutputter::SetOption(const std::string& name, int value)
{
	OutputterBase::SetOption(name, value);
	if (name.find("skip") != string::npos) SetSkip(value);
	if (name.find("dim") != string::npos) SetDimensionality(value);
}

void
Basic3DOutputter::SetOption(const std::string& name, bool value)
{
	OutputterBase::SetOption(name, value);
}
