#include <wx/filename.h>
#include <wx/config.h>

#include <iostream>
#include <string>

#include "SeriesHandler.h"
#include "OutputterBase.h"

using namespace jcs;

/** Retrieves all Options values from the user's configuration file.
 */
void
Options::ReadConfig()
{
#ifndef NO_CONFIG
    wxString oldPath = wxConfig::Get()->GetPath();
    wxConfig::Get()->SetPath(wxString("/", wxConvLocal));
    wxConfig::Get()->SetPath(wxString(pathname.c_str(), wxConvLocal));

    map<string, bool>::iterator bool_it;
    for (bool_it = boolOptions.begin(); bool_it != boolOptions.end(); ++bool_it) 
	wxConfig::Get()->Read(wxString(bool_it->first.c_str(), wxConvLocal), 
			      &bool_it->second, bool_it->second);

    map<string, int>::iterator int_it;
    for (int_it = intOptions.begin(); int_it != intOptions.end(); ++int_it) 
	wxConfig::Get()->Read(wxString(int_it->first.c_str(), wxConvLocal),
			      &int_it->second, int_it->second);

    map<string, wxString>::iterator string_it;
    for (string_it = stringOptions.begin(); string_it != stringOptions.end(); ++string_it) 
	wxConfig::Get()->Read(wxString(string_it->first.c_str(), wxConvLocal), 
			      &string_it->second, string_it->second);

    map<string, double>::iterator double_it;
    for (double_it = doubleOptions.begin(); double_it != doubleOptions.end(); ++double_it) 
	wxConfig::Get()->Read(wxString(double_it->first.c_str(), wxConvLocal), 
			      &double_it->second, double_it->second);

    wxConfig::Get()->SetPath(oldPath);
#endif
}

/** Stores all Options values to the user's configuration file.
 */
void
Options::WriteConfig()
{
#ifndef NO_CONFIG
    wxString oldPath = wxConfig::Get()->GetPath();
    wxConfig::Get()->SetPath(wxString("/", wxConvLocal));
    wxConfig::Get()->SetPath(wxString(pathname.c_str(), wxConvLocal));

    map<string, bool>::iterator bool_it;
    for (bool_it = boolOptions.begin(); bool_it != boolOptions.end(); ++bool_it) 
	wxConfig::Get()->Write(wxString(bool_it->first.c_str(), wxConvLocal), bool_it->second);

    map<string, int>::iterator int_it;
    for (int_it = intOptions.begin(); int_it != intOptions.end(); ++int_it) 
	wxConfig::Get()->Write(wxString(int_it->first.c_str(), wxConvLocal), int_it->second);

    map<string, wxString>::iterator string_it;
    for (string_it = stringOptions.begin(); string_it != stringOptions.end(); ++string_it) 
	wxConfig::Get()->Write(wxString(string_it->first.c_str(), wxConvLocal), string_it->second);

    map<string, double>::iterator double_it;
    for (double_it = doubleOptions.begin(); double_it != doubleOptions.end(); ++double_it) 
	wxConfig::Get()->Write(wxString(double_it->first.c_str(), wxConvLocal), double_it->second);

    wxConfig::Get()->SetPath(oldPath);

#endif
}

/** Loads option values from the user's configuration file, if found.
    Populates the 'defaultNameFields' array.
    \param options A reference to an Options object.
*/
OutputterBase::OutputterBase(const Options& options) : mOptions(options)
{
    mOptions.ReadConfig();

    dfn[PatientName] = "PatientName";
    dfn[PatientId] = "PatientId";
    dfn[SeriesDate] = "SeriesDate";
    dfn[SeriesTime] = "SeriesTime";
    dfn[StudyId] = "StudyId";
    dfn[StudyDescription] = "StudyDescription";
    dfn[SeriesNumber] = "SeriesNumber";
    dfn[SequenceName] = "SequenceName";
    dfn[ProtocolName] = "ProtocolName";
    dfn[SeriesDescription] = "SeriesDescription";
    
    defaultNameFields[PatientName] = NameField(dfn[PatientName], mOptions.boolOptions[dfn[PatientName]]);
    defaultNameFields[PatientId] = NameField(dfn[PatientId], mOptions.boolOptions[dfn[PatientId]]);
    defaultNameFields[SeriesDate] = NameField(dfn[SeriesDate], mOptions.boolOptions[dfn[SeriesDate]]);
    defaultNameFields[SeriesTime] = NameField(dfn[SeriesTime], mOptions.boolOptions[dfn[SeriesTime]]);
    defaultNameFields[StudyId] = NameField(dfn[StudyId], mOptions.boolOptions[dfn[StudyId]]);
    defaultNameFields[StudyDescription] = NameField(dfn[StudyDescription], mOptions.boolOptions[dfn[StudyDescription]]);
    defaultNameFields[SeriesNumber] = NameField(dfn[SeriesNumber], mOptions.boolOptions[dfn[SeriesNumber]]);
    defaultNameFields[SequenceName] = NameField(dfn[SequenceName], mOptions.boolOptions[dfn[SequenceName]]);
    defaultNameFields[ProtocolName] = NameField(dfn[ProtocolName], mOptions.boolOptions[dfn[ProtocolName]]);
    defaultNameFields[SeriesDescription] = NameField(dfn[SeriesDescription], mOptions.boolOptions[dfn[SeriesDescription]]);
}


/// The Destructor
/** Gathers option values and writes a configuration file for the user.
 */
OutputterBase::~OutputterBase()
{
    mOptions.boolOptions[dfn[PatientName]] = defaultNameFields[PatientName].value;
    mOptions.boolOptions[dfn[PatientId]] = defaultNameFields[PatientId].value;
    mOptions.boolOptions[dfn[SeriesDate]] = defaultNameFields[SeriesDate].value;
    mOptions.boolOptions[dfn[SeriesTime]] = defaultNameFields[SeriesTime].value;
    mOptions.boolOptions[dfn[StudyId]] = defaultNameFields[StudyId].value;
    mOptions.boolOptions[dfn[StudyDescription]] = defaultNameFields[StudyDescription].value;
    mOptions.boolOptions[dfn[SeriesNumber]] = defaultNameFields[SeriesNumber].value;
    mOptions.boolOptions[dfn[SequenceName]] = defaultNameFields[SequenceName].value;
    mOptions.boolOptions[dfn[ProtocolName]] = defaultNameFields[ProtocolName].value;
    mOptions.boolOptions[dfn[SeriesDescription]] = defaultNameFields[SeriesDescription].value;

    mOptions.WriteConfig();
}


///
/** Create and initialize an Options object with program default values for output operations.
    \return an Options object initialized to program defaults.
*/
Options
OutputterBase::GetBaseOptions()
{
    Options options;

    // Flag whether to save in separate directories.
    options.boolOptions["split_dir"] = true;
    options.boolOptions["split_subj"] = true;

    // Options pertaining to output file naming.
    options.boolOptions["PatientName"] = true;
    options.boolOptions["PatientId"] = false;
    options.boolOptions["SeriesDate"] = true;
    options.boolOptions["SeriesTime"] = false;
    options.boolOptions["StudyId"] = true;
    options.boolOptions["StudyDescription"] = false;
    options.boolOptions["SeriesNumber"] = true;
    options.boolOptions["SequenceName"] = false;
    options.boolOptions["ProtocolName"] = true;
    options.boolOptions["SeriesDescription"] = true;

    return options;
}


///
/** Generate and return a filename prefix according to program rules
    and user selection.
    \param series A pointer to a SeriesHandler
    \return A string to be used as a prefix for output files
*/
string
OutputterBase::GenerateDefaultPrefix(SeriesHandler* series)
{
    std::string prefix;
    bool first = true;
    std::string pn = "";

    FieldMap::iterator it = defaultNameFields.begin();
    FieldMap::iterator it_end = defaultNameFields.end();
    for (; it != it_end; it++) {
	if (it->second.value) {
	    std::string str;
	    if (series->Find(it->second.name, str)) {
		// If using protocol name AND series description, don't use
		// both if they are the same
		if (it->second.name == dfn[ProtocolName]) {
		    pn = str;
		}
		else {
		    if (str == pn)
			continue;
		}
      		// Separate filename elements with an underscore character.
		if (!first)
		    prefix.append("_");
		// Any numbers should be padded out to at least 3 digits
		if (itos(stoi(str)) == str)
		    str = itos(stoi(str), 3);
		prefix.append(str);
		first = false;
	    }
	}
    }
    return RemoveInvalidChars(prefix);
}


///
/** Fills in default dirs
    \param name
    \param series
*/
void
OutputterBase::FillInDefaultDirs(ImageFileName& name, SeriesHandler* series)
{
    name.ResetPath();

    if (GetSplitSubj()) {
	string subject_dir = series->seriesInfo.subject_name;
	if (subject_dir.empty())
	    subject_dir = series->seriesInfo.subject_id;
	subject_dir = RemoveInvalidChars(subject_dir);
	if (subject_dir.empty())
	    subject_dir = "unknown";

	name.AppendDir(subject_dir, subject_dir);
    }

    if (GetSplit()) {
	string series_dir = series->seriesInfo.study_number
	    + "_" + series->seriesInfo.series_number;

	if (!series->seriesInfo.series_description.empty()) {
	    series_dir.append("_");
	    series_dir.append(series->seriesInfo.series_description);
	}

	if (!series->seriesInfo.SeriesDate.empty()) {
	    series_dir.append("_");
	    series_dir.append(series->seriesInfo.SeriesDate);
	}
	series_dir = RemoveInvalidChars(series_dir);
	name.AppendDir(series->GetSeriesUid(), series_dir);
    }
}


///
/** Gets the Image file name.
    \param series_uid
    \param name Name of file to match.
    \return Existing ImageFileName object if found in mOutputList, otherwise a new object.
*/
ImageFileName
OutputterBase::GetImageFileName(const string& series_uid, const string& name)
{
    ImageFileName retval = ImageFileName();
    OutputList::ListType::iterator pos = mOutputList.fileNameList.lower_bound(series_uid);
    OutputList::ListType::iterator end_pos = mOutputList.fileNameList.end();
    for (;pos != end_pos; pos++)
	if (pos->second.GetFullName() == name) {
	    retval = pos->second;
	    break;
	}
    return retval;
}


///
/** Gets the file name for the SeriesUid.
    \param series_uid SeriesUid to act upon.
    \return A wxFileName
*/
wxFileName
OutputterBase::GetFileName(const string& series_uid)
{
    wxFileName name = wxFileName();

    OutputList::ListType::iterator pos = mOutputList.fileNameList.lower_bound(series_uid);
    if (pos != mOutputList.fileNameList.end()) {

	name = wxFileName(mOutputList.rootDir + wxFileName::GetPathSeparator());
	wxFileName fname(wxString(pos->second.GetFullPath().c_str(), wxConvLocal));
	wxArrayString dirs = fname.GetDirs();
	for (unsigned int i = 0; i < dirs.size(); ++i) 
	    name.AppendDir(dirs[i]);  
	name.SetFullName(fname.GetFullName());
    }
    return name;
}


///
/** Changes the file name of the given SeriesUid
    \param series_uid The SeriesUid to act upon.
    \param new_name The new name to bestow.
*/
void
OutputterBase::ChangeFileName(const string& series_uid, const string& new_name)
{
    OutputList::ListType::iterator pos;
    pos = mOutputList.fileNameList.lower_bound(series_uid);
    while (pos != mOutputList.fileNameList.end() && !pos->first.compare(0, series_uid.size(), series_uid)) {
	if (pos->second.isRenameable)
	    pos->second.SetPrefix(new_name);
	++pos;
    }
}

/** Changes the directory for a vector of SeriesUids
    \param series_uids The vector of SeriesUids to change.
    \param new_name The new directory name
    \param position 
*/
void
OutputterBase::ChangeDirName(const vector<string>& series_uids, 
			     const string& new_name, int position)
{
    vector<string>::const_iterator it;
    for (it = series_uids.begin(); it != series_uids.end(); ++it) {
	OutputList::ListType::iterator pos;
	pos = mOutputList.fileNameList.lower_bound(*it);
	while (pos != mOutputList.fileNameList.end() && !pos->first.compare(0, it->size(), *it)) {
	    pos->second.SetDirName(position, new_name);
	    ++pos;
	} 
    }
}


/** Sets a boolean option.
    \param name Name of the option.
    \param value Value of the option.
*/
void
OutputterBase::SetOption(const string& name, bool value)
{
    if (name.find("split_dir") != string::npos) {
	SetSplit(value);
    }

    if (name.find("split_subj") != string::npos) {
	SetSplitSubj(value);
    }

}

/// Seeks an integer override.
/** Discovers whether there is an 'overrides' entry for 'series_uid'. If so, sets 'value'
    to its value and returns true, otherwise returns false.
    \param series_uid String representing a SeriesUid.
    \param seek String representing an Option value.
    \param value Option value is returned in this parameter.
    \return True if an override is found, false otherwise.
*/
bool
OutputterBase::FindIntInOverrides(const string& series_uid, const string& seek, int& value)
{
    bool retval = false;
    map<string, Options>::const_iterator pos = overrides.find(series_uid);
    if (pos != overrides.end()) {
	map <string, int>::const_iterator find_seek = pos->second.intOptions.find(seek);
	if (find_seek != pos->second.intOptions.end()) {
	    value = find_seek->second;
	    retval = true;
	}
    }
    return retval;
}

/// Seeks a boolean override.
/** Discovers whether there is an 'overrides' entry for 'series_uid'. If so, sets 'value'
    to its value and returns true, otherwise returns false.
    \param series_uid String representing a SeriesUid.
    \param seek String representing an Option value.
    \param value Option value is returned in this parameter.
    \return True if an override is found, false otherwise.
*/
bool
OutputterBase::FindBoolInOverrides(const string& series_uid, const string& seek, bool& value)
{
    bool retval = false;
    map<string, Options>::const_iterator pos = overrides.find(series_uid);
    if (pos != overrides.end()) {
	map <string, bool>::const_iterator find_seek = pos->second.boolOptions.find(seek);
	if (find_seek != pos->second.boolOptions.end()) {
	    value = find_seek->second;
	    retval = true;
	}
    }
    return retval;
}
