// Copyright 2010-2012, Google Inc.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
//     * Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above
// copyright notice, this list of conditions and the following disclaimer
// in the documentation and/or other materials provided with the
// distribution.
//     * Neither the name of Google Inc. nor the names of its
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

// Qt component of configure dialog for Mozc
#include "gui/config_dialog/config_dialog.h"

#ifdef OS_WINDOWS
#include <QtGui/QWindowsStyle>
#include <windows.h>
#endif

#include <algorithm>
#include <sstream>
#include <stdlib.h>
#include <QtGui/QMessageBox>
#include "base/config_file_stream.h"
#include "base/const.h"
#ifdef OS_MACOSX
#include "base/mac_util.h"
#endif  // OS_MACOSX
#include "base/util.h"
#include "base/run_level.h"
#include "config/config_handler.h"
#include "config/config.pb.h"
#include "config/stats_config_util.h"
#ifdef ENABLE_CLOUD_SYNC
#include "gui/config_dialog/auth_dialog.h"
#endif  // ENABLE_CLOUD_SYNC
#include "gui/config_dialog/keymap_editor.h"
#include "gui/config_dialog/roman_table_editor.h"
#include "gui/base/win_util.h"
#include "ipc/ipc.h"
#include "session/commands.pb.h"
#include "session/internal/keymap.h"
#include "client/client.h"

namespace mozc {

namespace gui {

using mozc::config::StatsConfigUtil;

ConfigDialog::ConfigDialog()
    : client_(client::ClientFactory::NewClient()),
      initial_preedit_method_(0),
      initial_use_keyboard_to_change_preedit_method_(false) {
  setupUi(this);
  setWindowFlags(Qt::WindowSystemMenuHint);
  setWindowModality(Qt::NonModal);

#ifdef OS_WINDOWS
  miscStartupWidget->setVisible(false);
#endif  // OS_WINDOWS

#ifdef OS_MACOSX
  miscDefaultIMEWidget->setVisible(false);
  miscAdministrationWidget->setVisible(false);
#endif  // OS_MACOSX

#if defined(OS_LINUX)
  miscDefaultIMEWidget->setVisible(false);
  miscAdministrationWidget->setVisible(false);
  miscStartupWidget->setVisible(false);
#endif  // OS_LINUX

#ifdef NO_LOGGING
  // disable logging options
  miscLoggingWidget->setVisible(false);

#if defined(OS_LINUX)
  // The last "misc" tab has no valid configs on Linux
  const int kMiscTabIndex = 6;
  configDialogTabWidget->removeTab(kMiscTabIndex);
#endif  // OS_LINUX
#endif  // NO_LOGGING

#ifndef ENABLE_CLOUD_SYNC
  syncDescriptionLabel->setVisible(false);
  clearSyncButton->setVisible(false);
  removeAllDescriptionLabel->setVisible(false);
  syncStatusLabel->setVisible(false);
  syncButtonsLayoutWidget->setVisible(false);
  syncHeaderLayoutWidget->setVisible(false);
#endif  // !ENABLE_CLOUD_SYNC

#ifndef ENABLE_CLOUD_HANDWRITING
  cloudHandwritingCheckBox->setVisible(false);
  cloudServersLayoutWidget->setVisible(false);
#endif  // !ENABLE_CLOUD_HANDWRITING

#if !defined(ENABLE_CLOUD_SYNC) && !defined(ENABLE_CLOUD_HANDWRITING)
  // Hide "Cloud" tab when all the cloud features are not available.
  const int kCloudTabIndex = 5;
  configDialogTabWidget->removeTab(kCloudTabIndex);
#endif  // !ENABLE_CLOUD_SYNC && !ENABLE_CLOUD_HANDWRITING

  suggestionsSizeSpinBox->setRange(1, 9);

  // punctuationsSettingComboBox->addItem(QString::fromUtf8("、。"));
  // punctuationsSettingComboBox->addItem(QString::fromUtf8("，．"));
  // punctuationsSettingComboBox->addItem(QString::fromUtf8("、．"));
  // punctuationsSettingComboBox->addItem(QString::fromUtf8("，。"));
  punctuationsSettingComboBox->addItem
      (QString::fromUtf8("\xE3\x80\x81\xE3\x80\x82"));
  punctuationsSettingComboBox->addItem
      (QString::fromUtf8("\xEF\xBC\x8C\xEF\xBC\x8E"));
  punctuationsSettingComboBox->addItem
      (QString::fromUtf8("\xE3\x80\x81\xEF\xBC\x8E"));
  punctuationsSettingComboBox->addItem
      (QString::fromUtf8("\xEF\xBC\x8C\xE3\x80\x82"));

  // symbolsSettingComboBox->addItem(QString::fromUtf8("「」・"));
  // symbolsSettingComboBox->addItem(QString::fromUtf8("[]／"));
  // symbolsSettingComboBox->addItem(QString::fromUtf8("「」／"));
  // symbolsSettingComboBox->addItem(QString::fromUtf8("[]・"));
  symbolsSettingComboBox->addItem(
      QString::fromUtf8("\xE3\x80\x8C\xE3\x80\x8D\xE3\x83\xBB"));
  symbolsSettingComboBox->addItem(
      QString::fromUtf8("[]\xEF\xBC\x8F"));
  symbolsSettingComboBox->addItem(
      QString::fromUtf8("\xE3\x80\x8C\xE3\x80\x8D\xEF\xBC\x8F"));
  symbolsSettingComboBox->addItem(
      QString::fromUtf8("[]\xE3\x83\xBB"));

  keymapSettingComboBox->addItem(tr("Custom keymap"));
  keymapSettingComboBox->addItem(tr("ATOK"));
  keymapSettingComboBox->addItem(tr("MS-IME"));
  keymapSettingComboBox->addItem(tr("Kotoeri"));

  keymapname_sessionkeymap_map_[tr("ATOK")] = config::Config::ATOK;
  keymapname_sessionkeymap_map_[tr("MS-IME")] = config::Config::MSIME;
  keymapname_sessionkeymap_map_[tr("Kotoeri")] = config::Config::KOTOERI;

  inputModeComboBox->addItem(tr("Romaji"));
  inputModeComboBox->addItem(tr("Kana"));
#ifdef OS_WINDOWS
  // These options changing the preedit method by a hot key are only
  // supported by Windows.
  inputModeComboBox->addItem(tr("Romaji (switchable)"));
  inputModeComboBox->addItem(tr("Kana (switchable)"));
#endif  // OS_WINDOWS

  spaceCharacterFormComboBox->addItem(tr("Follow input mode"));
  spaceCharacterFormComboBox->addItem(tr("Fullwidth"));
  spaceCharacterFormComboBox->addItem(tr("Halfwidth"));

  selectionShortcutModeComboBox->addItem(tr("No shortcut"));
  selectionShortcutModeComboBox->addItem(tr("1 -- 9"));
  selectionShortcutModeComboBox->addItem(tr("A -- L"));

  historyLearningLevelComboBox->addItem(tr("Yes"));
  historyLearningLevelComboBox->addItem(tr("Yes (don't record new data)"));
  historyLearningLevelComboBox->addItem(tr("No"));

  shiftKeyModeSwitchComboBox->addItem(tr("Off"));
  shiftKeyModeSwitchComboBox->addItem(tr("Alphanumeric"));
  shiftKeyModeSwitchComboBox->addItem(tr("Katakana"));

  numpadCharacterFormComboBox->addItem(tr("Follow input mode"));
  numpadCharacterFormComboBox->addItem(tr("Fullwidth"));
  numpadCharacterFormComboBox->addItem(tr("Halfwidth"));
  numpadCharacterFormComboBox->addItem(tr("Direct input"));

  verboseLevelComboBox->addItem(tr("0"));
  verboseLevelComboBox->addItem(tr("1"));
  verboseLevelComboBox->addItem(tr("2"));

  yenSignComboBox->addItem(tr("Yen Sign \xC2\xA5"));
  yenSignComboBox->addItem(tr("Backslash \\"));

#ifndef OS_MACOSX
  // On Windows/Linux, yenSignCombBox can be hidden.
  yenSignLabel->hide();
  yenSignComboBox->hide();
  // On Windows/Linux, useJapaneseLayout checkbox should be invisible.
  useJapaneseLayout->hide();
#endif  // !OS_MACOSX

  // signal/slot
  QObject::connect(configDialogButtonBox,
                   SIGNAL(clicked(QAbstractButton *)),
                   this,
                   SLOT(clicked(QAbstractButton *)));
  QObject::connect(clearUserHistoryButton,
                   SIGNAL(clicked()),
                   this,
                   SLOT(ClearUserHistory()));
  QObject::connect(clearUserPredictionButton,
                   SIGNAL(clicked()),
                   this,
                   SLOT(ClearUserPrediction()));
  QObject::connect(clearUnusedUserPredictionButton,
                   SIGNAL(clicked()),
                   this,
                   SLOT(ClearUnusedUserPrediction()));
  QObject::connect(editUserDictionaryButton,
                   SIGNAL(clicked()),
                   this,
                   SLOT(EditUserDictionary()));
  QObject::connect(editKeymapButton,
                   SIGNAL(clicked()),
                   this,
                   SLOT(EditKeymap()));
  QObject::connect(resetToDefaultsButton,
                   SIGNAL(clicked()),
                   this,
                   SLOT(ResetToDefaults()));
  QObject::connect(editRomanTableButton,
                   SIGNAL(clicked()),
                   this,
                   SLOT(EditRomanTable()));
  QObject::connect(inputModeComboBox,
                   SIGNAL(currentIndexChanged(int)),
                   this,
                   SLOT(SelectInputModeSetting(int)));
  QObject::connect(useAutoConversion,
                   SIGNAL(stateChanged(int)),
                   this,
                   SLOT(SelectAutoConversionSetting(int)));
  QObject::connect(historySuggestCheckBox,
                   SIGNAL(stateChanged(int)),
                   this,
                   SLOT(SelectSuggestionSetting(int)));
  QObject::connect(dictionarySuggestCheckBox,
                   SIGNAL(stateChanged(int)),
                   this,
                   SLOT(SelectSuggestionSetting(int)));
  QObject::connect(realtimeConversionCheckBox,
                   SIGNAL(stateChanged(int)),
                   this,
                   SLOT(SelectSuggestionSetting(int)));
  QObject::connect(launchAdministrationDialogButton,
                   SIGNAL(clicked()),
                   this,
                   SLOT(LaunchAdministrationDialog()));
  QObject::connect(launchAdministrationDialogButtonForUsageStats,
                   SIGNAL(clicked()),
                   this,
                   SLOT(LaunchAdministrationDialog()));

#ifdef ENABLE_CLOUD_SYNC
  QObject::connect(syncToggleButton,
                   SIGNAL(clicked()),
                   this,
                   SLOT(SyncToggleButtonClicked()));
  QObject::connect(syncCustomizationButton,
                   SIGNAL(clicked()),
                   this,
                   SLOT(LaunchSyncCustomizationDialog()));
  QObject::connect(clearSyncButton,
                   SIGNAL(clicked()),
                   this,
                   SLOT(ClearSyncClicked()));
#endif  // ENABLE_CLOUD_SYNC

  // When clicking these messages, CheckBoxs corresponding
  // to them should be toggled.
  // We cannot use connect/slot as QLabel doesn't define
  // clicked slot by default.
  usageStatsMessage->installEventFilter(this);
  incognitoModeMessage->installEventFilter(this);

#ifndef OS_WINDOWS
  checkDefaultCheckBox->setVisible(false);
  checkDefaultLine->setVisible(false);
  checkDefaultLabel->setVisible(false);
#endif   // !OS_WINDOWS

#ifdef OS_WINDOWS
  launchAdministrationDialogButton->setEnabled(true);
  // if the current application is not elevated by UAC,
  // add a shield icon
  if (mozc::Util::IsVistaOrLater()) {
    if (!mozc::RunLevel::IsElevatedByUAC()) {
      QWindowsStyle style;
      QIcon vista_icon(style.standardIcon(QStyle::SP_VistaShield));
      launchAdministrationDialogButton->setIcon(vista_icon);
      launchAdministrationDialogButtonForUsageStats->setIcon(vista_icon);
    }
  } else {
    dictionaryPreloadingAndUACLabel->setText(
        tr("Dictionary preloading"));
  }

  usageStatsCheckBox->setDisabled(true);
  usageStatsCheckBox->setVisible(false);
  usageStatsMessage->setDisabled(true);
  usageStatsMessage->setVisible(false);
#else
  launchAdministrationDialogButton->setEnabled(false);
  launchAdministrationDialogButton->setVisible(false);
  launchAdministrationDialogButtonForUsageStats->setEnabled(false);
  launchAdministrationDialogButtonForUsageStats->setVisible(false);
  administrationLine->setVisible(false);
  administrationLabel->setVisible(false);
  dictionaryPreloadingAndUACLabel->setVisible(false);
#endif  // OS_WINDOWS

#ifdef OS_LINUX
  // On Linux, disable all fields for UsageStats
  usageStatsLabel->setEnabled(false);
  usageStatsLabel->setVisible(false);
  usageStatsLine->setEnabled(false);
  usageStatsLine->setVisible(false);
  usageStatsMessage->setEnabled(false);
  usageStatsMessage->setVisible(false);
  usageStatsCheckBox->setEnabled(false);
  usageStatsCheckBox->setVisible(false);
#endif // OS_LINUX

  webUsageDictionaryCheckBox->setVisible(false);
  editWebServiceEntryButton->setVisible(false);

#ifdef ENABLE_CLOUD_SYNC
  sync_customize_dialog_.reset(new SyncCustomizeDialog(this));
  sync_running_ = false;
  // We do not wait with NamedEventListener to update the sync status,
  // but just do polling every second with QTimer.  If we introduce
  // NamedEventListener in a watcher thread, the main thread has to
  // cancel the listener when the window is closed.  That case would
  // cause another tricky threading issue.
  timer_.reset(new QTimer(this));
  QObject::connect(
      timer_.get(), SIGNAL(timeout()), this, SLOT(UpdateSyncStatus()));
  timer_->start(1000);
  last_synced_timestamp_ = 0;
  // all sync buttons are not enabled for the first time.
  clearSyncButton->setEnabled(false);
  syncCustomizationButton->setEnabled(false);
  syncToggleButton->setEnabled(false);
#endif  // ENABLE_CLOUD_SYNC

  Reload();

  editWebServiceEntryButton->setEnabled(
      webUsageDictionaryCheckBox->isChecked());

#ifdef OS_WINDOWS
  IMEHotKeyDisabledCheckBox->setChecked(WinUtil::GetIMEHotKeyDisabled());
#else
  IMEHotKeyDisabledCheckBox->setVisible(false);
#endif

#ifdef CHANNEL_DEV
  usageStatsCheckBox->setEnabled(false);
#endif  // CHANNEL_DEV
}

ConfigDialog::~ConfigDialog() {
#ifdef ENABLE_CLOUD_SYNC
  timer_->stop();
#endif  // ENABLE_CLOUD_SYNC
}

bool ConfigDialog::SetConfig(const config::Config &config) {
  if (!client_->CheckVersionOrRestartServer()) {
    LOG(ERROR) << "CheckVersionOrRestartServer failed";
    return false;
  }

  if (!client_->SetConfig(config)) {
    LOG(ERROR) << "SetConfig failed";
    return false;
  }

  return true;
}

bool ConfigDialog::GetConfig(config::Config *config) {
  if (!client_->CheckVersionOrRestartServer()) {
    LOG(ERROR) << "CheckVersionOrRestartServer failed";
    return false;
  }

  if (!client_->GetConfig(config)) {
    LOG(ERROR) << "GetConfig failed";
    return false;
  }

  return true;
}

void ConfigDialog::Reload() {
  config::Config config;
  if (!GetConfig(&config)) {
    QMessageBox::critical(this,
                          tr("Mozc settings"),
                          tr("Failed to get current config values"));
  }
  ConvertFromProto(config);

  SelectAutoConversionSetting(static_cast<int>(config.use_auto_conversion()));

  initial_preedit_method_ = static_cast<int>(config.preedit_method());
  initial_use_keyboard_to_change_preedit_method_ =
      config.use_keyboard_to_change_preedit_method();
}

bool ConfigDialog::Update() {
  config::Config config;
  ConvertToProto(&config);

  if (config.session_keymap() == config::Config::CUSTOM &&
      config.custom_keymap_table().empty()) {
    QMessageBox::warning(
        this,
        tr("Mozc settings"),
        tr("The current custom keymap table is empty. "
           "When custom keymap is selected, "
           "you must customize it."));
    return false;
  }


#if defined(OS_WINDOWS) || defined(OS_LINUX)
  if (initial_preedit_method_ !=
      static_cast<int>(config.preedit_method()) ||
      initial_use_keyboard_to_change_preedit_method_ !=
      static_cast<int>(config.use_keyboard_to_change_preedit_method())) {
    QMessageBox::information(this,
                             tr("Mozc settings"),
                             tr("Romaji/Kana setting is enabled from"
                                " new applications."));
  }
#endif

  if (!SetConfig(config)) {
    QMessageBox::critical(this,
                          tr("Mozc settings"),
                          tr("Failed to update config"));
  }

#ifdef OS_WINDOWS
  if (!WinUtil::SetIMEHotKeyDisabled(IMEHotKeyDisabledCheckBox->isChecked())) {
    // Do not show any dialog here, since this operation will not fail
    // in almost all cases.
    // TODO(taku): better to show dialog?
    LOG(ERROR) << "Failed to update IME HotKey status";
  }
#endif

#ifdef OS_MACOSX
  if (startupCheckBox->isChecked()) {
    if (!MacUtil::CheckPrelauncherLoginItemStatus()) {
      MacUtil::AddPrelauncherLoginItem();
    }
  } else {
    if (MacUtil::CheckPrelauncherLoginItemStatus()) {
      MacUtil::RemovePrelauncherLoginItem();
    }
  }
#endif  // OS_MACOSX

  return true;
}

void ConfigDialog::SetSendStatsCheckBox() {
  // On windows, usage_stats flag is managed by
  // administration_dialog. http://b/2889759
#ifndef OS_WINDOWS
  const bool val = StatsConfigUtil::IsEnabled();
  usageStatsCheckBox->setChecked(val);
#endif  // OS_WINDOWS
}

void ConfigDialog::GetSendStatsCheckBox() const {
  // On windows, usage_stats flag is managed by
  // administration_dialog. http://b/2889759
#ifndef OS_WINDOWS
  const bool val = usageStatsCheckBox->isChecked();
  StatsConfigUtil::SetEnabled(val);
#endif  // OS_WINDOWS
}

#define SET_COMBOBOX(combobox, enumname, field) \
do { \
  (combobox)->setCurrentIndex(static_cast<int>(config.field()));  \
} while (0)

#define SET_CHECKBOX(checkbox, field) \
do { (checkbox)->setChecked(config.field()); } while (0)

#define GET_COMBOBOX(combobox, enumname, field) \
do {  \
  config->set_##field(static_cast<config::Config_##enumname> \
                   ((combobox)->currentIndex())); \
} while (0)

#define GET_CHECKBOX(checkbox, field) \
do { config->set_##field((checkbox)->isChecked());  } while (0)


namespace {
static const int kPreeditMethodSize = 2;

void SetComboboxForPreeditMethod(const config::Config &config,
                                 QComboBox *combobox) {
  int index = static_cast<int>(config.preedit_method());
#ifdef OS_WINDOWS
  if (config.use_keyboard_to_change_preedit_method()) {
    index += kPreeditMethodSize;
  }
#endif  // OS_WINDOWS
  combobox->setCurrentIndex(index);
}

void GetComboboxForPreeditMethod(const QComboBox *combobox,
                                 config::Config *config) {
  int index = combobox->currentIndex();
  if (index >= kPreeditMethodSize) {
    // |use_keyboard_to_change_preedit_method| should be true and
    // |index| should be adjusted to smaller than kPreeditMethodSize.
    config->set_preedit_method(
        static_cast<config::Config_PreeditMethod>(index - kPreeditMethodSize));
    config->set_use_keyboard_to_change_preedit_method(true);
  } else {
    config->set_preedit_method(
        static_cast<config::Config_PreeditMethod>(index));
    config->set_use_keyboard_to_change_preedit_method(false);
  }
}
}  // anonymous namespace


// TODO(taku)
// Actually ConvertFromProto and ConvertToProto are almost the same.
// The difference only SET_ and GET_. We would like to unify the twos.
void ConfigDialog::ConvertFromProto(const config::Config &config) {
  // tab1
  SetComboboxForPreeditMethod(config, inputModeComboBox);
  SET_COMBOBOX(punctuationsSettingComboBox, PunctuationMethod,
               punctuation_method);
  SET_COMBOBOX(symbolsSettingComboBox, SymbolMethod, symbol_method);
  SET_COMBOBOX(spaceCharacterFormComboBox, FundamentalCharacterForm,
               space_character_form);
  SET_COMBOBOX(selectionShortcutModeComboBox, SelectionShortcut,
               selection_shortcut);
  SET_COMBOBOX(numpadCharacterFormComboBox, NumpadCharacterForm,
               numpad_character_form);
  SET_COMBOBOX(keymapSettingComboBox, SessionKeymap, session_keymap);

  custom_keymap_table_ = config.custom_keymap_table();
  custom_roman_table_ = config.custom_roman_table();

  // tab2
  SET_COMBOBOX(historyLearningLevelComboBox, HistoryLearningLevel,
               history_learning_level);
  SET_CHECKBOX(singleKanjiDictionaryCheckBox, use_single_kanji_conversion);
  SET_CHECKBOX(symbolDictionaryCheckBox, use_symbol_conversion);
  SET_CHECKBOX(emoticonDictionaryCheckBox, use_emoticon_conversion);
  SET_CHECKBOX(dateConversionCheckBox, use_date_conversion);
  SET_CHECKBOX(numberConversionCheckBox, use_number_conversion);
  SET_CHECKBOX(calculatorCheckBox, use_calculator);
  SET_CHECKBOX(t13nDictionaryCheckBox, use_t13n_conversion);
  SET_CHECKBOX(zipcodeDictionaryCheckBox, use_zip_code_conversion);
  SET_CHECKBOX(spellingCorrectionCheckBox, use_spelling_correction);

  // InfoListConfig
  localUsageDictionaryCheckBox->setChecked(
      config.information_list_config().use_local_usage_dictionary());
  information_list_config_.CopyFrom(config.information_list_config());

  // tab3
  SET_CHECKBOX(useAutoImeTurnOff, use_auto_ime_turn_off);

  SET_CHECKBOX(useAutoConversion, use_auto_conversion);
  kutenCheckBox->setChecked(
      config.auto_conversion_key() &
      config::Config::AUTO_CONVERSION_KUTEN);
  toutenCheckBox->setChecked(
      config.auto_conversion_key() &
      config::Config::AUTO_CONVERSION_TOUTEN);
  questionMarkCheckBox->setChecked(
      config.auto_conversion_key() &
      config::Config::AUTO_CONVERSION_QUESTION_MARK);
  exclamationMarkCheckBox->setChecked(
      config.auto_conversion_key() &
      config::Config::AUTO_CONVERSION_EXCLAMATION_MARK);

  SET_COMBOBOX(shiftKeyModeSwitchComboBox,
               ShiftKeyModeSwitch,
               shift_key_mode_switch);

  SET_CHECKBOX(useJapaneseLayout, use_japanese_layout);

  // tab4
  SET_CHECKBOX(historySuggestCheckBox, use_history_suggest);
  SET_CHECKBOX(dictionarySuggestCheckBox, use_dictionary_suggest);
  SET_CHECKBOX(realtimeConversionCheckBox, use_realtime_conversion);

  suggestionsSizeSpinBox->setValue
      (max(1, min(9, static_cast<int>(config.suggestions_size()))));

  // tab5
  SetSendStatsCheckBox();
  SET_CHECKBOX(incognitoModeCheckBox, incognito_mode);
  SET_CHECKBOX(presentationModeCheckBox, presentation_mode);

  // tab6
  SET_COMBOBOX(verboseLevelComboBox, int, verbose_level);
  SET_CHECKBOX(checkDefaultCheckBox, check_default);
  SET_COMBOBOX(yenSignComboBox, YenSignCharacter, yen_sign_character);

  characterFormEditor->Load(config);
#ifdef ENABLE_CLOUD_SYNC
  sync_customize_dialog_->Load(config);
  if (config.has_sync_config()) {
    sync_running_ = true;
    syncStatusLabel->setText(tr("Checking sync status"));
  } else {
    syncStatusLabel->setText(tr("Sync is not enabled"));
  }
  UpdateSyncToggleButtonText();
#endif  // ENABLE_CLOUD_SYNC
  SET_CHECKBOX(cloudHandwritingCheckBox, allow_cloud_handwriting);

#ifdef OS_MACOSX
  startupCheckBox->setChecked(
      MacUtil::CheckPrelauncherLoginItemStatus());
#endif  // OS_MACOSX

}

void ConfigDialog::ConvertToProto(config::Config *config) const {
  // tab1
  GetComboboxForPreeditMethod(inputModeComboBox, config);
  GET_COMBOBOX(punctuationsSettingComboBox, PunctuationMethod,
               punctuation_method);
  GET_COMBOBOX(symbolsSettingComboBox, SymbolMethod, symbol_method);
  GET_COMBOBOX(spaceCharacterFormComboBox, FundamentalCharacterForm,
               space_character_form);
  GET_COMBOBOX(selectionShortcutModeComboBox, SelectionShortcut,
               selection_shortcut);
  GET_COMBOBOX(numpadCharacterFormComboBox, NumpadCharacterForm,
               numpad_character_form);
  GET_COMBOBOX(keymapSettingComboBox, SessionKeymap, session_keymap);

  config->set_custom_keymap_table(custom_keymap_table_);

  config->clear_custom_roman_table();
  if (!custom_roman_table_.empty()) {
    config->set_custom_roman_table(custom_roman_table_);
  }

  // tab2
  GET_COMBOBOX(historyLearningLevelComboBox, HistoryLearningLevel,
               history_learning_level);
  GET_CHECKBOX(singleKanjiDictionaryCheckBox, use_single_kanji_conversion);
  GET_CHECKBOX(symbolDictionaryCheckBox, use_symbol_conversion);
  GET_CHECKBOX(emoticonDictionaryCheckBox, use_emoticon_conversion);
  GET_CHECKBOX(dateConversionCheckBox, use_date_conversion);
  GET_CHECKBOX(numberConversionCheckBox, use_number_conversion);
  GET_CHECKBOX(calculatorCheckBox, use_calculator);
  GET_CHECKBOX(t13nDictionaryCheckBox, use_t13n_conversion);
  GET_CHECKBOX(zipcodeDictionaryCheckBox, use_zip_code_conversion);
  GET_CHECKBOX(spellingCorrectionCheckBox, use_spelling_correction);

  // InformationListConfig
  config->mutable_information_list_config()->CopyFrom(
      information_list_config_);
  config->mutable_information_list_config()->set_use_local_usage_dictionary(
      localUsageDictionaryCheckBox->isChecked());

  // tab3
  GET_CHECKBOX(useAutoImeTurnOff, use_auto_ime_turn_off);

  GET_CHECKBOX(useAutoConversion, use_auto_conversion);

  GET_CHECKBOX(useJapaneseLayout, use_japanese_layout);

  uint32 auto_conversion_key = 0;
  if (kutenCheckBox->isChecked()) {
    auto_conversion_key |=
        config::Config::AUTO_CONVERSION_KUTEN;
  }
  if (toutenCheckBox->isChecked()) {
    auto_conversion_key |=
        config::Config::AUTO_CONVERSION_TOUTEN;
  }
  if (questionMarkCheckBox->isChecked()) {
    auto_conversion_key |=
        config::Config::AUTO_CONVERSION_QUESTION_MARK;
  }
  if (exclamationMarkCheckBox->isChecked()) {
    auto_conversion_key |=
        config::Config::AUTO_CONVERSION_EXCLAMATION_MARK;
  }
  config->set_auto_conversion_key(auto_conversion_key);

  GET_COMBOBOX(shiftKeyModeSwitchComboBox,
               ShiftKeyModeSwitch,
               shift_key_mode_switch);

  // tab4
  GET_CHECKBOX(historySuggestCheckBox, use_history_suggest);
  GET_CHECKBOX(dictionarySuggestCheckBox, use_dictionary_suggest);
  GET_CHECKBOX(realtimeConversionCheckBox, use_realtime_conversion);

  config->set_suggestions_size
      (static_cast<uint32>(suggestionsSizeSpinBox->value()));

  // tab5
  GetSendStatsCheckBox();
  GET_CHECKBOX(incognitoModeCheckBox, incognito_mode);
  GET_CHECKBOX(presentationModeCheckBox, presentation_mode);

  // tab6
  config->set_verbose_level(verboseLevelComboBox->currentIndex());
  GET_CHECKBOX(checkDefaultCheckBox, check_default);
  GET_COMBOBOX(yenSignComboBox, YenSignCharacter, yen_sign_character);

  characterFormEditor->Save(config);
#ifdef ENABLE_CLOUD_SYNC
  sync_customize_dialog_->Save(sync_running_, config);
#endif  // ENABLE_CLOUD_SYNC
#ifdef ENABLE_CLOUD_HANDWRITING
  GET_CHECKBOX(cloudHandwritingCheckBox, allow_cloud_handwriting);
#endif  // ENABLE_CLOUD_HANDWRITING
}

#undef SET_COMBOBOX
#undef SET_CHECKBOX
#undef GET_COMBOBOX
#undef GET_CHECKBOX

void ConfigDialog::clicked(QAbstractButton *button) {
  switch (configDialogButtonBox->buttonRole(button)) {
    case QDialogButtonBox::AcceptRole:
      if (Update()) {
        QWidget::close();
      }
      break;
    case QDialogButtonBox::ApplyRole:
      Update();
      break;
    case QDialogButtonBox::RejectRole:
      QWidget::close();
      break;
    default:
      break;
  }
}

void ConfigDialog::ClearUserHistory() {
  if (QMessageBox::Ok !=
      QMessageBox::question(
          this,
          tr("Mozc settings"),
          tr("Do you want to clear personalization data? "
             "Input history is not reset with this operation. "
             "Please open \"suggestion\" tab to remove input history data."),
          QMessageBox::Ok | QMessageBox::Cancel,
          QMessageBox::Cancel)) {
    return;
  }

  client_->CheckVersionOrRestartServer();

  if (!client_->ClearUserHistory()) {
    QMessageBox::critical(
        this,
        tr("Mozc settings"),
        tr("Mozc Converter is not running. "
           "Settings were not saved."));
  }
}

void ConfigDialog::ClearUserPrediction() {
  if (QMessageBox::Ok !=
      QMessageBox::question(
          this,
          tr("Mozc settings"),
          tr("Do you want to clear all history data?"),
          QMessageBox::Ok | QMessageBox::Cancel,
          QMessageBox::Cancel)) {
    return;
  }

  client_->CheckVersionOrRestartServer();

  if (!client_->ClearUserPrediction()) {
    QMessageBox::critical(
        this,
        tr("Mozc settings"),
        tr("Mozc Converter is not running. "
           "Settings were not saved."));
  }
}

void ConfigDialog::ClearUnusedUserPrediction() {
  if (QMessageBox::Ok !=
      QMessageBox::question(
          this,
          tr("Mozc settings"),
          tr("Do you want to clear unused history data?"),
          QMessageBox::Ok | QMessageBox::Cancel,
          QMessageBox::Cancel)) {
    return;
  }

  client_->CheckVersionOrRestartServer();

  if (!client_->ClearUnusedUserPrediction()) {
    QMessageBox::critical(
        this,
        tr("Mozc settings"),
        tr("Mozc Converter is not running. "
           "Operation was not executed."));
  }
}

void ConfigDialog::EditUserDictionary() {
  client_->LaunchTool("dictionary_tool", "");
}

void ConfigDialog::EditKeymap() {
  string current_keymap_table = "";
  const QString keymap_name = keymapSettingComboBox->currentText();
  const map<QString, config::Config::SessionKeymap>::const_iterator itr =
      keymapname_sessionkeymap_map_.find(keymap_name);
  if (itr != keymapname_sessionkeymap_map_.end()) {
    // Load from predefined mapping file.
    const char *keymap_file =
        keymap::KeyMapManager::GetKeyMapFileName(itr->second);
    scoped_ptr<istream> ifs(
        ConfigFileStream::LegacyOpen(keymap_file));
    CHECK(ifs.get() != NULL);  // should never happen
    stringstream buffer;
    buffer << ifs->rdbuf();
    current_keymap_table = buffer.str();
  } else {
    current_keymap_table = custom_keymap_table_;
  }
  string output;
  if (gui::KeyMapEditorDialog::Show(this,
                                    current_keymap_table,
                                    &output)) {
    custom_keymap_table_ = output;
    // set keymapSettingComboBox to "Custom keymap"
    keymapSettingComboBox->setCurrentIndex(0);
  }
}

void ConfigDialog::EditRomanTable() {
  string output;
  if (gui::RomanTableEditorDialog::Show(this,
                                        custom_roman_table_,
                                        &output)) {
    custom_roman_table_ = output;
  }
}

void ConfigDialog::EditInforlistConfig() {
}

void ConfigDialog::SelectInputModeSetting(int index) {
  // enable "EDIT" button if roman mode is selected
  editRomanTableButton->setEnabled((index == 0));
}

void ConfigDialog::SelectAutoConversionSetting(int state) {
  kutenCheckBox->setEnabled(static_cast<bool>(state));
  toutenCheckBox->setEnabled(static_cast<bool>(state));
  questionMarkCheckBox->setEnabled(static_cast<bool>(state));
  exclamationMarkCheckBox->setEnabled(static_cast<bool>(state));
}

void ConfigDialog::SelectSuggestionSetting(int state) {
  if (historySuggestCheckBox->isChecked() ||
      dictionarySuggestCheckBox->isChecked() ||
      realtimeConversionCheckBox->isChecked()) {
    presentationModeCheckBox->setEnabled(true);
  } else {
    presentationModeCheckBox->setEnabled(false);
  }
}

void ConfigDialog::SelectWebUsageDictionarySetting(bool checked) {
  if (checked) {
    if (QMessageBox::question(
            this,
            tr("Mozc settings"),
            tr("Web Service extension enables Mozc "
               "to display third party usage dictionaries provided as an "
               "Web service API, e.g. REST (POX/JSON over HTTP). "
               "Note that candidate/preedit strings are "
               "sent to the Web service when this feature is enabled. "
               "Do you want to use Web Service extension?"),
            QMessageBox::Yes | QMessageBox::No, QMessageBox::No)
        == QMessageBox::Yes) {
      editWebServiceEntryButton->setEnabled(true);
    } else {
      webUsageDictionaryCheckBox->setChecked(false);
      editWebServiceEntryButton->setEnabled(false);
    }
  } else {
    editWebServiceEntryButton->setEnabled(false);
  }
}

void ConfigDialog::ResetToDefaults() {
  if (QMessageBox::Ok ==
      QMessageBox::question(
          this,
          tr("Mozc settings"),
          tr("When you reset Mozc settings, any changes you've made "
             "will be reverted to the default settings. "
             "Do you want to reset settings? "
             "The following items are not reset with this operation.\n"
             " - Personalization data\n"
             " - Input history\n"
             " - Sync settings\n"
             " - Usage statistics and crash reports\n"
             " - Administrator settings"),
          QMessageBox::Ok | QMessageBox::Cancel,
          QMessageBox::Cancel)) {
    // TODO(taku): remove the dependency to config::ConfigHandler
    // nice to have GET_DEFAULT_CONFIG command
    config::Config config;
    config::ConfigHandler::GetDefaultConfig(&config);
    ConvertFromProto(config);
#ifdef ENABLE_CLOUD_SYNC
    if (sync_running_) {
      StopSyncForcely();
    }
#endif  // ENABLE_CLOUD_SYNC
  }
}

void ConfigDialog::LaunchAdministrationDialog() {
#ifdef OS_WINDOWS
  client_->LaunchTool("administration_dialog", "");
#endif  // OS_WINDOWS
}

void ConfigDialog::SyncToggleButtonClicked() {
#ifdef ENABLE_CLOUD_SYNC
  SyncToggleButtonClickedImpl();
#endif  // ENABLE_CLOUD_SYNC
}

void ConfigDialog::LaunchSyncCustomizationDialog() {
#ifdef ENABLE_CLOUD_SYNC
  LaunchSyncCustomizationDialogImpl();
#endif  // ENABLE_CLOUD_SYNC
}

void ConfigDialog::UpdateSyncStatus() {
#ifdef ENABLE_CLOUD_SYNC
  UpdateSyncStatusImpl();
#endif  // ENABLE_CLOUD_SYNC
}

void ConfigDialog::ClearSyncClicked() {
#ifdef ENABLE_CLOUD_SYNC
  ClearSyncClickedImpl();
#endif  // ENABLE_CLOUD_SYNC
}

#ifdef ENABLE_CLOUD_SYNC
void ConfigDialog::UpdateSyncToggleButtonText() {
  if (sync_running_) {
    syncToggleButton->setText(tr("Stop Sync"));
  } else {
    syncToggleButton->setText(tr("Start Sync"));
  }
}

void ConfigDialog::SyncToggleButtonClickedImpl() {
  if (sync_running_) {
    StopSync();
  } else {
    LaunchAuthDialog();
  }
}

void ConfigDialog::StopSync() {
  if (QMessageBox::Cancel ==
      QMessageBox::question(
          this,
          tr("Stop sync and reset the auth token"),
          tr("You are trying to stop sync and reset the auth token. "
             "To restart sync, you will need to get a new auth token.\n"
             "Do you really want to continue?"),
          QMessageBox::Ok | QMessageBox::Cancel,
          QMessageBox::Cancel)) {
    return;
  }

  StopSyncForcely();
}

void ConfigDialog::StopSyncForcely() {
  // To stop the current synchronization, it resets the authorization code.
  commands::Input::AuthorizationInfo auth_info;
  auth_info.set_auth_code("");
  if (!client_->AddAuthCode(auth_info)) {
    LOG(WARNING) << "Sending auth code failed";
    return;
  }
  sync_running_ = false;
  UpdateSyncToggleButtonText();
}

void ConfigDialog::LaunchAuthDialog() {
  string auth_code;
  if (!gui::AuthDialog::Show(this, &auth_code)) {
    LOG(WARNING) << "Canceled.  Do nothing";
    return;
  }

  // The default configuration at the starting timing will be "all".
  config::Config mock_config;
  config::SyncConfig *sync_config = mock_config.mutable_sync_config();
  sync_config->set_use_config_sync(true);
  sync_config->set_use_user_dictionary_sync(true);
  sync_config->set_use_user_history_sync(true);
  sync_config->set_use_learning_preference_sync(true);
  sync_config->set_use_contact_list_sync(true);

  // Store the current config for the rollback when failed.
  config::Config current_config;
  sync_customize_dialog_->Save(false, &current_config);
  sync_customize_dialog_->Load(mock_config);

  // Then ask users to customize the sync configuration.
  if (QDialog::Accepted != sync_customize_dialog_->exec()) {
    LOG(WARNING) << "Canceled during sync customization. "
                 << "It means do not activate sync.";
    // Rollback the config.
    sync_customize_dialog_->Load(current_config);
    return;
  }

  sync_running_ = true;
  // Before starting sync, the sync customization should be applied.
  // WARNING: the whole configuration (items other than sync) should
  // be applied at the same time.
  Update();

  commands::Input::AuthorizationInfo auth_info;
  auth_info.set_auth_code(auth_code);
  if (!client_->AddAuthCode(auth_info)) {
    LOG(WARNING) << "Sending auth code failed";
    return;
  }
  if (!client_->StartCloudSync()) {
    LOG(WARNING) << "Cloud sync is failed to start";
  }
  UpdateSyncToggleButtonText();
  // It does not wait for the end of cloud sync.
}

void ConfigDialog::LaunchSyncCustomizationDialogImpl() {
  if (QDialog::Accepted == sync_customize_dialog_->exec()) {
    // Before starting sync, the sync customization should be applied.
    // WARNING: the whole configuration (items other than sync) should
    // be applied at the same time.
    Update();
    if (!client_->StartCloudSync()) {
      LOG(WARNING) << "Cloud sync is failed to start";
    }
  }
}

void ConfigDialog::AddLastSyncedDateTime(
    uint64 timestamp, QString *output) const {
  // 0 is default value, and means to have no sync before.
  if (timestamp == 0) {
    output->append(tr("Not synced yet"));
    return;
  }

  output->append(tr("Last synced time: "));
  const int time_offset = Util::GetTime() - timestamp;
  QDateTime datetime = QDateTime::currentDateTime().addSecs(-time_offset);
  output->append(datetime.toString("yyyy/MM/dd hh:mm:ss"));
}

void ConfigDialog::UpdateSyncStatusImpl() {
  // This method always updates the sync status even when sync is not
  // running.  That's because we should change the message to "not
  // enabled" when the user explicitly turns off sync feature.
  QString sync_message;
  commands::CloudSyncStatus cloud_sync_status;
  client_->GetCloudSyncStatus(&cloud_sync_status);
  if (sync_running_) {
    DLOG(INFO) << cloud_sync_status.DebugString();
  } else {
    // The default status is 'NOSYNC'.  Only changed via IPC when sync
    // is running.
    cloud_sync_status.set_global_status(commands::CloudSyncStatus::NOSYNC);
  }

  commands::CloudSyncStatus::SyncGlobalStatus sync_global_status =
      cloud_sync_status.global_status();

  // clearing sync is only available when sync is succeeding ==
  // the current authorization is valid.
  clearSyncButton->setEnabled(
      sync_global_status == commands::CloudSyncStatus::SYNC_SUCCESS);
  syncCustomizationButton->setEnabled(true);
  syncToggleButton->setEnabled(true);

  switch (sync_global_status) {
    case commands::CloudSyncStatus::SYNC_SUCCESS:
    case commands::CloudSyncStatus::SYNC_FAILURE: {
      uint64 last_synced_timestamp = cloud_sync_status.last_synced_timestamp();
      AddLastSyncedDateTime(last_synced_timestamp, &sync_message);
      if (last_synced_timestamp_ != last_synced_timestamp) {
        Reload();
        last_synced_timestamp_ = last_synced_timestamp;
      }
      break;
    }
    case commands::CloudSyncStatus::WAITSYNC:
      sync_message = tr("Waiting for server to be ready");
      syncCustomizationButton->setEnabled(false);
      syncToggleButton->setEnabled(false);
      clearSyncButton->setEnabled(false);
      break;
    case commands::CloudSyncStatus::INSYNC:
      sync_message = tr("During synchronization");
      syncCustomizationButton->setEnabled(false);
      syncToggleButton->setEnabled(false);
      clearSyncButton->setEnabled(false);
      break;
    case commands::CloudSyncStatus::NOSYNC:
      sync_message = tr("Sync is not enabled");
      sync_running_ = false;
      syncCustomizationButton->setEnabled(false);
      clearSyncButton->setEnabled(false);
      break;
  }

  QString tooltip;
  // Check error messages in sync.
  if (cloud_sync_status.sync_errors_size() > 0) {
    for (size_t i = 0; i < cloud_sync_status.sync_errors_size(); ++i) {
      if (i > 0) {
        tooltip += "\n";
      }

      switch (cloud_sync_status.sync_errors(i).error_code()) {
        case commands::CloudSyncStatus::AUTHORIZATION_FAIL:
          tooltip += tr("Authorization failed.");
          break;
        case commands::CloudSyncStatus::USER_DICTIONARY_NUM_ENTRY_EXCEEDED:
          tooltip += tr("Cannot save dictionaries because Sync Dictionary "
                        "exceeds its entry size limit.");
          break;
        case commands::CloudSyncStatus::USER_DICTIONARY_BYTESIZE_EXCEEDED:
          tooltip += tr("Cannot save dictionaries because Sync Dictionary "
                        "exceeds its binary size limit.");
          break;
        case commands::CloudSyncStatus::USER_DICTIONARY_NUM_DICTIONARY_EXCEEDED:
          tooltip += tr("Cannot save dictionaries because the number of "
                        "dictionaries for sync exceeds its limit.");
          break;
        default:
          tooltip += tr("Unknown error was found.");
          LOG(WARNING) << "Unknown error ("
                       << cloud_sync_status.sync_errors(i).error_code()
                       << ") was found in cloud sync errors.";
          break;
      }
    }

    sync_message += "  ";
    // Display the first error category obviously.
    switch (cloud_sync_status.sync_errors(0).error_code()) {
      case commands::CloudSyncStatus::AUTHORIZATION_FAIL:
        sync_message += tr("Authorization error");
        break;
      case commands::CloudSyncStatus::USER_DICTIONARY_NUM_ENTRY_EXCEEDED:
      case commands::CloudSyncStatus::USER_DICTIONARY_BYTESIZE_EXCEEDED:
      case commands::CloudSyncStatus::USER_DICTIONARY_NUM_DICTIONARY_EXCEEDED:
        sync_message += tr("Dictionary sync error");
        break;
      default:
        sync_message += tr("Unknown sync error");
        break;
    }
  }

  UpdateSyncToggleButtonText();
  syncStatusLabel->setText(sync_message);
  syncStatusLabel->setToolTip(tooltip);
}

void ConfigDialog::ClearSyncClickedImpl() {
  if (QMessageBox::Cancel ==
      QMessageBox::question(
          this,
          tr("Clear all sync data in Google servers"),
          tr("You are trying to clear all the data in the sync server.\n"
             "Do you really want to continue?"),
          QMessageBox::Ok | QMessageBox::Cancel,
          QMessageBox::Cancel)) {
    return;
  }

  // Stop doing sync configurations.
  config::Config config;
  GetConfig(&config);
  config.clear_sync_config();
  sync_running_ = false;
  ConvertFromProto(config);
  Update();

  // Do the actual clear.
  client_->ClearCloudSync();
}
#endif  // ENABLE_CLOUD_SYNC

// Catch MouseButtonRelease event to toggle the CheckBoxes
bool ConfigDialog::eventFilter(QObject *obj, QEvent *event) {
  if (event->type() == QEvent::MouseButtonRelease) {
    if (obj == usageStatsMessage) {
#ifndef CHANNEL_DEV
      usageStatsCheckBox->toggle();
#endif
    } else if (obj == incognitoModeMessage) {
      incognitoModeCheckBox->toggle();
    }
  }
  return QObject::eventFilter(obj, event);
}

}  // namespace gui
}  // namespace mozc
