/* 
 *   Creation Date: <2001/05/06 22:27:09 samuel>
 *   Time-stamp: <2002/12/22 17:23:58 samuel>
 *   
 *	<fs.c>
 *	
 *	Generic file system access
 *   
 *   Copyright (C) 2001, 2002 Samuel Rydh (samuel@ibrium.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#include "of.h"
#include "prom.h"
#include "fs.h"
#include "partition_table.h"
#include "ablk_sh.h"
#include "osi_calls.h"
#include "hfs_mdb.h"
#include "os.h"
#include "pseudofs_sh.h"

typedef struct filespec {
	llong	seek_pos;		/* offset (within the partition) */

	llong	par_offs;		/* offset to the start of partition */
	llong	total_size;		/* total size of the _partition_ */

	int	unit;
	int	channel;
	int	(*read)( struct filespec *v, char *buf, size_t count );
} filespec_t;


static int
find_partition_offs( void **priv, int parnum )
{
	filespec_t *v = *(filespec_t**)priv;
	desc_map_t dmap;
	part_entry_t par;
	int bs;
	
	memset( &dmap, 0, sizeof(dmap) );
	
	os_seek( priv, 0, 0 );
	os_read( priv, &dmap, sizeof(dmap), 0 );

	if( dmap.sbSig != DESC_MAP_SIGNATURE )
		return -1;

	/* Partition maps might supports multiple block sizes; in this case,
	 * pmPyPartStart is typically given in terms of 512 byte blocks.
	 */
	bs = dmap.sbBlockSize;
	if( bs != 512 ) {
		os_seek( priv, 512, 0 );
		os_read( priv, &par, sizeof(par), 0 );
		if( par.pmSig == 0x504d )
			bs = 512;
	}
	
	os_seek( priv, bs, 0 );
	os_read( priv, &par, sizeof(part_entry_t), 0 );

	if( parnum > par.pmMapBlkCnt || par.pmSig != 0x504d /* 'PM' */ )
		return -1;

	os_seek( priv, ((long long)bs) * parnum, 0 );
	os_read( priv, &par, sizeof(par), 0 );

	if( par.pmSig != 0x504d /*'PM'*/ || !par.pmPartBlkCnt )
		return -1;

	v->par_offs = (long long)par.pmPyPartStart * bs;
	v->total_size = (long long)par.pmPartBlkCnt * bs;
	return 0;
}

char *
get_hfs_vol_name( const char *spec, char *buf, int size )
{
	char sect[512];
	hfs_mdb_t *mdb = (hfs_mdb_t*)&sect;
	void *priv;

	if( os_open( &priv, spec ) )
		return strcpy( buf, "Error");

	os_seek( &priv, 0x400, 0 );
	os_read( &priv, sect, sizeof(sect), 0 );
	if( hfs_get_ushort(mdb->drSigWord) == HFS_SIGNATURE ) {
		memcpy( buf, &mdb->drVN[1], mdb->drVN[0] );		
		buf[mdb->drVN[0]] = 0;
	} else if( hfs_get_ushort(mdb->drSigWord) == HFS_PLUS_SIGNATURE ) {
		strcpy(buf, "Unembedded HFS+");
	} else {
		strcpy(buf, "Error");
	}
	os_close( &priv );
	return buf;
}


/* dev_spec is an OF-path, something like /mol-blk@channnel/disk@unit:par */
int
os_open( void **priv, const char *dev_spec )
{
	int i, chan, unit=0, channel=0, part=0;
	char namebuf[32], *pp;
	ablk_disk_info_t info;
	mol_phandle_t ph;
	filespec_t *fds;
	llong total_size;

	if( (ph=prom_find_device(dev_spec)) == -1 || (ph=prom_parent(ph)) == -1 ) {
		printm("Open: Could not open node %s\n", dev_spec );
		return -1;
	}
	if( prom_get_prop( ph, "channel", (char*)&chan, 4) == 4 )
		channel = chan;

	for( i=strlen(dev_spec)-1 ; i >= 0 ; i-- ) {
		if( dev_spec[i] == '@' ) {
			unit = strtol( dev_spec+i+1, &pp, 16 );
			if( *pp == ':' )
				part = strtol( pp+1, NULL, 10 );
			break;
		}
	}

	/* determine which kind of osi interface we should use (osi_blk or ablk) */
	prom_get_prop( ph, "name", namebuf, sizeof(namebuf) );

	/* get the basic facts */
	if( OSI_ABlkDiskInfo( channel, unit, &info ) )
		return -1;
	total_size = (ullong)info.nblks << 9;

	/* printm("os_open channel %d, unit %d, partition %d\n", channel, unit, part ); */
	fds = malloc( sizeof(filespec_t) );
	fds->seek_pos = 0;
	fds->total_size = total_size;
	fds->par_offs = 0;
	fds->unit = unit;
	fds->channel = channel;

	if( part && find_partition_offs((void**)&fds, part) < 0 ) {
		os_close( (void**)&fds );
		return -1;
	}
	*priv = (void*)fds;
	return 0;
}

ulong
os_read( void **priv, void *buf, ulong len, int blksize_bits )
{
	filespec_t *fds = *(filespec_t**)priv;
	llong offs = fds->seek_pos + fds->par_offs;

	len <<= blksize_bits;
	
	if( (offs & 0x1ff) || (len & 0x1ff) ) {
		printm("os_read: bad offset or size\n");
		return -1;
	}
	if( OSI_ABlkSyncRead( fds->channel, fds->unit, offs >> 9, (ulong)buf, len ) )
		return -1;

	fds->seek_pos += len;
	return len >> blksize_bits;
}

ulong
os_seek( void **priv, ulong offset, int blksize_bits )
{
	filespec_t *fds = *(filespec_t**)priv;
	llong offs = offset;
	
	offs <<= blksize_bits;

	/* offset == -1 means seek to EOF */
	fds->seek_pos = (offset == -1) ? fds->total_size : offs;

	if( fds->seek_pos > fds->total_size ) {
		printm("os_seek error\n");
		return -1;
	}
	return fds->seek_pos >> blksize_bits;
}

int
os_close( void **priv )
{
	filespec_t *fds = *(filespec_t**)priv;
	free( fds );
	return 0;
}

int
os_same( void **priv, const char *path )
{
	printm("os_same unimplemented\n");
	return 0;
}


/************************************************************************/
/*	pseudo filesystem (used to publish files from the linux side)	*/
/************************************************************************/

typedef struct {
	char		filename[64];
	int		fd;
	int		pos;
	fs_ops_t	*fs;
} pseudo_fd_t;

static void
ps_close_fs( fs_ops_t *fs ) 
{
	free( fs );
}

static file_desc_t
ps_open_path( fs_ops_t *fs, const char *path )
{
	pseudo_fd_t *p;

	while( *path == '/' )
		path++;
	p = malloc( sizeof(pseudo_fd_t) );
	strncpy( p->filename, path, sizeof(p->filename) );
	p->filename[ sizeof(p->filename)-1] = 0;
	p->pos = 0;

	if( (p->fd=PseudoFSOpen(p->filename)) < 0 ) {
		free( p );
		return NULL;
	}
	p->fs = fs;
	return (file_desc_t)p;
}

static int
ps_lseek( file_desc_t fd, off_t offset, int whence )
{
	pseudo_fd_t *p = (pseudo_fd_t*)fd;
	int size = PseudoFSGetSize(p->fd);
	
	if( whence == SEEK_SET ) {
		p->pos = offset;
	} else if( whence == SEEK_CUR ) {
		p->pos += offset;
	} else {
		p->pos = size + offset;
	}
	if( p->pos < 0 )
		p->pos = 0;
	if( p->pos > size )
		p->pos = size;
	return p->pos;
}

static ssize_t
ps_read( file_desc_t fd, void *buf, size_t count )
{
	ssize_t ret;
	pseudo_fd_t *p = (pseudo_fd_t*)fd;

	ret = PseudoFSRead( p->fd, p->pos, buf, count );
	if( ret >= 0 )
		p->pos += ret;
	return ret;
}

static void
ps_close( file_desc_t fd )
{
	pseudo_fd_t *p = (pseudo_fd_t*)fd;
	PseudoFSClose( p->fd );
	free( p );
}

static char *
ps_get_path( file_desc_t fd, char *buf, int len )
{
	pseudo_fd_t *p = (pseudo_fd_t*)fd;
	strncpy( buf, p->filename, len );
	return buf;
}

static file_desc_t
dummy_search_rom( fs_ops_t *fs )
{
	return NULL;
}

static file_desc_t
dummy_search_file( fs_ops_t *fs, const char *name )
{
	return NULL;
}

static char *
ps_vol_name( fs_ops_t *fs, char *buf, int size )
{
	return strncpy( buf, "pseudo-fs", size );
}

static fs_ops_t pseudo_ops = {
	close_fs:	ps_close_fs,
	open_path:	ps_open_path,
	get_path:	ps_get_path,
	close:		ps_close,
	read:		ps_read,
	lseek:		ps_lseek,

	search_rom:	dummy_search_rom,
	search_file:	dummy_search_file,
	vol_name:	ps_vol_name
};

fs_ops_t *
fs_pseudo_open( void )
{
	fs_ops_t *fs = malloc( sizeof(fs_ops_t) );
	*fs = pseudo_ops;
	return fs;
}

/* no-wrapper ops */

file_desc_t
pseudo_open( const char *path )
{
	fs_ops_t *fs = fs_pseudo_open();
	file_desc_t fd;
	
	if( !(fd=fs->open_path(fs, path)) )
		fs_close(fs);
	return fd;
}

void
pseudo_close( file_desc_t fd )
{
	fs_ops_t *fs = ((pseudo_fd_t*)fd)->fs;
	fs->close( fd );
	fs_close( fs );
}

int
pseudo_read( file_desc_t fd, char *buf, int size )
{
	fs_ops_t *fs = ((pseudo_fd_t*)fd)->fs;
	return fs->read( fd, buf, size );
}

int
pseudo_lseek( file_desc_t fd, off_t offs, int whence )
{
	fs_ops_t *fs = ((pseudo_fd_t*)fd)->fs;
	return fs->lseek( fd, offs, whence );
}


/************************************************************************/
/*	fs_open								*/
/************************************************************************/

fs_ops_t *
fs_open( int fs_type, const char *spec )
{
	switch( fs_type ) {
	case FS_HFS:
		return fs_hfs_open( spec );
	case FS_HFS_PLUS:
		return fs_hfsp_open( spec );
	case FS_PSEUDO:
		return fs_pseudo_open();
	}

	return NULL;
}
